// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The `stage` action.
//!
//! This action is intended to manage a temporary integration branch (normally `stage`) to perform
//! testing on a collection of branches which are on their way into the main integration branch
//! (normally `master`).

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate itertools;
use self::itertools::Itertools;

extern crate kitware_git;
use self::kitware_git::{Identity, MergeStatus, check_status};

extern crate kitware_stager;
use self::kitware_stager::{CandidateTopic, IntegrationResult, Stager, Topic, UnstageReason};

use super::super::host::*;

use std::borrow::Cow;
use std::io;
use std::process::Stdio;

quick_error! {
    #[derive(Debug)]
    /// An error within the `stage` action.
    pub enum Error {
        /// An error occurred while managing the stage branch.
        Stage(err: kitware_stager::Error) {
            cause(err)
            display("stage error: {:?}", err)
            from()
        }
        /// An error occurred while performing a git action.
        Git(err: io::Error) {
            cause(err)
            display("git error: {:?}", err)
            from()
        }
        /// An error occurred while communicating with the hosting service.
        Host(err: HostError) {
            cause(err)
            display("hosting error: {:?}", err)
            from()
        }
    }
}

enum MissedUpdateBridge<'a> {
    Good(&'a HostedCommit),
    Munged(Box<HostedCommit>),
}

impl<'a> MissedUpdateBridge<'a> {
    fn commit(&self) -> &HostedCommit {
        match *self {
            MissedUpdateBridge::Good(c) => c,
            MissedUpdateBridge::Munged(ref c) => c.as_ref(),
        }
    }
}

/// Implementation of the `stage` action.
///
/// The stage is a collection of topic branches which should be tested together. The stage is meant
/// to be "tagged" on a regular basis and pushed for testing. In the meantime, topics may be added
/// to and removed from the staging branch. If any topic is updated, it is removed from the stage
/// and put at the end of the set of topics ready for merging. Additionally, if the base of the
/// stage updates, the entire stage is recreated.
pub struct Stage {
    branch: String,
    stager: Stager,
    project: HostedProject,
    quiet: bool,
    keep_topics: bool,
}

impl Stage {
    /// Create a new stage action.
    pub fn new<B: ToString>(stager: Stager, branch: B, project: HostedProject) -> Self {
        Stage {
            branch: branch.to_string(),
            stager: stager,
            project: project,
            quiet: false,
            keep_topics: false,
        }
    }

    /// Reduce the number of comments made by the stage action.
    ///
    /// The comments created by this action can be a bit much. This reduces the comments to those
    /// which are errors or are important.
    pub fn quiet(&mut self) -> &mut Self {
        self.quiet = true;
        self
    }

    /// Keep staged topics when tagging a stage for nightly testing.
    pub fn keep_topics(&mut self) -> &mut Self {
        self.keep_topics = true;
        self
    }

    /// A reference to the internal stager.
    pub fn stager(&self) -> &Stager {
        &self.stager
    }

    /// Update the base commit for the stage.
    ///
    /// FIXME: Currently does no checks that this is the actual base.
    pub fn base_branch_update(&mut self, commit: &HostedCommit, who: &Identity,
                              when: &DateTime<UTC>)
                              -> Result<(), Error> {
        // Fetch the commit into the stager's git context.
        try!(self.project.service.fetch_commit(self.stager.git_context(), commit));

        let candidate = CandidateTopic {
            old_id: Some(Topic::new(self.stager.base().clone(),
                                    who.clone(),
                                    UTC::now(),
                                    0,
                                    "base",
                                    "url")),
            new_id: Topic::new(commit.commit_id(),
                               who.clone(),
                               *when,
                               0,
                               "base",
                               "url"),
        };

        try!(self.update_stage_base(candidate));
        Ok(try!(self.update_current_ref()))
    }

    /// Add a merge request to the stage.
    pub fn stage_merge_request(&mut self, mr: &HostedMergeRequest, who: &Identity,
                               when: &DateTime<UTC>)
                               -> Result<(), Error> {
        // Fetch the MR commit into the stager's git context.
        try!(self.project.service.fetch_mr(self.stager.git_context(), mr));

        let mut old_commit = mr.old_commit()
            .map(MissedUpdateBridge::Good);

        if let Some(staged) = self.stager.find_topic_by_id(mr.id()) {
            old_commit = match old_commit {
                Some(old_commit) => {
                    let actual = old_commit.commit().commit_id();
                    let expected = staged.commit();

                    if &actual != expected {
                        warn!(target: "workflow/stage",
                              "it appears as though an update for the merge request {} \
                               was missed; munging the request so that it removes the stale \
                               branch ({:?}) from the stage instead of the indicated branch \
                               ({:?}).",
                              mr.url(),
                              expected,
                              actual);

                        let project = old_commit.commit().project().name();
                        let commit = try!(self.project.service.commit(project, expected));
                        Some(MissedUpdateBridge::Munged(commit))
                    } else {
                        Some(old_commit)
                    }
                },
                None => None,
            };

            if &mr.commit_id() == staged.commit() {
                self.send_info_mr_comment(mr,
                                          "This topic has already been staged; ignoring the \
                                           request to stage.");

                return Ok(());
            }
        }

        // Create the candidate topic for the MR.
        let old_hosted_commit = old_commit.as_ref().map(|c| c.commit());
        let candidate = CandidateTopic {
            old_id: old_hosted_commit.map(|c| {
                Topic::new(c.commit_id(),
                           who.clone(),
                           UTC::now(),
                           mr.id(),
                           mr.source_branch(),
                           mr.url())
            }),
            new_id: Topic::new(mr.commit_id(),
                               who.clone(),
                               *when,
                               mr.id(),
                               mr.source_branch(),
                               mr.url()),
        };

        // Update the stage.
        try!(self.update_stage_mr(candidate, old_hosted_commit));
        // Push the new stage state to the remote.
        Ok(try!(self.update_current_ref()))
    }

    /// Remove a merge request from the stage.
    pub fn unstage_merge_request(&mut self, mr: &HostedMergeRequest) -> Result<(), Error> {
        let topic = Topic::new(mr.commit_id(),
                               self.stager.identity().clone(),
                               UTC::now(),
                               mr.id(),
                               mr.source_branch(),
                               mr.url());
        let staged_topic_opt = self.stager.find_topic(&topic).cloned();

        Ok(if let Some(staged_topic) = staged_topic_opt {
            let stage_result = try!(self.stager.unstage(staged_topic));

            self.send_info_mr_comment(mr,
                                      "This merge request has been removed from the stage upon \
                                       request.");
            self.send_mr_commit_status(mr, CommitStatusState::Success, "removed from the stage");

            // Update topics have been punted off of the stage (successfully staged commits are
            // fine).
            for topic in &stage_result.results {
                try!(self.update_mr_state(topic, false));
            }

            // Push the new stage state to the remote.
            try!(self.update_current_ref())
        } else {
            let mr_res = self.hosted_mr(&topic);

            match mr_res {
                Ok(mr) => {
                    self.send_info_mr_comment(mr.as_ref(),
                                              "Failed to find this merge request on the stage; \
                                               ignoring the request to unstage it.");
                },
                Err(err) => {
                    error!(target: "workflow/stage",
                           "failed to fetch mr {} for {}: {:?}",
                           topic.id,
                           self.project.name,
                           err);
                },
            }
        })
    }

    /// Tag the current stage into a ref and reset the state of the stage.
    pub fn tag_stage(&mut self) -> Result<(), Error> {
        // Tag the current state of the stage.
        let stage_ref = try!(self.tag_current_ref());
        let commit_res = self.project.commit(self.stager.head());

        match commit_res {
            Ok(commit) => {
                self.send_commit_comment(commit.as_ref(),
                                         "This commit has been pushed for nightly testing.")
            },
            Err(err) => {
                error!(target: "workflow/stage",
                       "failed to fetch commit {} for {}: {:?}",
                       self.stager.head().0,
                       self.project.name,
                       err);
            },
        }

        let (staged_topics, msg) = if !self.keep_topics {
                let msg = "Pushed for nightly testing. It is no longer on the stage; any further \
                           updates will require the topic to be staged again.";
                (Cow::Owned(self.stager.clear()), msg)
            } else {
                let msg = "Pushed for nightly testing.";
                (Cow::Borrowed(self.stager.topics()), msg)
            };
        let state_desc = format!("staged for nightly testing {}", stage_ref);
        for staged_topic in staged_topics.iter() {
            let mr_res = self.hosted_mr(&staged_topic.topic);
            match mr_res {
                Ok(mr) => {
                    self.send_mr_commit_status(mr.as_ref(),
                                               CommitStatusState::Success,
                                               &state_desc);

                    self.send_mr_comment(mr.as_ref(), msg);
                },
                Err(err) => {
                    error!(target: "workflow/stage",
                           "failed to fetch mr {} for {}: {:?}",
                           staged_topic.topic.id,
                           self.project.name,
                           err);
                },
            }
        }

        // Push the new stage to the remote.
        Ok(try!(self.update_current_ref()))
    }

    fn update_stage_base(&mut self, candidate: CandidateTopic) -> Result<(), Error> {
        let stage_result = try!(self.stager.stage(candidate));

        // Update topics have been punted off of the stage (successfully staged commits are fine).
        for topic in &stage_result.results {
            try!(self.update_mr_state(topic, false));
        }

        Ok(())
    }

    fn update_stage_mr(&mut self, candidate: CandidateTopic, old_commit: Option<&HostedCommit>)
                       -> Result<(), Error> {
        let stage_result = try!(self.stager.stage(candidate));

        old_commit.map(|commit| {
            // We use success here because it was successfully removed from the stage. A failure
            // would cause a old commits to never be shown as "passing" where this information
            // might be useful at a glance.
            self.send_commit_status(commit, CommitStatusState::Success, "removed from the stage")
        });

        // If no results were made, the base branch was updated and no topics were already staged;
        // everything is fine.
        let results = stage_result.results[..].split_last();
        if let Some((new_topic, restaged_topics)) = results {
            // Update topics have been punted off of the stage (successfully staged commits are
            // fine).
            for topic in restaged_topics {
                try!(self.update_mr_state(topic, false));
            }

            try!(self.update_mr_state(new_topic, true));
        }

        Ok(())
    }

    fn update_current_ref(&self) -> Result<(), io::Error> {
        let ctx = self.stager.git_context();
        let refname = format!("refs/stage/{}/current", self.branch);

        let update_ref = try!(ctx.git()
            .arg("update-ref")
            .arg(&refname)
            .arg(&self.stager.head().0)
            .status());
        try!(check_status(update_ref, "update-ref current failed"));

        let push = try!(ctx.git()
            .arg("push")
            .arg("origin")
            .arg("--atomic")
            .arg(format!("+{}:{}", refname, refname))
            .stdout(Stdio::null())
            .stderr(Stdio::null())
            .status());
        check_status(push, "push failed")
    }

    fn tag_current_ref(&self) -> Result<String, io::Error> {
        let ctx = self.stager.git_context();
        let now = UTC::now();
        let refname = format!("refs/stage/{}/{}",
                              self.branch,
                              now.format("%Y/%m%d").to_string());

        let update_ref = try!(ctx.git()
            .arg("update-ref")
            .arg(&refname)
            .arg(&self.stager.head().0)
            .arg("0000000000000000000000000000000000000000")
            .status());
        try!(check_status(update_ref, "update-ref nightly tag failed"));

        let nightly_refname = format!("refs/stage/{}/nightly", self.branch);

        let update_ref_nightly = try!(ctx.git()
            .arg("update-ref")
            .arg(&nightly_refname)
            .arg(&self.stager.head().0)
            .status());
        try!(check_status(update_ref_nightly, "update-ref nightly failed"));

        let push = try!(ctx.git()
            .arg("push")
            .arg("origin")
            .arg("--atomic")
            .arg(format!("+{}:{}", nightly_refname, nightly_refname))
            .arg(&refname)
            .stdout(Stdio::null())
            .stderr(Stdio::null())
            .status());
        try!(check_status(push, "push failed"));

        Ok(refname)
    }

    fn hosted_mr(&self, topic: &Topic) -> Result<Box<HostedMergeRequest>, HostError> {
        self.project.merge_request(topic.id)
    }

    fn update_mr_state(&self, result: &IntegrationResult, post_success: bool) -> Result<(), Error> {
        let mr = try!(self.hosted_mr(result.topic()));
        match *result {
            IntegrationResult::Staged(_) => {
                self.send_mr_commit_status(mr.as_ref(),
                                           CommitStatusState::Success,
                                           "merged into the stage");

                if post_success {
                    self.send_info_mr_comment(mr.as_ref(), "Successfully staged!");
                }
            },
            IntegrationResult::Unstaged(_, ref reason) => {
                self.send_mr_commit_status(mr.as_ref(),
                                           CommitStatusState::Failed,
                                           &format!("failed to merge: {}",
                                                    unstaged_status_desc(&reason)));
                self.send_mr_comment(mr.as_ref(), &unstaged_status_message(reason));
            },
            IntegrationResult::Unmerged(_, ref reason) => {
                self.send_mr_commit_status(mr.as_ref(),
                                           CommitStatusState::Failed,
                                           &format!("removed from the stage: {}",
                                                    unmerged_status_desc(&reason)));
                self.send_mr_comment(mr.as_ref(), &unmerged_status_message(reason));
            },
        }

        Ok(())
    }

    fn send_mr_commit_status(&self, mr: &HostedMergeRequest, status: CommitStatusState,
                             desc: &str) {
        let status = mr.create_commit_status(status, "kwrobot-stager", desc);
        if let Err(err) = self.project.service.post_commit_status(&status) {
            warn!(target: "workflow/stage",
                  "failed to post a commit status for mr {} on {} for '{}': {:?}",
                  mr.id(),
                  mr.commit().id(),
                  desc,
                  err);
        }
    }

    fn send_commit_status(&self, commit: &HostedCommit, status: CommitStatusState, desc: &str) {
        let status = commit.create_commit_status(status, "kwrobot-stager", desc);
        if let Err(err) = self.project.service.post_commit_status(&status) {
            warn!(target: "workflow/stage",
                  "failed to post a commit status on {} for '{}': {:?}",
                  commit.id(),
                  desc,
                  err);
        }
    }

    fn send_commit_comment(&self, commit: &HostedCommit, content: &str) {
        if let Err(err) = self.project.service.post_commit_comment(commit, content) {
            error!(target: "workflow/stage",
                   "failed to post a comment to commit: {}, {}: {:?}",
                   self.project.name,
                   commit.id(),
                   err);
        }
    }

    fn send_mr_comment(&self, mr: &HostedMergeRequest, content: &str) {
        if let Err(err) = self.project.service.post_mr_comment(mr, content) {
            error!(target: "workflow/stage",
                   "failed to post a comment to merge request: {}, {}: {:?}",
                   self.project.name,
                   mr.id(),
                   err);
        }
    }

    fn send_info_mr_comment(&self, mr: &HostedMergeRequest, content: &str) {
        if !self.quiet {
            self.send_mr_comment(mr, content)
        }
    }
}

fn unstaged_status_desc(reason: &UnstageReason) -> String {
    match *reason {
        UnstageReason::MergeConflict(ref conflicts) => {
            format!("{} conflicting paths", conflicts.iter().dedup().count())
        },
    }
}

fn unstaged_status_message(reason: &UnstageReason) -> String {
    let reason_message = match *reason {
        UnstageReason::MergeConflict(ref conflicts) => {
            let mut conflict_paths = conflicts.iter()
                .map(|conflict| conflict.path().to_string_lossy())
                .dedup();

            format!("Merge conflicts in the following paths:  \n\
                     `{}`",
                    conflict_paths.join("`  \n`"))
        },
    };

    format!("This merge request has been removed from the stage \
             due to:  \n{}", reason_message)
}

fn unmerged_status_desc(reason: &MergeStatus) -> &str {
    match *reason {
        MergeStatus::NoCommonHistory => "no common history",
        MergeStatus::AlreadyMerged => "already merged",
        MergeStatus::Mergeable(_) => {
            error!(target: "workflow/stage",
                   "mergeable unmergeable state?");
            "mergeable?"
        },
    }
}

fn unmerged_status_message(reason: &MergeStatus) -> String {
    let reason_message = match *reason {
        MergeStatus::NoCommonHistory => "there is no common history",
        MergeStatus::AlreadyMerged => "it has already been merged",
        MergeStatus::Mergeable(_) => "it is…mergeable? Sorry, something went wrong",
    };

    format!("This merge request has been removed from the stage \
             because {}.", reason_message)
}
