// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The `clone` action.
//!
//! This action clones a repository such that it is configured for use by other workflow actions.

extern crate kitware_git;
use self::kitware_git::{check_status, GitContext};

use super::super::host::*;

use std::fs::create_dir_all;
use std::io;
use std::path::{Path, PathBuf};

quick_error! {
    #[derive(Debug)]
    /// An error within the `stage` action.
    pub enum Error {
        /// An error occurred while performing a git action.
        Git(err: io::Error) {
            cause(err)
            display("git error: {:?}", err)
            from()
        }
        /// An error occurred while communicating with the hosting service.
        Host(err: HostError) {
            cause(err)
            display("hosting error: {:?}", err)
            from()
        }
    }
}

/// Implementation of the `clone` action.
///
/// Repositories need to be cloned in order for other actions to work. This action bootstraps gets
/// a repository onto the local filesystem and prepared the right way.
pub struct Clone_ {
    workdir: PathBuf,
    project: HostedProject,
}

impl Clone_ {
    /// Create a new clone action.
    pub fn new<W: AsRef<Path>>(workdir: W, project: HostedProject) -> Self {
        Clone_ {
            workdir: workdir.as_ref().join(format!("{}.git", project.name)),
            project: project,
        }
    }

    /// Check if the repository is already cloned.
    pub fn exists(&self) -> bool {
        self.repo_dir().exists()
    }

    /// Clone a repository which is set up to mirror specific refs of a remote repository.
    pub fn clone_mirror_repo<R: ToString>(self, refs: &[R]) -> Result<GitContext, Error> {
        let repo = try!(self.project.service.repo(&self.project.name));

        let ctx = try!(self.setup_clone_from(repo.url()));

        let clear_fetch = try!(ctx.git()
            .arg("config")
            .arg("--unset-all")
            .arg("remote.origin.fetch")
            .status());
        if let Some(5) = clear_fetch.code() {
            // git config --unset return 5 if there were no matches.
        } else {
            try!(check_status(clear_fetch, "config --unset-all remote.origin.fetch failed"));
        }

        for ref_ in refs {
            let refname = ref_.to_string();
            let refs = try!(ctx.git()
                .arg("config")
                .arg("--add")
                .arg("remote.origin.fetch")
                .arg(format!("+{}:{}", refname, refname))
                .status());
            try!(check_status(refs,
                              &format!("config remote.origin.fetch for {} failed", refname)));
        }

        try!(self.fetch_configured(&ctx));

        Ok(ctx)
    }

    /// Clone a repository which will be updated manually.
    ///
    /// These repositories should be managed manually, such as triggered by notifications that the
    /// remote repository has been updated or on a timer.
    pub fn clone_watched_repo(self) -> Result<GitContext, Error> {
        let repo = try!(self.project.service.repo(&self.project.name));

        let ctx = try!(self.setup_clone_from(repo.url()));

        // Tags should not be part of watched repos.
        let no_tags = try!(ctx.git()
            .arg("config")
            .arg("remote.origin.tagopt")
            .arg("--no-tags")
            .status());
        try!(check_status(no_tags, "config remote.origin.tagopt failed"));

        // Fetch the data into the repository.
        try!(self.fetch_heads(&ctx));

        Ok(ctx)
    }

    fn setup_clone_from(&self, url: &str) -> Result<GitContext, Error> {
        if self.exists() {
            return Ok(GitContext::new(self.repo_dir()));
        }

        try!(create_dir_all(self.repo_dir()));

        let ctx = GitContext::new(self.repo_dir());

        info!(target: "workflow/clone",
              "creating a clone from {} into {:?} for {}",
              url,
              self.repo_dir(),
              self.project.name);

        let init = try!(ctx.git()
            .arg("--bare")
            .arg("init")
            .status());
        try!(check_status(init, "init failed"));

        // Set the url for the origin remote.
        let remote = try!(ctx.git()
            .arg("config")
            .arg("remote.origin.url")
            .arg(url)
            .status());
        try!(check_status(remote, "config remote.origin.url failed"));

        // All ref updates should be logged.
        let log_all_ref_updates = try!(ctx.git()
            .arg("config")
            .arg("core.logAllRefUpdates")
            .arg("true")
            .status());
        try!(check_status(log_all_ref_updates, "config core.logAllRefUpdates failed"));

        Ok(ctx)
    }

    fn fetch_configured(&self, ctx: &GitContext) -> Result<(), Error> {
        info!(target: "workflow/clone",
              "fetching initial pre-configured refs into {:?}",
              self.repo_dir());

        let fetch = try!(ctx.git()
            .arg("fetch")
            .arg("origin")
            .status());
        try!(check_status(fetch, "fetch failed"));

        Ok(())
    }

    fn fetch_heads(&self, ctx: &GitContext) -> Result<(), Error> {
        info!(target: "workflow/clone",
              "fetching initial branch refs into {:?}",
              self.repo_dir());

        let fetch = try!(ctx.git()
            .arg("fetch")
            .arg("origin")
            .arg("refs/heads/*:refs/heads/*")
            .status());
        try!(check_status(fetch, "fetch failed"));

        Ok(())
    }

    fn repo_dir(&self) -> &PathBuf {
        &self.workdir
    }
}
