// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::fs::create_dir_all;
use std::path::Path;
use std::process::Command;
use std::sync::Arc;

use git_workarea::GitContext;

use crate::actions::clone::Clone_;
use crate::actions::sync::*;
use crate::host::HostedProject;
use crate::tests::mock::{MockData, MockService};
use crate::tests::utils::test_workspace_dir;

static PROJECT_NAME: &str = "origin";

fn create_sync(path: &Path, service: &Arc<MockService>) -> (GitContext, Sync_) {
    let project = HostedProject {
        name: PROJECT_NAME.to_string(),
        service: service.clone(),
    };

    let refs = [
        "refs/heads/*",
        "refs/tags/*",
    ];

    let localdir = path.join("local");
    let ctx = Clone_::new(localdir, project).clone_mirror_repo(refs).unwrap();

    (ctx.clone(), Sync_::new(ctx))
}

fn add_mirror<Rs>(sync: &mut Sync_, path: &Path, name: &str, refs: &[Rs]) -> Result<GitContext, SyncError>
where
    Rs: AsRef<str>,
{
    let mirrordir = path.join(name);
    create_dir_all(&mirrordir).unwrap();

    let init = Command::new("git")
        .arg("init")
        .arg("--bare")
        .arg(&mirrordir)
        .output()
        .unwrap();
    if !init.status.success() {
        panic!(
            "mirror {} init failed: {}",
            name,
            String::from_utf8_lossy(&init.stderr),
        );
    }

    sync.add_mirror(name, mirrordir.to_string_lossy(), refs)?;

    Ok(GitContext::new(mirrordir))
}

fn list_refs(ctx: &GitContext, refs: &str) -> Vec<String> {
    let for_each_ref = ctx.git()
        .arg("for-each-ref")
        .arg(refs)
        .output()
        .unwrap();
    if !for_each_ref.status.success() {
        panic!(
            "for-each-ref failed: {}",
            String::from_utf8_lossy(&for_each_ref.stderr),
        );
    }

    String::from_utf8_lossy(&for_each_ref.stdout)
        .lines()
        .map(ToString::to_string)
        .collect()
}

fn sync_service() -> Arc<MockService> {
    let base = MockData::repo(PROJECT_NAME);
    let mut builder = MockData::builder();
    builder.add_project(base);
    builder.service()
}

#[test]
fn test_sync_origin_mirror() {
    let tempdir = test_workspace_dir();
    let service = sync_service();
    let (_, mut sync) = create_sync(tempdir.path(), &service);

    let refs = &[
        "refs/heads/*",
        "refs/tags/*",
    ];
    let res = add_mirror(&mut sync, tempdir.path(), "origin", refs);
    assert!(res.is_err());

    let err = res.unwrap_err();

    if let SyncError::OriginMirror = err {
        // expected error
    } else {
        panic!("unexpected error: {:?}", err);
    }
}

#[test]
fn test_sync_duplicate_mirror() {
    let tempdir = test_workspace_dir();
    let service = sync_service();
    let (_, mut sync) = create_sync(tempdir.path(), &service);
    let mirror_name = "mirror";

    let refs = &[
        "refs/heads/*",
        "refs/tags/*",
    ];
    add_mirror(&mut sync, tempdir.path(), mirror_name, refs).unwrap();
    let res = add_mirror(&mut sync, tempdir.path(), mirror_name, refs);
    assert!(res.is_err());

    let err = res.unwrap_err();

    if let SyncError::DuplicateMirror { remote } = err {
        assert_eq!(remote, mirror_name);
    } else {
        panic!("unexpected error: {:?}", err);
    }
}

#[test]
fn test_sync_simple_mirror() {
    let tempdir = test_workspace_dir();
    let service = sync_service();
    let (ctx, mut sync) = create_sync(tempdir.path(), &service);

    let refs = &[
        "refs/heads/*",
        "refs/tags/*",
    ];
    let mirror = add_mirror(&mut sync, tempdir.path(), "mirror", refs).unwrap();

    sync.sync().unwrap();

    let local_heads_refs = list_refs(&ctx, "refs/heads/");
    let local_tags_refs = list_refs(&ctx, "refs/tags/");

    let mirror_heads_refs = list_refs(&mirror, "refs/heads/");
    let mirror_tags_refs = list_refs(&mirror, "refs/tags/");

    assert_eq!(local_heads_refs, mirror_heads_refs);
    assert_eq!(local_tags_refs, mirror_tags_refs);
}

#[test]
fn test_sync_mirror_update() {
    let tempdir = test_workspace_dir();
    let service = sync_service();
    let (ctx, mut sync) = create_sync(tempdir.path(), &service);

    let refs = &[
        "refs/heads/*",
        "refs/tags/*",
    ];
    let mirror = add_mirror(&mut sync, tempdir.path(), "mirror", refs).unwrap();

    sync.sync().unwrap();

    let old_local_heads_refs = list_refs(&ctx, "refs/heads/");
    let old_local_tags_refs = list_refs(&ctx, "refs/tags/");

    let commit = ctx.git()
        .arg("commit")
        .arg("--allow-empty")
        .arg("-m").arg("a new commit\n")
        .output()
        .unwrap();
    if !commit.status.success() {
        panic!(
            "failed to create a new commit: {}",
            String::from_utf8_lossy(&commit.stderr),
        );
    }

    sync.update().unwrap();

    let local_heads_refs = list_refs(&ctx, "refs/heads/");
    let local_tags_refs = list_refs(&ctx, "refs/tags/");

    assert_ne!(old_local_heads_refs, local_heads_refs);
    assert_ne!(old_local_tags_refs, local_tags_refs);

    sync.sync().unwrap();

    let mirror_heads_refs = list_refs(&mirror, "refs/heads/");
    let mirror_tags_refs = list_refs(&mirror, "refs/tags/");

    assert_eq!(local_heads_refs, mirror_heads_refs);
    assert_eq!(local_tags_refs, mirror_tags_refs);
}

#[test]
fn test_sync_multi_mirror() {
    let tempdir = test_workspace_dir();
    let service = sync_service();
    let (ctx, mut sync) = create_sync(tempdir.path(), &service);

    let refs = &[
        "refs/heads/*",
        "refs/tags/*",
    ];
    let m1 = add_mirror(&mut sync, tempdir.path(), "mirror1", refs).unwrap();
    let m2 = add_mirror(&mut sync, tempdir.path(), "mirror2", refs).unwrap();

    sync.sync().unwrap();

    unimplemented!()
}
