// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The `sync` action.
//!
//! This action keeps multiple repositories up to date based on a master repository.

use std::fmt::{self, Debug};

use git_workarea::{GitContext, GitError};
use itertools::Itertools;
use log::info;
use rayon::prelude::*;
use thiserror::Error;

/// Errors which may occur when synchronizing repositories.
#[derive(Debug, Error)]
#[non_exhaustive]
pub enum SyncError {
    /// The "origin" remote may not be a mirror.
    #[error("'origin' may not be a remote mirror")]
    OriginMirror,
    /// Two mirrors may not have the same name.
    #[error("the {} remote mirror already exists", remote)]
    DuplicateMirror {
        /// The duplicate remote name.
        remote: String,
    },
    /// Failed to set the URL for a remote.
    #[error("failed to set the url for the {} remote: {}", remote, output)]
    SetRemoteUrl {
        /// The remote being set.
        remote: String,
        /// The output of `git config`.
        output: String,
    },
    /// Failed to clear the push refspecs for a remote.
    #[error("failed to clear the push refspecs for the {} remote: {}", remote, output)]
    ClearRemotePush {
        /// The remote being cleared.
        remote: String,
        /// The output of `git config`.
        output: String,
    },
    /// Failed to add a push refspec for a remote.
    #[error("failed to add the {} refspec to the push refspecs for the {} remote: {}", refspec, remote, output)]
    AddRemotePush {
        /// The remote being modified.
        remote: String,
        /// The refspec being added.
        refspec: String,
        /// The output of `git config`.
        output: String,
    },
    /// Failed to fetch from the main repository.
    #[error("failed to fetch from origin: {}", output)]
    Fetch {
        /// The output of `git fetch`.
        output: String,
    },
    /// Failed to push to a mirror.
    #[error("failed to push to {}: {}", remote, output)]
    Push {
        /// The remote being pushed to.
        remote: String,
        /// The output of `git push`.
        output: String,
    },
    /// Failure to execute a `git` command.
    #[error("git error: {}", source)]
    Git {
        /// The source of the error.
        #[from]
        source: GitError,
    },
}

impl SyncError {
    fn origin_mirror() -> Self {
        Self::OriginMirror
    }

    fn duplicate_mirror(remote: String) -> Self {
        Self::DuplicateMirror {
            remote,
        }
    }

    fn set_remote_url(remote: String, output: &[u8]) -> Self {
        Self::SetRemoteUrl {
            remote,
            output: String::from_utf8_lossy(output).into(),
        }
    }

    fn clear_remote_push(remote: String, output: &[u8]) -> Self {
        Self::ClearRemotePush {
            remote,
            output: String::from_utf8_lossy(output).into(),
        }
    }

    fn add_remote_push(remote: String, refspec: String, output: &[u8]) -> Self {
        Self::AddRemotePush {
            remote,
            refspec,
            output: String::from_utf8_lossy(output).into(),
        }
    }

    fn fetch(output: &[u8]) -> Self {
        Self::Fetch {
            output: String::from_utf8_lossy(output).into(),
        }
    }

    fn push(remote: String, output: &[u8]) -> Self {
        Self::Push {
            remote,
            output: String::from_utf8_lossy(output).into(),
        }
    }
}

type SyncResult<T> = Result<T, SyncError>;

/// Implementation of the `sync` action.
pub struct Sync_ {
    /// The context to use for synchronizing repositories.
    ctx: GitContext,
    /// The list of remotes to sync into.
    remotes: Vec<String>,
}

impl Sync_ {
    /// Create a new sync action.
    pub fn new(ctx: GitContext) -> Self {
        Sync_ {
            ctx,
            remotes: Vec::new(),
        }
    }

    /// Add a remote repository which should mirror the main repository.
    pub fn add_mirror<R, U, I, Rs>(&mut self, remote: R, url: U, refs: I) -> SyncResult<()>
    where
        R: Into<String>,
        U: AsRef<str>,
        I: IntoIterator<Item = Rs>,
        Rs: AsRef<str>,
    {
        self.add_mirror_impl(remote.into(),
                             url.as_ref(),
                             refs.into_iter()
                                 .map(|refname| {
                                     format!("+{}:{}", refname.as_ref(), refname.as_ref())
                                 })
                                 .collect())
    }

    fn add_mirror_impl(&mut self, remote: String, url: &str, refspecs: Vec<String>) -> SyncResult<()> {
        if remote == "origin" {
            return Err(SyncError::origin_mirror());
        }

        if self.remotes.iter().any(|r| r == &remote) {
            return Err(SyncError::duplicate_mirror(remote));
        }

        // Set the remote url.
        let config_remote = self.ctx
            .git()
            .arg("config")
            .arg(format!("remote.{}.url", remote))
            .arg(url)
            .output()
            .map_err(|err| GitError::subcommand("config remote.url", err))?;
        if !config_remote.status.success() {
            return Err(SyncError::set_remote_url(remote, &config_remote.stderr));
        }

        // Clear all push refs.
        let config_push_clear = self.ctx
            .git()
            .arg("config")
            .arg("--unset-all")
            .arg(format!("remote.{}.push", remote))
            .output()
            .map_err(|err| GitError::subcommand("config --unset-all remote.push", err))?;
        if let Some(5) = config_push_clear.status.code() {
            // git config --unset return 5 if there were no matches.
        } else if !config_push_clear.status.success() {
            return Err(SyncError::clear_remote_push(remote, &config_push_clear.stderr));
        }

        // Set the requested push refs.
        refspecs.into_iter()
            .map(|refspec| {
                let config_add_ref = self.ctx
                    .git()
                    .arg("config")
                    .arg("--add")
                    .arg(format!("remote.{}.push", remote))
                    .arg(&refspec)
                    .output()
                    .map_err(|err| GitError::subcommand("config --add remote.push", err))?;
                if !config_add_ref.status.success() {
                    return Err(SyncError::add_remote_push(remote.clone(), refspec, &config_add_ref.stderr));
                }

                Ok(())
            })
            .collect::<SyncResult<Vec<_>>>()?;

        self.remotes.push(remote);

        Ok(())
    }

    /// Update the local repository from the main repository.
    pub fn update(&self) -> SyncResult<()> {
        info!(
            target: "ghostflow/sync",
            "fetching updates",
        );

        let fetch = self.ctx
            .git()
            .arg("fetch")
            .arg("origin")
            .output()
            .map_err(|err| GitError::subcommand("fetch", err))?;
        if !fetch.status.success() {
            return Err(SyncError::fetch(&fetch.stderr));
        }

        Ok(())
    }

    /// Push to all mirrors.
    pub fn sync(&self) -> SyncResult<()> {
        self.remotes
            .par_iter()
            .map(|remote| {
                info!(
                    target: "ghostflow/sync",
                    "pushing to {}",
                    remote,
                );

                let push = self.ctx
                    .git()
                    .arg("push")
                    .arg("--atomic")
                    .arg(remote)
                    .output()
                    .map_err(|err| GitError::subcommand("push", err))?;
                if !push.status.success() {
                    return Err(SyncError::push(remote.clone(), &push.stderr));
                }

                Ok(())
            })
            .collect::<Vec<SyncResult<_>>>()
            .into_iter()
            .collect::<SyncResult<Vec<_>>>()?;

        Ok(())
    }
}

impl Debug for Sync_ {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            "Sync {{ gitdir: {}, remotes: ['{}'] }}",
            self.ctx.gitdir().to_string_lossy(),
            self.remotes.iter().format("', '"),
        )
    }
}
