// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::collections::hash_map::HashMap;
use std::fmt;
use std::iter;
use std::sync::{Arc, OnceLock};
use std::thread;
use std::time::Duration;

use chrono::Utc;
use ghostflow::host::*;
use git_workarea::CommitId;
use itertools::Itertools;
use regex::Regex;
use serde::de::DeserializeOwned;
use thiserror::Error;
use url::Url;

/// The `gitlab` crate in use.
///
/// Due to the odd versioning scheme of the `gitlab` crate, this export makes it easier to match
/// the version of `gitlab` crate.
pub use gitlab;

use gitlab::api::{self, Query};

mod types;

#[derive(Debug, Clone, Copy, PartialEq)]
enum Retry {
    Yes,
    No,
}

// The maximum number of times we will retry server errors.
const BACKOFF_LIMIT: usize = if cfg!(test) { 2 } else { 5 };
// The number of seconds to start retries at.
const BACKOFF_START: Duration = Duration::from_secs(1);
// How much to scale retry timeouts for a single query.
const BACKOFF_SCALE: u32 = 2;

fn should_backoff<E>(err: &api::ApiError<E>) -> bool
where
    E: std::error::Error + Send + Sync + 'static,
{
    match err {
        api::ApiError::GitlabWithStatus {
            status, ..
        } => {
            // Sometimes on an MR, a webhook can be received and acted upon before the target
            // project "sees" the commit through the API.
            *status == http::StatusCode::NOT_FOUND
        },
        _ => false,
    }
}

fn retry_with_backoff<F, E, K>(mut tryf: F) -> Result<K, api::ApiError<E>>
where
    F: FnMut() -> Result<K, api::ApiError<E>>,
    E: std::error::Error + Send + Sync + 'static,
{
    iter::repeat_n((), BACKOFF_LIMIT)
        .scan(BACKOFF_START, |timeout, _| {
            match tryf() {
                Ok(r) => Some(Some(Ok(r))),
                Err(err) => {
                    if should_backoff(&err) {
                        thread::sleep(*timeout);
                        *timeout *= BACKOFF_SCALE;
                        Some(None)
                    } else {
                        Some(Some(Err(err)))
                    }
                },
            }
        })
        .flatten()
        .next()
        .unwrap_or_else(|| {
            let msg = "failed even after exponential backoff".into();
            Err(api::ApiError::GitlabWithStatus {
                status: http::StatusCode::NOT_FOUND,
                msg,
            })
        })
}

/// Match messages generated by GitLab `app/services/system_note_service.rb` when a MR branch is
/// updated.
fn mr_update_re() -> &'static Regex {
    static CELL: OnceLock<Regex> = OnceLock::new();
    CELL.get_or_init(|| {
        Regex::new(
            "^[Aa]dded [0-9][0-9]* (new )?commits?:?\n\
             (\n<ul>(<li>[0-9a-f.]+ - .*?</li>)*</ul>|(\n\\* [0-9a-f.]+ - [^\n]*)*)\
             (\n\n\\[Compare with previous versions?\\]\\(.*\\))?\
             $",
        )
        .expect("invalid `mr_update_re` regex")
    })
}

fn ghostflow_user(user: types::FullUser) -> User {
    User {
        handle: user.username,
        name: user.name,
        email: user.email,
    }
}

fn ghostflow_pipeline_state(status: types::PipelineStatus) -> PipelineState {
    match status {
        types::PipelineStatus::Manual => PipelineState::Manual,
        types::PipelineStatus::Created
        | types::PipelineStatus::WaitingForResource
        | types::PipelineStatus::Preparing
        | types::PipelineStatus::WaitingForCallback
        | types::PipelineStatus::Pending
        | types::PipelineStatus::Scheduled
        | types::PipelineStatus::Running => PipelineState::InProgress,
        types::PipelineStatus::Canceling
        | types::PipelineStatus::Canceled
        | types::PipelineStatus::Skipped => PipelineState::Canceled,
        types::PipelineStatus::Failed => PipelineState::Failed,
        types::PipelineStatus::Success => PipelineState::Success,
    }
}

fn gitlab_commit_status_state(
    state: CommitStatusState,
) -> api::projects::repository::commits::CommitStatusState {
    match state {
        CommitStatusState::Pending => {
            api::projects::repository::commits::CommitStatusState::Pending
        },
        CommitStatusState::Running => {
            api::projects::repository::commits::CommitStatusState::Running
        },
        CommitStatusState::Success => {
            api::projects::repository::commits::CommitStatusState::Success
        },
        CommitStatusState::Failed => api::projects::repository::commits::CommitStatusState::Failed,
    }
}

fn ghostflow_commit_status_state(state: types::StatusState) -> CommitStatusState {
    match state {
        types::StatusState::Manual
        | types::StatusState::Skipped
        | types::StatusState::Created
        | types::StatusState::Canceling
        | types::StatusState::Canceled
        | types::StatusState::Pending
        | types::StatusState::Scheduled => CommitStatusState::Pending,
        types::StatusState::Running => CommitStatusState::Running,
        types::StatusState::Success => CommitStatusState::Success,
        types::StatusState::Failed => CommitStatusState::Failed,
    }
}

/// An entity on the service which may be linked to with a shorthand.
trait ReferenceTarget {
    /// The sigil to use for the target.
    fn sigil() -> char;
    /// The ID of the target.
    fn id(&self) -> u64;
}

impl ReferenceTarget for types::Issue {
    fn sigil() -> char {
        '#'
    }

    fn id(&self) -> u64 {
        self.iid
    }
}

impl ReferenceTarget for types::MergeRequest {
    fn sigil() -> char {
        '!'
    }

    fn id(&self) -> u64 {
        self.iid
    }
}

/// The level of reference needed to link to a target from a source.
#[derive(Debug, Default, Clone, Copy, PartialEq, Eq)]
enum ReferenceLevel {
    /// The source and target are part of the same project.
    #[default]
    Project,
    /// The source and target are part of the same namespace.
    Namespace,
    /// The source and target are part of the same site.
    Site,
    // If support is added, a `url` method needs to be added to the trait.
    // /// The source and target are on separate instances.
    // Full,
}

impl ReferenceLevel {
    fn between(source: &types::Project, target: &types::Project) -> Self {
        if source.id == target.id {
            ReferenceLevel::Project
        } else if source.namespace == target.namespace {
            ReferenceLevel::Namespace
        } else {
            ReferenceLevel::Site
        }
    }

    fn to<T>(self, project: &types::Project, target: &T) -> String
    where
        T: ReferenceTarget,
    {
        match self {
            ReferenceLevel::Project => format!("{}{}", T::sigil(), target.id()),
            ReferenceLevel::Namespace => format!("{}{}{}", project.path, T::sigil(), target.id()),
            ReferenceLevel::Site => {
                format!(
                    "{}/{}{}{}",
                    project.namespace.path,
                    project.path,
                    T::sigil(),
                    target.id(),
                )
            },
        }
    }
}

/// Structure used to communicate with a Gitlab instance.
pub struct GitlabService {
    /// The Gitlab client.
    gitlab: gitlab::Gitlab,
    /// The user the service is acting as.
    user: User,
    /// Whether the user is an administrator or not.
    is_admin: bool,
    /// The domain we're communicating with.
    domain: &'static str,
}

impl GitlabService {
    /// Create a new GitLab communication channel.
    pub fn new(gitlab: gitlab::Gitlab) -> Result<Self, HostingServiceError> {
        let endpoint = api::users::CurrentUser::builder().build().unwrap();
        let self_user: types::SelfUser =
            endpoint.query(&gitlab).map_err(HostingServiceError::host)?;
        let is_admin = self_user.is_admin;
        let gitlab_user = types::FullUser {
            username: self_user.username,
            name: self_user.name,
            email: self_user.email,
        };
        let user = ghostflow_user(gitlab_user);

        Ok(Self {
            user,
            gitlab,
            is_admin,
            // TODO: get the domain from the client.
            domain: "gitlab.invalid",
        })
    }

    /// Access the underlying GitLab client.
    pub fn gitlab(&self) -> &gitlab::Gitlab {
        &self.gitlab
    }

    fn raw_query<Q, T>(&self, query: &Q) -> Result<T, api::ApiError<gitlab::RestError>>
    where
        Q: api::Query<T, gitlab::Gitlab>,
        T: DeserializeOwned,
    {
        query.query(&self.gitlab)
    }

    fn query<Q, T>(&self, query: &Q) -> Result<T, HostingServiceError>
    where
        Q: api::Query<T, gitlab::Gitlab>,
        T: DeserializeOwned,
    {
        self.raw_query(query).map_err(HostingServiceError::host)
    }

    fn full_project<'a, T>(&self, project: T) -> Result<types::Project, HostingServiceError>
    where
        T: Into<api::common::NameOrId<'a>>,
    {
        let endpoint = api::projects::Project::builder()
            .project(project)
            .build()
            .unwrap();
        self.query(&endpoint)
    }

    fn full_user_by_id(&self, id: u64) -> Result<types::User, HostingServiceError> {
        let endpoint = api::users::User::builder().user(id).build().unwrap();
        self.query(&endpoint)
    }

    fn full_user_by_name(&self, name: &str) -> Result<types::User, HostingServiceError> {
        let endpoint = api::users::Users::builder().username(name).build().unwrap();
        // We're assuming the search by username shows up on the first page.
        let users: Vec<types::User> = self.query(&endpoint)?;

        users
            .into_iter()
            .find(|user| user.username == name)
            .ok_or_else(|| HostingServiceError::host(GitlabServiceError::no_such_user(name.into())))
    }

    fn user(&self, id: u64) -> Result<User, HostingServiceError> {
        self.full_user_by_id(id)
            .map(|user: types::User| ghostflow_user(user.for_domain(self.domain)))
    }

    fn user_by_name(&self, name: &str) -> Result<User, HostingServiceError> {
        self.full_user_by_name(name)
            .map(|user| ghostflow_user(user.for_domain(self.domain)))
    }

    fn pipeline(
        &self,
        pipeline: types::SinglePipeline,
        repo: Repo,
        latest: Option<u64>,
    ) -> Pipeline {
        Pipeline {
            id: pipeline.id,
            state: ghostflow_pipeline_state(pipeline.status),
            commit: Commit {
                repo,
                id: CommitId::new(pipeline.sha),
                refname: pipeline.ref_,
                last_pipeline: latest,
            },
            user: ghostflow_user(pipeline.user.for_domain(self.domain)),
            archived: pipeline.archived,
        }
    }

    fn job(&self, job: types::PipelineJob, repo: Repo) -> PipelineJob {
        PipelineJob {
            id: job.id,
            state: ghostflow_pipeline_state(job.status),
            repo,
            stage: Some(job.stage),
            name: job.name,
            user: ghostflow_user(job.user.for_domain(self.domain)),
            archived: job.archived,
        }
    }

    /// Create a repository from a Gitlab project.
    fn repo_from_project(&self, project: types::Project) -> Result<Repo, HostingServiceError> {
        let parent_project = if let Some(ref upstream) = project.forked_from_project {
            let parent_project = self.full_project(upstream.id)?;
            Some(Box::new(self.repo_from_project(parent_project)?))
        } else {
            None
        };

        Ok(Repo {
            name: project.path_with_namespace,
            url: project.ssh_url_to_repo,
            http_url: project.http_url_to_repo,
            forked_from: parent_project,
        })
    }

    /// Create a commit from a Gitlab commit.
    fn commit_from_project(
        &self,
        project: types::Project,
        commit: &CommitId,
        retry: Retry,
    ) -> Result<Commit, HostingServiceError> {
        let endpoint = api::projects::repository::commits::Commit::builder()
            .project(project.id)
            .commit(commit.as_str())
            .build()
            .unwrap();
        let commit: types::Commit = if retry == Retry::Yes {
            retry_with_backoff(|| self.raw_query(&endpoint)).map_err(HostingServiceError::host)?
        } else {
            self.query(&endpoint)?
        };

        Ok(Commit {
            repo: self.repo_from_project(project)?,
            refname: None,
            id: CommitId::new(commit.id),
            last_pipeline: commit.last_pipeline.map(|pipe| pipe.id),
        })
    }

    fn gitlab_issue(
        &self,
        project: types::Project,
        issue: types::Issue,
        referrer: Option<&types::Project>,
    ) -> Result<Issue, HostingServiceError> {
        let reference = referrer.map_or(ReferenceLevel::Project, |source| {
            ReferenceLevel::between(source, &project)
        });

        Ok(Issue {
            reference: reference.to(&project, &issue),
            repo: self.repo_from_project(project)?,
            id: issue.iid,
            url: issue.web_url,
            labels: issue.labels,
        })
    }

    /// Create a merge request from a Gitlab merge request.
    fn merge_request_from_project(
        &self,
        project: types::Project,
        id: u64,
    ) -> Result<MergeRequest, HostingServiceError> {
        let endpoint = api::projects::merge_requests::MergeRequest::builder()
            .project(project.id)
            .merge_request(id)
            .build()
            .unwrap();
        let mr: types::MergeRequest = endpoint
            .query(&self.gitlab)
            .map_err(HostingServiceError::host)?;
        let source_project = self.full_project(mr.source_project_id)?;
        let author = self.user(mr.author.id)?;

        let reference = ReferenceLevel::default().to(&project, &mr);

        let source_repo = self.repo_from_project(source_project.clone())?;
        let target_repo = self.repo_from_project(project.clone())?;

        // If `mr.sha` is `None`, the source branch has been deleted. Using an empty string here
        // will basically make anything that uses it fail since it can never be a valid ID for Git
        // or Gitlab. We let this happen because `CommitId` structures are vetted at their usage
        // rather than at construction by going through a call to `git rev-parse` to validate them.
        let mr_commit = if let Some(ref sha) = &mr.sha {
            // Given that the commit exists, we now need to determine whether to associate it with
            // the source or the target repository. We do two queries, one for the commit on each
            // of the two projects. If a pipeline exists on the target project, assume that all
            // checks should go there and make it the primary repo for the commit. Otherwise,
            // fallback to the source project. Note that this can be in a race condition with the
            // processing of the CI configuration file by the GitLab server.
            let commit_id = CommitId::new(sha);
            let source_commit = self.commit_from_project(source_project, &commit_id, Retry::No)?;
            // Retry this as the commit may not be present on the target project by the time we get
            // here.
            let target_retry = if mr.source_project_id == project.id {
                // But not if the MR is to itself; it should always be available here.
                Retry::No
            } else {
                Retry::Yes
            };
            let target_name = project.path_with_namespace.clone();
            let target_commit = self.commit_from_project(project, &commit_id, target_retry)?;

            if let Some(last_pipeline_id) = target_commit.last_pipeline {
                // Discover the refname for the pipeline on the target project. GitLab requires
                // that a refname to be present when creating a commit status.
                let endpoint = api::projects::pipelines::Pipeline::builder()
                    .project(target_name)
                    .pipeline(last_pipeline_id)
                    .build()
                    .unwrap();
                let pipeline: types::SinglePipeline = self.query(&endpoint)?;
                let mut commit = target_commit;
                commit.refname = pipeline.ref_;
                commit
            } else {
                let mut commit = source_commit;
                // Ensure that the commit's refname is preserved.
                commit.refname = Some(mr.source_branch.clone());
                commit
            }
        } else {
            // The source branch has been deleted.
            Commit {
                repo: source_repo.clone(),
                refname: Some(mr.source_branch.clone()),
                id: CommitId::new(""),
                last_pipeline: None,
            }
        };

        Ok(MergeRequest {
            source_repo: Some(source_repo),
            source_branch: mr.source_branch,
            target_repo,
            target_branch: mr.target_branch,
            id: mr.iid,
            url: mr.web_url,
            work_in_progress: mr.work_in_progress,
            description: mr.description.unwrap_or_default(),
            old_commit: None,
            commit: mr_commit,
            author,
            reference,
            remove_source_branch: mr.force_remove_source_branch.unwrap_or(false),
            merge_after: mr.merge_after,
        })
    }

    /// Sort comments according to the ID numbers.
    fn sort_notes(&self, notes: Vec<types::Note>) -> Result<Vec<Comment>, HostingServiceError> {
        Ok(notes
            .into_iter()
            .map(|note| {
                Ok(Comment {
                    id: format!("{}", note.id),
                    is_system: note.system,
                    is_branch_update: note.system && mr_update_re().is_match(&note.body),
                    created_at: note.created_at,
                    author: self.user(note.author.id)?,
                    content: note
                        .body
                        // Remove ZERO WIDTH SPACE from the command. This can occur when
                        // copy/pasting contents from an email or rendering of a command into a
                        // comment box. Due to its invisibility, it causes confusion when the
                        // resulting `not recognized at all` text appears with the invisible
                        // character hidden in there.
                        .replace('\u{200b}', ""),
                })
            })
            .collect::<Result<Vec<_>, HostingServiceError>>()?
            .into_iter()
            .sorted_by(|a, b| a.id.cmp(&b.id))
            .collect())
    }
}

#[derive(Debug, Error)]
#[non_exhaustive]
enum GitlabServiceError {
    #[error("failed to find a user named '{}'", name)]
    NoSuchUser { name: String },
    #[error("invalid repo URL")]
    InvalidUrl {
        #[source]
        source: url::ParseError,
    },
    #[error("invalid repo URL: could not set username")]
    UrlNoUsername,
    #[error("invalid repo URL: could not set password")]
    UrlNoPassword,
    #[error(
        "job `{}` is archived and may not be started; the pipeline needs recreated",
        name
    )]
    ArchivedJob { name: String },
}

impl GitlabServiceError {
    fn no_such_user(name: String) -> Self {
        GitlabServiceError::NoSuchUser {
            name,
        }
    }

    fn invalid_url(source: url::ParseError) -> Self {
        Self::InvalidUrl {
            source,
        }
    }

    fn url_no_username() -> Self {
        Self::UrlNoUsername
    }

    fn url_no_password() -> Self {
        Self::UrlNoPassword
    }

    fn archived_job(name: String) -> Self {
        Self::ArchivedJob {
            name,
        }
    }
}

impl From<GitlabServiceError> for HostingServiceError {
    fn from(gitlab: GitlabServiceError) -> Self {
        HostingServiceError::service(gitlab)
    }
}

impl HostingService for GitlabService {
    fn suppress_ci_push_option(&self, branch: &str) -> Option<String> {
        // TODO: https://gitlab.com/gitlab-org/gitlab/-/merge_requests/170960
        Some(format!("branch.{branch}.ci.skip"))
    }

    fn as_pipeline_service(self: Arc<Self>) -> Option<Arc<dyn HostedPipelineService>> {
        Some(self as Arc<dyn HostedPipelineService>)
    }

    fn service_user(&self) -> &User {
        &self.user
    }

    fn user(&self, project: &str, user: &str) -> Result<User, HostingServiceError> {
        let _ = self.full_project(project)?;
        self.user_by_name(user)
    }

    fn commit(&self, project: &str, commit: &CommitId) -> Result<Commit, HostingServiceError> {
        let project = self.full_project(project)?;
        self.commit_from_project(project, commit, Retry::No)
    }

    fn merge_request(&self, project: &str, id: u64) -> Result<MergeRequest, HostingServiceError> {
        let project = self.full_project(project)?;
        self.merge_request_from_project(project, id)
    }

    fn repo(&self, project: &str) -> Result<Repo, HostingServiceError> {
        let project = self.full_project(project)?;
        self.repo_from_project(project)
    }

    fn repo_as_user(
        &self,
        repo: &Repo,
        user: &User,
        reason: &str,
    ) -> Result<Option<Repo>, HostingServiceError> {
        // Impersonation tokens require administrator access.
        if !self.is_admin {
            return Ok(None);
        }

        let url = Url::parse(&repo.http_url).map_err(GitlabServiceError::invalid_url)?;

        let token = {
            let gl_user = self.full_user_by_name(&user.handle)?;
            let token_name = format!("ghostflow-impersonation-{}-{reason}", user.handle);
            let today = Utc::now().date_naive();
            // Add up to 2 days so that tokens created at a date boundary do not expire before use.
            let expires_at = iter::repeat_n((), 2).fold(today, |day, _| {
                if let Some(after) = day.succ_opt() {
                    after
                } else {
                    day
                }
            });
            let endpoint = api::users::impersonation_tokens::CreateImpersonationToken::builder()
                .user(gl_user.id)
                .name(token_name)
                .scope(api::users::impersonation_tokens::ImpersonationTokenScope::Api)
                .expires_at(expires_at)
                .build()
                .unwrap();
            let token: types::ImpersonationToken = self.query(&endpoint)?;
            token.token
        };

        let url_with_token = {
            let mut url = url;
            // GitLab doesn't care about the username in Git repository names.
            url.set_username("ghostflow")
                .map_err(|_| GitlabServiceError::url_no_username())?;
            url.set_password(Some(&token))
                .map_err(|_| GitlabServiceError::url_no_password())?;
            url
        };

        let mut new_repo = repo.clone();
        new_repo.url = url_with_token.into();

        Ok(Some(new_repo))
    }

    fn get_mr_comments(&self, mr: &MergeRequest) -> Result<Vec<Comment>, HostingServiceError> {
        let endpoint = api::projects::merge_requests::notes::MergeRequestNotes::builder()
            .project(&mr.target_repo.name)
            .merge_request(mr.id)
            .build()
            .unwrap();
        let endpoint = api::paged(endpoint, api::Pagination::All);
        let notes: Vec<types::Note> = self.query(&endpoint)?;

        self.sort_notes(notes)
    }

    fn post_mr_comment(&self, mr: &MergeRequest, content: &str) -> Result<(), HostingServiceError> {
        let endpoint = api::projects::merge_requests::notes::CreateMergeRequestNote::builder()
            .project(&mr.target_repo.name)
            .merge_request(mr.id)
            .body(content)
            .build()
            .unwrap();
        let endpoint = api::ignore(endpoint);
        self.query(&endpoint)
    }

    fn get_commit_statuses(
        &self,
        commit: &Commit,
    ) -> Result<Vec<CommitStatus>, HostingServiceError> {
        let endpoint = api::projects::repository::commits::CommitStatuses::builder()
            .project(commit.repo.name.as_str())
            .commit(commit.id.as_str())
            .build()
            .unwrap();
        let endpoint = api::paged(endpoint, api::Pagination::All);
        let statuses = self.query(&endpoint)?;

        Ok(statuses
            .into_iter()
            .map(move |status: types::CommitStatus| {
                CommitStatus {
                    state: ghostflow_commit_status_state(status.status),
                    author: ghostflow_user(status.author.for_domain(self.domain)),
                    refname: status.ref_,
                    name: status.name,
                    description: status.description.unwrap_or_default(),
                    target_url: status.target_url,
                }
            })
            .collect())
    }

    fn post_commit_status(&self, status: PendingCommitStatus) -> Result<(), HostingServiceError> {
        const REFS_HEADS_PREFIX: &str = "refs/heads/";
        const REFS_TAGS_PREFIX: &str = "refs/tags/";

        // GitLab uses "common" refnames rather than fully-qualified refnames when possible.
        let refname = status.commit.refname.as_ref().map(|refname| {
            if let Some(head_name) = refname.strip_prefix(REFS_HEADS_PREFIX) {
                head_name
            } else if let Some(tag_name) = refname.strip_prefix(REFS_TAGS_PREFIX) {
                tag_name
            } else {
                refname
            }
        });

        let mut builder = api::projects::repository::commits::CreateCommitStatus::builder();
        builder
            .project(status.commit.repo.name.as_str())
            .commit(status.commit.id.as_str())
            .state(gitlab_commit_status_state(status.state))
            .name(status.name)
            .description(status.description);

        if let Some(refname) = refname {
            builder.ref_(refname);
        }
        if let Some(target_url) = status.target_url {
            builder.target_url(target_url);
        }
        if let Some(last_pipeline) = status.commit.last_pipeline {
            builder.pipeline_id(last_pipeline);
        }

        let endpoint = builder.build().unwrap();
        let endpoint = api::ignore(endpoint);
        self.query(&endpoint)
    }

    fn get_mr_awards(&self, mr: &MergeRequest) -> Result<Vec<Award>, HostingServiceError> {
        let endpoint = api::projects::merge_requests::awards::MergeRequestAwards::builder()
            .project(&mr.target_repo.name)
            .merge_request(mr.id)
            .build()
            .unwrap();
        let endpoint = api::paged(endpoint, api::Pagination::All);

        self.query(&endpoint)?
            .into_iter()
            .map(|award: types::AwardEmoji| {
                let author = self.user(award.user.id)?;

                Ok(Award {
                    name: award.name,
                    author,
                })
            })
            .collect()
    }

    fn issues_closed_by_mr(&self, mr: &MergeRequest) -> Result<Vec<Issue>, HostingServiceError> {
        let target_name = &mr.target_repo.name;
        let target_project = self.full_project(target_name.as_str())?;

        let endpoint = api::projects::merge_requests::IssuesClosedBy::builder()
            .project(target_project.id)
            .merge_request(mr.id)
            .build()
            .unwrap();
        let endpoint = api::paged(endpoint, api::Pagination::All);
        let issues: Vec<types::Issue> = self.query(&endpoint)?;

        // Cache projects to reduce hitting the service so much.
        let projects = issues
            .iter()
            .map(|issue| issue.project_id)
            .unique()
            .map(|project_id| {
                self.full_project(project_id)
                    .map(|project| (project_id, project))
            })
            .collect::<Result<HashMap<_, _>, HostingServiceError>>();

        // Link to each issue.
        projects.and_then(|projects| {
            issues
                .into_iter()
                .map(|issue| {
                    let project = projects
                        .get(&issue.project_id)
                        .expect("the fetched project ID should exist");
                    self.gitlab_issue(project.clone(), issue, Some(&target_project))
                })
                .collect()
        })
    }

    fn add_issue_labels(&self, issue: &Issue, labels: &[&str]) -> Result<(), HostingServiceError> {
        let endpoint = {
            let mut endpoint = api::projects::issues::EditIssue::builder();
            endpoint.project(issue.repo.name.as_str()).issue(issue.id);

            for label in labels {
                endpoint.add_label(*label);
            }

            endpoint.build().unwrap()
        };
        let endpoint = api::ignore(endpoint);
        self.query(&endpoint)
    }

    fn remove_issue_labels(
        &self,
        issue: &Issue,
        labels: &[&str],
    ) -> Result<(), HostingServiceError> {
        let endpoint = {
            let mut endpoint = api::projects::issues::EditIssue::builder();
            endpoint.project(issue.repo.name.as_str()).issue(issue.id);

            for label in labels {
                endpoint.remove_label(*label);
            }

            endpoint.build().unwrap()
        };
        let endpoint = api::ignore(endpoint);
        self.query(&endpoint)
    }
}

impl HostedPipelineService for GitlabService {
    fn pipelines_for_mr(
        &self,
        mr: &MergeRequest,
    ) -> Result<Option<Vec<Pipeline>>, HostingServiceError> {
        let source_pipelines = if let Some(source_repo) = mr.source_repo.as_ref() {
            let project = self.full_project(source_repo.name.as_str())?;

            if project.builds_access_level != types::AccessLevel::Disabled {
                let endpoint = api::projects::pipelines::Pipelines::builder()
                    .project(source_repo.name.as_str())
                    .ref_(mr.source_branch.as_str())
                    .sha(mr.commit.id.as_str())
                    .build()
                    .unwrap();
                let endpoint = api::paged(endpoint, api::Pagination::All);
                let pipelines: Vec<types::Pipeline> = self.query(&endpoint)?;
                let repo = self.repo_from_project(project)?;
                let latest = pipelines.iter().map(|pipeline| pipeline.id).max();
                pipelines
                    .into_iter()
                    .map(|pipeline| {
                        let endpoint = api::projects::pipelines::Pipeline::builder()
                            .project(source_repo.name.as_str())
                            .pipeline(pipeline.id)
                            .build()
                            .unwrap();
                        let pipeline: types::SinglePipeline = self.query(&endpoint)?;
                        Ok(self.pipeline(pipeline, repo.clone(), latest))
                    })
                    .collect::<Result<_, HostingServiceError>>()?
            } else {
                Vec::new()
            }
        } else {
            Vec::new()
        };

        let target_pipelines = {
            let project = self.full_project(mr.target_repo.name.as_str())?;

            if project.builds_access_level != types::AccessLevel::Disabled {
                let mr_ref = format!("refs/merge-requests/{}/head", mr.id);
                let endpoint = api::projects::pipelines::Pipelines::builder()
                    .project(mr.target_repo.name.as_str())
                    .ref_(mr_ref)
                    .sha(mr.commit.id.as_str())
                    .build()
                    .unwrap();
                let endpoint = api::paged(endpoint, api::Pagination::All);
                let pipelines: Vec<types::Pipeline> = self.query(&endpoint)?;
                let repo = self.repo_from_project(project)?;
                let latest = pipelines.iter().map(|pipeline| pipeline.id).max();
                pipelines
                    .into_iter()
                    .map(|pipeline| {
                        let endpoint = api::projects::pipelines::Pipeline::builder()
                            .project(mr.target_repo.name.as_str())
                            .pipeline(pipeline.id)
                            .build()
                            .unwrap();
                        let pipeline: types::SinglePipeline = self.query(&endpoint)?;
                        Ok(self.pipeline(pipeline, repo.clone(), latest))
                    })
                    .collect::<Result<_, HostingServiceError>>()?
            } else {
                Vec::new()
            }
        };

        Ok(Some(
            source_pipelines
                .into_iter()
                .chain(target_pipelines)
                .collect(),
        ))
    }

    fn pipeline_jobs(
        &self,
        pipeline: &Pipeline,
    ) -> Result<Option<Vec<PipelineJob>>, HostingServiceError> {
        let project = self.full_project(pipeline.commit.repo.name.as_str())?;

        if project.builds_access_level == types::AccessLevel::Disabled {
            return Ok(None);
        }

        let endpoint = api::projects::pipelines::PipelineJobs::builder()
            .project(pipeline.commit.repo.name.as_str())
            .pipeline(pipeline.id)
            .build()
            .unwrap();
        let endpoint = api::paged(endpoint, api::Pagination::All);
        let jobs: Vec<types::PipelineJob> = self.query(&endpoint)?;
        Ok(Some(
            jobs.into_iter()
                .map(|job| self.job(job, pipeline.commit.repo.clone()))
                .collect(),
        ))
    }

    fn trigger_job(
        &self,
        job: &PipelineJob,
        user: Option<&str>,
    ) -> Result<(), HostingServiceError> {
        if job.archived {
            return Err(GitlabServiceError::archived_job(job.name.clone()).into());
        }

        if job.state.is_complete() {
            let endpoint = api::projects::jobs::RetryJob::builder()
                .project(job.repo.name.as_str())
                .job(job.id)
                .build()
                .unwrap();
            if let Some(user) = user {
                let endpoint = api::sudo(endpoint, user);
                let endpoint = api::ignore(endpoint);
                self.query(&endpoint)
            } else {
                let endpoint = api::ignore(endpoint);
                self.query(&endpoint)
            }
        } else {
            let endpoint = api::projects::jobs::PlayJob::builder()
                .project(job.repo.name.as_str())
                .job(job.id)
                .build()
                .unwrap();
            if let Some(user) = user {
                let endpoint = api::sudo(endpoint, user);
                let endpoint = api::ignore(endpoint);
                self.query(&endpoint)
            } else {
                let endpoint = api::ignore(endpoint);
                self.query(&endpoint)
            }
        }
    }
}

impl fmt::Debug for GitlabService {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.debug_struct("GitlabService")
            .field("user", &self.user.handle)
            .finish()
    }
}

#[cfg(test)]
mod tests {
    use std::collections::BTreeMap;

    use ghostflow::host::{CommitStatusState, PipelineState, User};
    use gitlab::api;
    use http::StatusCode;
    use thiserror::Error;

    use crate::types;

    use super::{ReferenceLevel, ReferenceTarget};

    #[derive(Debug, Error)]
    enum MyError {}

    fn mk_gitlab_status(status: StatusCode) -> api::ApiError<MyError> {
        api::ApiError::<MyError>::GitlabWithStatus {
            status,
            msg: String::new(),
        }
    }

    type TestResult = Result<(), api::ApiError<MyError>>;

    #[test]
    fn test_should_backoff() {
        let items = [
            (mk_gitlab_status(StatusCode::NOT_FOUND), true),
            (mk_gitlab_status(StatusCode::FORBIDDEN), false),
        ];

        for (i, e) in items {
            assert_eq!(super::should_backoff(&i), e);
        }
    }

    #[test]
    fn test_retry_with_backoff_first_success() {
        let mut call_count = 0;
        super::retry_with_backoff(|| -> TestResult {
            call_count += 1;
            Ok(())
        })
        .unwrap();
        assert_eq!(call_count, 1);
    }

    #[test]
    fn test_retry_with_backoff_second_success() {
        let mut call_count = 0;
        let mut did_err = false;
        super::retry_with_backoff(|| {
            call_count += 1;
            if did_err {
                Ok(())
            } else {
                did_err = true;
                Err(mk_gitlab_status(StatusCode::NOT_FOUND))
            }
        })
        .unwrap();
        assert_eq!(call_count, 2);
    }

    #[test]
    fn test_retry_with_backoff_no_success() {
        let mut call_count = 0;
        let err = super::retry_with_backoff(|| -> TestResult {
            call_count += 1;
            Err(mk_gitlab_status(StatusCode::NOT_FOUND))
        })
        .unwrap_err();
        assert_eq!(call_count, super::BACKOFF_LIMIT);
        if let api::ApiError::GitlabWithStatus {
            status,
            msg,
        } = err
        {
            assert_eq!(status, StatusCode::NOT_FOUND);
            assert_eq!(msg, "failed even after exponential backoff");
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_mr_update_re() {
        let comments = [
            // "Old" style with markdown items.
            "Added 1 commit:\n\n\
             * deadbeef0 - blah blah blah blah blah blah blab blah",
            "Added 4 commits:\n\n\
             * deadbeef1 - blah blab blah blah\n\
             * deadbeef2 - blah blab blah blah\n\
             * deadbeef3 - blah blab blah blah\n\
             * deadbeef4 - blah blah blah blah",
            "added 1 commit\n\n\
             * deadbeef5 - blah blah blah\
             \n\n[Compare with previous version](link_to_revision_diff)",
            "added 90 commits\n\n\
             * deadbeef6...deadbeef7 - 89 commits from branch `upstream:master`\n\
             * deadbeef8 - blah blah blah blah\
             \n\n[Compare with previous version](link_to_revision_diff)",
            // "New" style with explicit `<ul>` and `<li>` items.
            "added 1 commit\n\n\
             <ul>\
             <li>deadbeef7 - blah blah blah blah</li>\
             </ul>\
             \n\n[Compare with previous version](link_to_revision_diff)",
            "added 18 commits\n\n\
             <ul>\
             <li>deadbeef8...deadbeef9 - 17 commits from branch <code>upstream:master</code></li>\
             <li>deadbeef10 - Merge remote-tracking branch &#39;origin/master&#39; into this_topic</li>\
             </ul>\
             \n\n[Compare with previous version](link_to_revision_diff)",
        ];

        for comment in comments.iter() {
            assert!(super::mr_update_re().is_match(dbg!(comment)));
        }
    }

    #[test]
    fn test_ghostflow_user() {
        let expect_username = "uname";
        let expect_email = "foo@bar.invalid";
        let expect_name = "name";
        let full_user = types::FullUser {
            username: expect_username.into(),
            email: expect_email.into(),
            name: expect_name.into(),
        };

        let User {
            handle,
            name,
            email,
        } = super::ghostflow_user(full_user);
        assert_eq!(handle, expect_username);
        assert_eq!(email, expect_email);
        assert_eq!(name, expect_name);
    }

    #[test]
    fn test_ghostflow_pipeline_state() {
        let items = [
            (types::PipelineStatus::Created, PipelineState::InProgress),
            (
                types::PipelineStatus::WaitingForResource,
                PipelineState::InProgress,
            ),
            (types::PipelineStatus::Preparing, PipelineState::InProgress),
            (
                types::PipelineStatus::WaitingForCallback,
                PipelineState::InProgress,
            ),
            (types::PipelineStatus::Pending, PipelineState::InProgress),
            (types::PipelineStatus::Running, PipelineState::InProgress),
            (types::PipelineStatus::Failed, PipelineState::Failed),
            (types::PipelineStatus::Success, PipelineState::Success),
            (types::PipelineStatus::Canceling, PipelineState::Canceled),
            (types::PipelineStatus::Canceled, PipelineState::Canceled),
            (types::PipelineStatus::Skipped, PipelineState::Canceled),
            (types::PipelineStatus::Manual, PipelineState::Manual),
            (types::PipelineStatus::Scheduled, PipelineState::InProgress),
        ];

        for (gl, gf) in items {
            assert_eq!(super::ghostflow_pipeline_state(gl), gf);
        }
    }

    #[test]
    fn test_gitlab_status_state() {
        let items = [
            (
                CommitStatusState::Pending,
                api::projects::repository::commits::CommitStatusState::Pending,
            ),
            (
                CommitStatusState::Success,
                api::projects::repository::commits::CommitStatusState::Success,
            ),
            (
                CommitStatusState::Failed,
                api::projects::repository::commits::CommitStatusState::Failed,
            ),
            (
                CommitStatusState::Running,
                api::projects::repository::commits::CommitStatusState::Running,
            ),
        ];

        for (gf, gl) in items {
            assert_eq!(super::gitlab_commit_status_state(gf), gl);
        }
    }

    #[test]
    fn test_ghostflow_commit_status_state() {
        let items = [
            (types::StatusState::Created, CommitStatusState::Pending),
            (types::StatusState::Pending, CommitStatusState::Pending),
            (types::StatusState::Running, CommitStatusState::Running),
            (types::StatusState::Success, CommitStatusState::Success),
            (types::StatusState::Failed, CommitStatusState::Failed),
            (types::StatusState::Canceled, CommitStatusState::Pending),
            (types::StatusState::Skipped, CommitStatusState::Pending),
            (types::StatusState::Manual, CommitStatusState::Pending),
            (types::StatusState::Scheduled, CommitStatusState::Pending),
        ];

        for (gl, gf) in items {
            assert_eq!(super::ghostflow_commit_status_state(gl), gf);
        }
    }

    #[test]
    fn test_reference_target_issue() {
        let issue = types::Issue {
            labels: Vec::new(),
            project_id: 0,
            web_url: String::new(),
            iid: 100,
        };

        assert_eq!(types::Issue::sigil(), '#');
        assert_eq!(issue.id(), 100);
    }

    #[test]
    fn test_reference_target_merge_request() {
        let mr = types::MergeRequest {
            source_project_id: 0,
            source_branch: String::new(),
            target_branch: String::new(),
            description: None,
            sha: None,
            work_in_progress: false,
            force_remove_source_branch: None,
            author: types::Author {
                id: 0,
            },
            web_url: String::new(),
            iid: 100,
            merge_after: None,
        };

        assert_eq!(types::MergeRequest::sigil(), '!');
        assert_eq!(mr.id(), 100);
    }

    #[test]
    fn test_reference_level_default() {
        assert_eq!(ReferenceLevel::default(), ReferenceLevel::Project);
    }

    #[test]
    fn test_reference_level_between() {
        let namespaces: BTreeMap<&'static str, u64> = [("group", 100), ("other_group", 101)]
            .iter()
            .cloned()
            .collect();
        let projects: BTreeMap<&'static str, u64> = [
            ("project", 200),
            ("sibling_project", 201),
            ("other_project", 202),
        ]
        .iter()
        .cloned()
        .collect();
        let mk_project = |namespace: &str, project: &str| {
            types::Project {
                id: *projects.get(project).unwrap(),
                path_with_namespace: format!("{namespace}/{project}"),
                ssh_url_to_repo: String::new(),
                http_url_to_repo: String::new(),
                forked_from_project: None,
                namespace: types::Namespace {
                    id: *namespaces.get(namespace).unwrap(),
                    kind: types::NamespaceKind::Group,
                    path: namespace.into(),
                },
                path: project.into(),
                builds_access_level: types::AccessLevel::Disabled,
            }
        };

        let project_source = mk_project("group", "project");
        let project_target_same = mk_project("group", "project");
        let project_target_sibling = mk_project("group", "sibling_project");
        let project_target_elsewhere = mk_project("other_group", "other_project");

        let items = [
            (&project_source, ReferenceLevel::Project),
            (&project_target_same, ReferenceLevel::Project),
            (&project_target_sibling, ReferenceLevel::Namespace),
            (&project_target_elsewhere, ReferenceLevel::Site),
        ];

        for (p, rl) in items {
            assert_eq!(ReferenceLevel::between(&project_source, p), rl);
        }
    }

    #[test]
    fn test_reference_level_to() {
        let project = types::Project {
            id: 0,
            path_with_namespace: "namespace/project".into(),
            ssh_url_to_repo: String::new(),
            http_url_to_repo: String::new(),
            forked_from_project: None,
            namespace: types::Namespace {
                id: 0,
                kind: types::NamespaceKind::Group,
                path: "namespace".into(),
            },
            path: "project".into(),
            builds_access_level: types::AccessLevel::Disabled,
        };
        let issue = types::Issue {
            labels: Vec::new(),
            project_id: 0,
            web_url: String::new(),
            iid: 100,
        };
        let mr = types::MergeRequest {
            source_project_id: 0,
            source_branch: String::new(),
            target_branch: String::new(),
            description: None,
            sha: None,
            work_in_progress: false,
            force_remove_source_branch: None,
            author: types::Author {
                id: 0,
            },
            web_url: String::new(),
            iid: 200,
            merge_after: None,
        };

        let issue_items = [
            (ReferenceLevel::Project, "#100"),
            (ReferenceLevel::Namespace, "project#100"),
            (ReferenceLevel::Site, "namespace/project#100"),
        ];

        for (rl, expect) in issue_items {
            assert_eq!(rl.to(&project, &issue), expect);
        }

        let mr_items = [
            (ReferenceLevel::Project, "!200"),
            (ReferenceLevel::Namespace, "project!200"),
            (ReferenceLevel::Site, "namespace/project!200"),
        ];

        for (rl, expect) in mr_items {
            assert_eq!(rl.to(&project, &mr), expect);
        }
    }
}
