// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

#[macro_use]
extern crate lazy_static;

mod crates {
    pub extern crate chrono;
    pub extern crate ghostflow;
    pub extern crate gitlab;
    pub extern crate git_workarea;
    pub extern crate itertools;
    pub extern crate regex;
}

use crates::chrono::{DateTime, NaiveDate, Utc};
use crates::ghostflow::host::*;
use crates::git_workarea::CommitId;
use crates::gitlab;
use crates::itertools::Itertools;
use crates::regex::Regex;

use std::cmp::Ord;
use std::collections::hash_map::HashMap;
use std::fmt::{self, Debug};

/// Match messages generated by GitLab `app/services/system_note_service.rb`
/// when a MR branch is updated.
lazy_static! {
    static ref MR_UPDATE_RE: Regex =
        Regex::new("^[Aa]dded [0-9][0-9]* (new )?commits?:?\n\
                    \n\
                    (\\* [0-9a-f]+ - [^\n]*\n)*\
                    (\n\\[Compare with previous versions?\\]\\(.*\\))?\
                    $").unwrap();
}

/// Create a Gitlab commit status from a pending commit status.
fn gitlab_status_info(status: PendingCommitStatus) -> gitlab::CommitStatusInfo {
    gitlab::CommitStatusInfo {
        refname: status.refname,
        name: Some(status.name),
        target_url: None,
        description: Some(status.description),
    }
}

fn ghostflow_user(user: gitlab::UserPublic) -> User {
    User {
        id: user.id.value(),
        handle: user.username,
        name: user.name,
        email: user.email,
    }
}

fn gitlab_access_level(level: AccessLevel) -> gitlab::AccessLevel {
    match level {
        AccessLevel::Owner => gitlab::AccessLevel::Owner,
        AccessLevel::Maintainer => gitlab::AccessLevel::Master,
        AccessLevel::Developer => gitlab::AccessLevel::Developer,
        AccessLevel::Contributor => gitlab::AccessLevel::Guest,
    }
}

fn ghostflow_access_level(level: u64) -> AccessLevel {
    if level >= 50 {
        AccessLevel::Owner
    } else if level >= 40 {
        AccessLevel::Maintainer
    } else if level >= 30 {
        AccessLevel::Developer
    } else {
        AccessLevel::Contributor
    }
}

fn gitlab_state(state: CommitStatusState) -> gitlab::StatusState  {
    match state {
        CommitStatusState::Pending => gitlab::StatusState::Pending,
        CommitStatusState::Running => gitlab::StatusState::Running,
        CommitStatusState::Success => gitlab::StatusState::Success,
        CommitStatusState::Failed => gitlab::StatusState::Failed,
    }
}

fn ghostflow_state(state: gitlab::StatusState) -> CommitStatusState  {
    match state {
        gitlab::StatusState::Canceled |
        gitlab::StatusState::Pending => CommitStatusState::Pending,
        gitlab::StatusState::Running => CommitStatusState::Running,
        gitlab::StatusState::Success => CommitStatusState::Success,
        gitlab::StatusState::Failed => CommitStatusState::Failed,
    }
}

fn naivedate_to_datetime(naive_date: NaiveDate) -> DateTime<Utc> {
    DateTime::from_utc(naive_date.and_hms(0, 0, 0), Utc)
}

/// An entity on the service which may be linked to with a shorthand.
trait ReferenceTarget {
    /// The sigil to use for the target.
    fn sigil() -> char;
    /// The ID of the target.
    fn id(&self) -> u64;
}

impl ReferenceTarget for gitlab::Issue {
    fn sigil() -> char {
        '#'
    }

    fn id(&self) -> u64 {
        self.iid.value()
    }
}

impl ReferenceTarget for gitlab::MergeRequest {
    fn sigil() -> char {
        '!'
    }

    fn id(&self) -> u64 {
        self.iid.value()
    }
}

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
/// The level of reference needed to link to a target from a source.
enum ReferenceLevel {
    /// The source and target are part of the same project.
    Project,
    /// The source and target are part of the same namespace.
    Namespace,
    /// The source and target are part of the same site.
    Site,
    // If support is added, a `url` method needs to be added to the trait.
    // /// The source and target are on separate instances.
    // Full,
}

impl ReferenceLevel {
    fn between(source: &gitlab::Project, target: &gitlab::Project) -> Self {
        if source.id == target.id {
            ReferenceLevel::Project
        } else if source.namespace.id() == target.namespace.id() {
            ReferenceLevel::Namespace
        } else {
            ReferenceLevel::Site
        }
    }

    fn to<T>(self, project: &gitlab::Project, target: &T) -> String
        where T: ReferenceTarget,
    {
        match self {
            ReferenceLevel::Project => format!("{}{}", T::sigil(), target.id()),
            ReferenceLevel::Namespace => format!("{}{}{}", project.path, T::sigil(), target.id()),
            ReferenceLevel::Site => {
                format!("{}/{}{}{}",
                        project.namespace.path,
                        project.path,
                        T::sigil(),
                        target.id())
            },
        }
    }
}

impl Default for ReferenceLevel {
    fn default() -> Self {
        ReferenceLevel::Project
    }
}

/// Structure used to communicate with a Gitlab instance.
pub struct GitlabService {
    /// The Gitlab client.
    gitlab: gitlab::Gitlab,
    /// The user the service is acting as.
    user: User,
}

impl GitlabService {
    /// Create a new GitLab communication channel.
    pub fn new(gitlab: gitlab::Gitlab) -> Result<Self> {
        Ok(GitlabService {
            user: gitlab.current_user()
                .map(ghostflow_user)
                .chain_err(|| ErrorKind::Host)?,
            gitlab: gitlab,
        })
    }

    /// Create a repository from a Gitlab project.
    fn repo_from_project(&self, project: gitlab::Project) -> Result<Repo> {
        let parent_project = if let Some(ref upstream) = project.forked_from_project {
            let parent_project = self.gitlab
                .project(upstream.id)
                .chain_err(|| ErrorKind::Host)?;
            Some(Box::new(self.repo_from_project(parent_project)?))
        } else {
            None
        };

        Ok(Repo {
            name: project.path_with_namespace,
            url: project.ssh_url_to_repo,
            id: project.id.value(),
            forked_from: parent_project,
        })
    }

    /// Create a commit from a Gitlab commit.
    fn commit_from_project(&self, project: gitlab::Project, commit: &CommitId) -> Result<Commit> {
        let commit = self.gitlab
            .commit(project.id, commit.as_str())
            .chain_err(|| ErrorKind::Host)?;

        Ok(Commit {
            repo: self.repo_from_project(project)?,
            refname: None,
            id: CommitId::new(commit.id.value()),
        })
    }

    /// Create an issue from a Gitlab issue.
    fn issue_from_project(&self, project: gitlab::Project, id: u64) -> Result<Issue> {
        let issue = self.gitlab
            .issue(project.id, gitlab::IssueInternalId::new(id))
            .chain_err(|| ErrorKind::Host)?;

        self.issue(project, issue, None)
    }

    fn issue(&self, project: gitlab::Project, issue: gitlab::Issue,
             referrer: Option<&gitlab::Project>)
             -> Result<Issue> {
        let reference = referrer.map(|source| ReferenceLevel::between(source, &project))
            .unwrap_or(ReferenceLevel::Project);

        Ok(Issue {
            reference: reference.to(&project, &issue),
            repo: self.repo_from_project(project)?,
            id: issue.iid.value(),
            url: issue.web_url,
            description: issue.description.unwrap_or_else(String::new),
            labels: issue.labels,
            milestone: issue.milestone.map(|milestone| milestone.title),
            assignee: issue.assignee.map(|assignee| assignee.username),
        })
    }

    /// Create a merge request from a Gitlab merge request.
    fn merge_request_from_project(&self, project: gitlab::Project, id: u64)
                                  -> Result<MergeRequest> {
        let mr = self.gitlab
            .merge_request(project.id, gitlab::MergeRequestInternalId::new(id))
            .chain_err(|| ErrorKind::Host)?;
        let source_project = self.gitlab
            .project(mr.source_project_id)
            .chain_err(|| ErrorKind::Host)?;
        let author = self.gitlab
            .user::<gitlab::UserPublic>(mr.author.id)
            .map(ghostflow_user)
            .chain_err(|| ErrorKind::Host)?;

        let reference = ReferenceLevel::default().to(&project, &mr);

        let source_repo = self.repo_from_project(source_project)?;
        let target_repo = self.repo_from_project(project)?;

        Ok(MergeRequest {
            source_repo: source_repo.clone(),
            source_branch: mr.source_branch.clone(),
            target_repo: target_repo,
            target_branch: mr.target_branch,
            id: mr.iid.value(),
            url: mr.web_url,
            work_in_progress: mr.work_in_progress,
            description: mr.description.unwrap_or_else(String::new),
            old_commit: None,
            commit: Commit {
                repo: source_repo,
                refname: Some(mr.source_branch),
                // If `mr.sha` is `None`, the source branch has been deleted. Using an empty string
                // here will basically make anything that uses it fail since it can never be a
                // valid ID for Git or Gitlab. We let this happen because `CommitId` structures are
                // vetted at their usage rather than at construction by going through a call to
                // `git rev-parse` to validate them.
                id: CommitId::new(mr.sha.as_ref().map_or("", |sha| sha.value())),
            },
            author: author,
            reference: reference,
            remove_source_branch: mr.force_remove_source_branch.unwrap_or(false),
        })
    }

    /// Sort comments according to the ID numbers.
    fn sort_notes(&self, notes: Vec<gitlab::Note>) -> Result<Vec<Comment>> {
        Ok(notes.into_iter()
            .map(|note| {
                Ok(Comment {
                    id: note.id.value(),
                    is_system: note.system,
                    is_branch_update: note.system && MR_UPDATE_RE.is_match(&note.body),
                    created_at: note.created_at,
                    author: self.gitlab
                        .user::<gitlab::UserPublic>(note.author.id)
                        .map(ghostflow_user)
                        .chain_err(|| ErrorKind::Host)?,
                    content: note.body
                        // Remove ZERO WIDTH SPACE from the command. This can occur when
                        // copy/pasting contents from an email or rendering of a command into a
                        // comment box. Due to its invisibility, it causes confusion when the
                        // resulting `not recognized at all` text appears with the invisible
                        // character hidden in there.
                        .replace('\u{200b}', ""),
                })
            })
            .collect::<Result<Vec<_>>>()?
            .into_iter()
            .sorted_by(|a, b| a.id.cmp(&b.id)))
    }

    /// Create an award from a Gitlab award.
    fn award(&self, award: gitlab::AwardEmoji) -> Result<Award> {
        self.gitlab
            .user::<gitlab::UserPublic>(award.user.id)
            .map(move |author| {
                Award {
                    name: award.name,
                    author: ghostflow_user(author),
                }
            })
            .chain_err(|| ErrorKind::Host)
    }
}

impl HostingService for GitlabService {
    fn service_user(&self) -> &User {
        &self.user
    }

    fn add_member(&self, project: &str, user: &User, level: AccessLevel) -> Result<()> {
        let project = self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host)?;

        self.gitlab
            .add_user_to_project(project.id, gitlab::UserId::new(user.id), gitlab_access_level(level))
            .chain_err(|| ErrorKind::Host)?;

        Ok(())
    }

    fn members(&self, project: &str) -> Result<Vec<Membership>> {
        let project = self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host)?;

        let project_members = self.gitlab
            .project_members(project.id)
            .chain_err(|| ErrorKind::Host)?;
        let namespace_id = project.namespace.id();
        let group_members = if let gitlab::NamespaceId::Group(gid) = namespace_id {
            self.gitlab
                .group_members(gid)
                .chain_err(|| ErrorKind::Host)?
        } else {
            Vec::new()
        };

        project_members.into_iter()
            // Chain with the group membership list.
            .chain(group_members.into_iter())
            // Sort by the user IDs.
            .sorted_by(|a, b| Ord::cmp(&a.id.value(), &b.id.value()))
            .into_iter()
            // Group all members in the list by the ID.
            .group_by(|member| member.id)
            .into_iter()
            // Create Membership structs.
            .map(|(_, members)| {
                // Find the maximum access level from any membership.
                let max_access = members.max_by_key(|member| member.access_level)
                    .expect("expected there to be at least one member in each group");

                Ok(Membership {
                    user: self.gitlab.user::<gitlab::UserPublic>(max_access.id)
                        .map(ghostflow_user)
                        .chain_err(|| ErrorKind::Host)?,
                    access_level: ghostflow_access_level(max_access.access_level),
                    expiration: max_access.expires_at.map(naivedate_to_datetime),
                })
            })
            .collect()
    }

    fn add_hook(&self, project: &str, url: &str) -> Result<()> {
        let project = self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host)?;
        let events = gitlab::WebhookEvents::new()
            .with_issues()
            .with_merge_requests()
            .with_note()
            .with_push();

        let _ = self.gitlab
            .add_hook(project.id, url, events)
            .chain_err(|| ErrorKind::Host)?;

        Ok(())
    }

    fn user(&self, user: &str) -> Result<User> {
        Ok(self.gitlab
            .user_by_name(user)
            .map(ghostflow_user)
            .chain_err(|| ErrorKind::Host)?)
    }

    fn commit(&self, project: &str, commit: &CommitId) -> Result<Commit> {
        let project = self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host)?;
        self.commit_from_project(project, commit)
    }

    fn issue(&self, project: &str, id: u64) -> Result<Issue> {
        let project = self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host)?;
        self.issue_from_project(project, id)
    }

    fn merge_request(&self, project: &str, id: u64) -> Result<MergeRequest> {
        let project = self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host)?;
        self.merge_request_from_project(project, id)
    }

    fn repo(&self, project: &str) -> Result<Repo> {
        let project = self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host)?;
        self.repo_from_project(project)
    }

    fn user_by_id(&self, user: u64) -> Result<User> {
        Ok(self.gitlab
            .user(gitlab::UserId::new(user))
            .map(ghostflow_user)
            .chain_err(|| ErrorKind::Host)?)
    }

    fn commit_by_id(&self, project: u64, commit: &CommitId) -> Result<Commit> {
        let project = self.gitlab
            .project(gitlab::ProjectId::new(project))
            .chain_err(|| ErrorKind::Host)?;
        self.commit_from_project(project, commit)
    }

    fn issue_by_id(&self, project: u64, id: u64) -> Result<Issue> {
        let project = self.gitlab
            .project(gitlab::ProjectId::new(project))
            .chain_err(|| ErrorKind::Host)?;
        self.issue_from_project(project, id)
    }

    fn merge_request_by_id(&self, project: u64, id: u64) -> Result<MergeRequest> {
        let project = self.gitlab
            .project(gitlab::ProjectId::new(project))
            .chain_err(|| ErrorKind::Host)?;
        self.merge_request_from_project(project, id)
    }

    fn repo_by_id(&self, project: u64) -> Result<Repo> {
        let project = self.gitlab
            .project(gitlab::ProjectId::new(project))
            .chain_err(|| ErrorKind::Host)?;
        self.repo_from_project(project)
    }

    fn get_issue_comments(&self, issue: &Issue) -> Result<Vec<Comment>> {
        let notes = self.gitlab
            .issue_notes(gitlab::ProjectId::new(issue.repo.id),
                         gitlab::IssueInternalId::new(issue.id))
            .chain_err(|| ErrorKind::Host)?;

        self.sort_notes(notes)
    }

    fn post_issue_comment(&self, issue: &Issue, content: &str) -> Result<()> {
        self.gitlab
            .create_issue_note(gitlab::ProjectId::new(issue.repo.id),
                               gitlab::IssueInternalId::new(issue.id),
                               content)
            .chain_err(|| ErrorKind::Host)?;

        Ok(())
    }

    fn get_mr_comments(&self, mr: &MergeRequest) -> Result<Vec<Comment>> {
        let notes = self.gitlab
            .merge_request_notes(gitlab::ProjectId::new(mr.target_repo.id),
                                 gitlab::MergeRequestInternalId::new(mr.id))
            .chain_err(|| ErrorKind::Host)?;

        self.sort_notes(notes)
    }

    fn post_mr_comment(&self, mr: &MergeRequest, content: &str) -> Result<()> {
        self.gitlab
            .create_merge_request_note(gitlab::ProjectId::new(mr.target_repo.id),
                                       gitlab::MergeRequestInternalId::new(mr.id),
                                       content)
            .chain_err(|| ErrorKind::Host)?;

        Ok(())
    }

    fn post_commit_comment(&self, commit: &Commit, content: &str) -> Result<()> {
        self.gitlab
            .create_commit_comment(gitlab::ProjectId::new(commit.repo.id),
                                   commit.id.as_str(),
                                   content)
            .chain_err(|| ErrorKind::Host)?;

        Ok(())
    }

    fn get_commit_statuses(&self, commit: &Commit) -> Result<Vec<CommitStatus>> {
        self.gitlab
            .commit_latest_statuses(gitlab::ProjectId::new(commit.repo.id), commit.id.as_str())
            .chain_err(|| ErrorKind::Host)?
            .into_iter()
            .map(move |status| {
                Ok(self.gitlab
                    .user::<gitlab::UserPublic>(status.author.id)
                    .map(move |author| {
                        CommitStatus {
                            state: ghostflow_state(status.status),
                            author: ghostflow_user(author),
                            refname: status.ref_,
                            name: status.name,
                            description: status.description.unwrap_or_else(String::new),
                        }
                    })
                    .chain_err(|| ErrorKind::Host)?)
            })
            .collect::<Result<Vec<_>>>()
    }

    fn post_commit_status(&self, status: PendingCommitStatus) -> Result<()> {
        self.gitlab
            .create_commit_status(gitlab::ProjectId::new(status.commit.repo.id),
                                  status.commit.id.as_str(),
                                  gitlab_state(status.state),
                                  &gitlab_status_info(status))
            .chain_err(|| ErrorKind::Host)?;

        Ok(())
    }

    fn get_mr_awards(&self, mr: &MergeRequest) -> Result<Vec<Award>> {
        self.gitlab
            .merge_request_awards(gitlab::ProjectId::new(mr.target_repo.id),
                                  gitlab::MergeRequestInternalId::new(mr.id))
            .chain_err(|| ErrorKind::Host)
            .and_then(|awards| {
                awards.into_iter()
                    .map(|award| self.award(award))
                    .collect()
            })
    }

    fn get_mr_comment_awards(&self, mr: &MergeRequest, comment: &Comment) -> Result<Vec<Award>> {
        self.gitlab
            .merge_request_note_awards(gitlab::ProjectId::new(mr.target_repo.id),
                                       gitlab::MergeRequestInternalId::new(mr.id),
                                       gitlab::NoteId::new(comment.id))
            .chain_err(|| ErrorKind::Host)
            .and_then(|awards| {
                awards.into_iter()
                    .map(|award| self.award(award))
                    .collect()
            })
    }

    fn award_mr_comment(&self, mr: &MergeRequest, comment: &Comment, award: &str) -> Result<()> {
        self.gitlab
            .award_merge_request_note(gitlab::ProjectId::new(mr.target_repo.id),
                                      gitlab::MergeRequestInternalId::new(mr.id),
                                      gitlab::NoteId::new(comment.id),
                                      award)
            .chain_err(|| ErrorKind::Host)?;

        Ok(())
    }

    fn comment_award_name(&self) -> &str {
        "robot"
    }

    fn issues_closed_by_mr(&self, mr: &MergeRequest) -> Result<Vec<Issue>> {
        let target_id = gitlab::ProjectId::new(mr.target_repo.id);
        let target_project = self.gitlab
            .project(target_id)
            .chain_err(|| ErrorKind::Host)?;

        self.gitlab
            .get_issues_closed_by_merge_request(target_id, gitlab::MergeRequestInternalId::new(mr.id))
            .chain_err(|| ErrorKind::Host)
            .and_then(|issues| {
                // Cache projects to reduce hitting the service so much.
                let projects = issues.iter()
                    .map(|issue| issue.project_id.value())
                    .unique()
                    .map(|project_id| {
                        self.gitlab
                            .project(gitlab::ProjectId::new(project_id))
                            .map(|project| (project_id, project))
                            .chain_err(|| ErrorKind::Host)
                    })
                    .collect::<Result<HashMap<_, _>>>();

                // Link to each issue.
                projects.and_then(|projects| {
                    issues.into_iter()
                        .map(|issue| {
                            let project = projects.get(&issue.project_id.value())
                                .expect("the fetched project ID should exist");
                            self.issue(project.clone(), issue, Some(&target_project))
                        })
                        .collect()
                })
            })
    }

    fn add_issue_labels(&self, issue: &Issue, labels: &[&str]) -> Result<()> {
        let all_labels = issue.labels
            .iter()
            .map(|label| label.as_str())
            .chain(labels.iter().cloned())
            .unique();
        self.gitlab
            .set_issue_labels(gitlab::ProjectId::new(issue.repo.id),
                              gitlab::IssueInternalId::new(issue.id),
                              all_labels)
            .chain_err(|| ErrorKind::Host)?;

        Ok(())
    }
}

impl Debug for GitlabService {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.debug_struct("GitlabService")
            .field("user", &self.user.handle)
            .finish()
    }
}
