/*
 * Copyright 2003 Sandia Corporation.
 * Under the terms of Contract DE-AC04-94AL85000, there is a non-exclusive
 * license for use of this work by or on behalf of the
 * U.S. Government. Redistribution and use in source and binary forms, with
 * or without modification, are permitted provided that this Notice and any
 * statement of authorship are reproduced on all copies.
 */
#include <vtkObjectFactory.h>

#include <vtkStreamingTessellator.h>
#include <vtkSubdivisionAlgorithm.h>
#undef ALLOW_TET_INTERIOR_PT
#undef UGLY_ASPECT_RATIO_HACK
#undef DBG_MIDPTS

// Define RUPRECHT to use Ruprecht & Mller's tetrahedral subdivisions. (Good idea)
#define RUPRECHT

#ifdef RUPRECHT
#  undef ALLOW_TET_INTERIOR_PT
   // Include table for converting edge codes to a
   // configuration ID and permutation.
#  include "vtkStreamingTessellatorConfigurations.cxx"
#else
   // Include autogenerated list of tetrahedra that are generated
   // when various edges of an initial tetrahedron are subdivided.
#  include "vtkAdaptiveTetrahedraCases.cxx"
#endif // RUPRECHT

#include <algorithm>

vtkCxxRevisionMacro(vtkStreamingTessellator,"1.2");
vtkStandardNewMacro(vtkStreamingTessellator);

void vtkStreamingTessellator::PrintSelf( ostream& os, vtkIndent indent )
{
  this->Superclass::PrintSelf( os, indent );
  os << indent << "PointDimension:       "
     << this->PointDimension[1] << " " << this->PointDimension[2] << " " << this->PointDimension[3] << endl;
  os << indent << "EmbeddingDimension:   "
     << this->EmbeddingDimension[1] << " " << this->EmbeddingDimension[2] << " " << this->EmbeddingDimension[3] << endl;
  os << indent << "PrivateData:          " << this->PrivateData << endl;
  os << indent << "ConstPrivateData:     " << this->ConstPrivateData << endl;
  os << indent << "SubdivisionAlgorithm: " << this->Algorithm << endl;
  os << indent << "EdgeCallback:         " << this->Callback1 << endl;
  os << indent << "TriangleCallback:     " << this->Callback2 << endl;
  os << indent << "TetrahedronCallback:  " << this->Callback3 << endl;
}

vtkStreamingTessellator::vtkStreamingTessellator()
{
  this->PrivateData = 0;
  this->ConstPrivateData = 0;
  this->Algorithm = 0;
  this->Callback1 = 0;
  this->Callback2 = 0;
  this->MaximumNumberOfSubdivisions = 8;
  for ( int i=0; i<4; ++i )
    {
    this->EmbeddingDimension[i] = i;
    this->PointDimension[i] = i+3; // By default, FieldSize = 0
    }
}

vtkStreamingTessellator::~vtkStreamingTessellator()
{
  if ( this->Algorithm )
    this->Algorithm->UnRegister( this );
}

void vtkStreamingTessellator::SetEmbeddingDimension( int k, int d )
{
  if ( d > 8 )
    {
    vtkErrorMacro( "Embedding dimension may not be > 8. (You asked for " << d << "). Truncating to 8." );
    d = 8;
    }

  if ( k == 0 || k < -1 || k >= 4 )
    {
    vtkWarningMacro( "Invalid argument k=" << k );
    return;
    }

  if ( k < 0 )
    {
    for ( k=0; k<4; k++ )
      if ( this->EmbeddingDimension[k] != d )
        {
        this->PointDimension[k] += d - this->EmbeddingDimension[k] ;
        this->EmbeddingDimension[k] = d;
        this->Modified();
        }
    return;
    }
  if ( this->EmbeddingDimension[k] != d )
    {
    this->PointDimension[k] += d - this->EmbeddingDimension[k] ;
    this->EmbeddingDimension[k] = d;
    this->Modified();
    }
}

void vtkStreamingTessellator::SetFieldSize( int k, int s )
{
  if ( s > vtkStreamingTessellator::MaxFieldSize )
    {
    vtkErrorMacro( "Embedding dimension may not be > " << MaxFieldSize << ". (You asked for " << s << "). Truncating to " << MaxFieldSize );
    s = vtkStreamingTessellator::MaxFieldSize;
    }

  if ( k == 0 || k < -1 || k >= 4 )
    {
    vtkWarningMacro( "Invalid argument k=" << k );
    return;
    }

  if ( k < 0 )
    {
    // Use field size for all facet types (point, line, triangle, tet, ...)
    for ( k=0; k<4; k++ )
      if ( this->PointDimension[k] != s + this->EmbeddingDimension[k] + 3 )
        {
        this->PointDimension[k] = s + this->EmbeddingDimension[k] + 3;
        this->Modified();
        }
    return;
    }

  if ( this->PointDimension[k] != s + this->EmbeddingDimension[k] + 3 )
    {
    this->PointDimension[k] = s + this->EmbeddingDimension[k] + 3;
    this->Modified();
    }
}

void vtkStreamingTessellator::SetMaximumNumberOfSubdivisions( int num_subdiv_in )
{
  if ( this->MaximumNumberOfSubdivisions == num_subdiv_in )
    return;

  if ( num_subdiv_in < 0 )
    {
    vtkErrorMacro( "MaximumNumberOfSubdivisions must be 0 or greater (you requested " << num_subdiv_in << ")" );
    return;
    }

  this->MaximumNumberOfSubdivisions = num_subdiv_in;
  this->Modified();
}

void vtkStreamingTessellator::SetTriangleCallback( TriangleProcessorFunction f )
{
  this->Callback2 = f;
}

vtkStreamingTessellator::TriangleProcessorFunction vtkStreamingTessellator::GetTriangleCallback() const
{
  return this->Callback2;
}

void vtkStreamingTessellator::SetTetrahedronCallback( TetrahedronProcessorFunction f )
{
  this->Callback3 = f;
}

vtkStreamingTessellator::TetrahedronProcessorFunction vtkStreamingTessellator::GetTetrahedronCallback() const
{
  return this->Callback3;
}

void vtkStreamingTessellator::SetEdgeCallback( EdgeProcessorFunction f )
{
  this->Callback1 = f;
}

vtkStreamingTessellator::EdgeProcessorFunction vtkStreamingTessellator::GetEdgeCallback() const
{
  return this->Callback1;
}

void vtkStreamingTessellator::SetPrivateData( void* Private )
{
  this->PrivateData = Private;
}

void* vtkStreamingTessellator::GetPrivateData() const
{
  return this->PrivateData;
}

void vtkStreamingTessellator::SetConstPrivateData( const void* ConstPrivate )
{
  this->ConstPrivateData = ConstPrivate;
}

const void* vtkStreamingTessellator::GetConstPrivateData() const
{
  return this->ConstPrivateData;
}

void vtkStreamingTessellator::SetSubdivisionAlgorithm( vtkSubdivisionAlgorithm* a )
{
  if ( a != this->Algorithm )
    {
    if ( this->Algorithm )
      this->Algorithm->UnRegister( this );

    this->Algorithm = a;
    this->Modified();

    if ( this->Algorithm )
      this->Algorithm->Register( this );
    }
}

vtkSubdivisionAlgorithm* vtkStreamingTessellator::GetSubdivisionAlgorithm()
{
  return this->Algorithm;
}

const vtkSubdivisionAlgorithm* vtkStreamingTessellator::GetSubdivisionAlgorithm() const
{
  return this->Algorithm;
}


// Returns true if || a0a1 || < || b0b1 ||
// We use this to test which triangulation has the best
// aspect ratio when there are 2 to choose from.
bool compareHopfCrossStringDist( const double* a0, const double* a1, const double* b0, const double* b1 )
{
  double SqMagA = 0.;
  double SqMagB = 0.;
  for (int i=0; i<3; i++)
    {
    double tmp;
    SqMagA += (tmp=(a0[i]-a1[i]))*tmp;
    SqMagB += (tmp=(b0[i]-b1[i]))*tmp;
    }
  return SqMagA < SqMagB;
}

void vtkStreamingTessellator::AdaptivelySample1Facet( double* v0, double* v1, int maxDepth ) const
{
  int edgeCode = 0;

  double midpt0[11+vtkStreamingTessellator::MaxFieldSize];
#if 1
  // make valgrind happy
  vtkstd::fill(midpt0,midpt0+this->PointDimension[1],0.);
#endif // 1

  if ( maxDepth-- > 0 )
    {
      for ( int i=0; i<this->PointDimension[1]; i++ )
        midpt0[i] = (v0[i] + v1[i])/2.;

      if ( this->Algorithm->EvaluateEdge( v0, midpt0, v1, 3+this->EmbeddingDimension[1] ) )
        edgeCode += 1;
    }

  switch (edgeCode) {
    // No edges to subdivide
    case 0:
      Callback1( v0, v1, this->Algorithm, this->PrivateData, this->ConstPrivateData );
      break ;

      // One edge to subdivide
    case 1:
      this->AdaptivelySample1Facet( v0, midpt0, maxDepth );
      this->AdaptivelySample1Facet( midpt0, v1, maxDepth );
      break;
  }
}

void vtkStreamingTessellator::AdaptivelySample2Facet( double* v0, double* v1, double* v2, int maxDepth, int move ) const
{
  int edgeCode = 0;

  double midpt0[11+vtkStreamingTessellator::MaxFieldSize];
  double midpt1[11+vtkStreamingTessellator::MaxFieldSize];
  double midpt2[11+vtkStreamingTessellator::MaxFieldSize];

#if 1
  // Make valgrind happy
  vtkstd::fill(midpt0,midpt0+this->PointDimension[2],0.);
  vtkstd::fill(midpt1,midpt1+this->PointDimension[2],0.);
  vtkstd::fill(midpt2,midpt2+this->PointDimension[2],0.);
#endif // 0

  if ( maxDepth-- > 0 )
    {

    for ( int i=0; i<this->PointDimension[2]; i++ )
      {
      midpt0[i] = (v0[i] + v1[i])/2.;
      midpt1[i] = (v1[i] + v2[i])/2.;
      midpt2[i] = (v2[i] + v0[i])/2.;
      }

    if ( (move & 1) && Algorithm->EvaluateEdge( v0, midpt0, v1, 3+this->EmbeddingDimension[2] ) )
      edgeCode += 1;
    if ( (move & 2) && Algorithm->EvaluateEdge( v1, midpt1, v2, 3+this->EmbeddingDimension[2] ) )
      edgeCode += 2;
    if ( (move & 4) && Algorithm->EvaluateEdge( v2, midpt2, v0, 3+this->EmbeddingDimension[2] ) )
      edgeCode += 4;
#ifdef UGLY_ASPECT_RATIO_HACK
    double dist0=0.;
    double dist1=0.;
    double dist2=0.;
    double tmp;
    for ( int j=0; j<3; ++j )
      {
      tmp = v0[j] - v1[j];
      dist0 += tmp*tmp;
      tmp = v1[j] - v2[j];
      dist1 += tmp*tmp;
      tmp = v2[j] - v0[j];
      dist2 += tmp*tmp;
      }

    if ( edgeCode & 1 ) dist0 /= 2.;
    if ( edgeCode & 2 ) dist1 /= 2.;
    if ( edgeCode & 4 ) dist2 /= 2.;

#define MAR2 2.25
    if ( (!(edgeCode & 1)) && (move&1) && ((dist0/dist1 > MAR2) || (dist0/dist2 > MAR2)) )
      {
      edgeCode += 1;
      move &= 6;
      }
    if ( (!(edgeCode & 2)) && (move&2) && ((dist1/dist0 > MAR2) || (dist1/dist2 > MAR2)) )
      {
      edgeCode += 2;
      move &= 5;
      }
    if ( (!(edgeCode & 4)) && (move&4) && ((dist2/dist1 > MAR2) || (dist2/dist0 > MAR2)) )
      {
      edgeCode += 4;
      move &= 3;
      }
#endif // UGLY_ASPECT_RATIO_HACK
    }

#ifdef DBG_MIDPTS
  if ( maxDepth == 0 )
    {
    fprintf( stderr, "midpoint of v%d (%g %g %g/%g %g %g)-v%d (%g %g %g/%g %g %g) = (%g %g %g/%g %g %g)\n",
      0, v0[0], v0[1], v0[2], v0[3], v0[4], v0[5],
      1, v1[0], v1[1], v1[2], v1[3], v1[4], v1[5],
         midpt0[0], midpt0[1], midpt0[2], midpt0[3], midpt0[4], midpt0[5]
    );

    fprintf( stderr, "midpoint of v%d (%g %g %g/%g %g %g)-v%d (%g %g %g/%g %g %g) = (%g %g %g/%g %g %g)\n",
      1, v1[0], v1[1], v1[2], v1[3], v1[4], v1[5],
      2, v2[0], v2[1], v2[2], v2[3], v2[4], v2[5],
         midpt1[0], midpt1[1], midpt1[2], midpt1[3], midpt1[4], midpt1[5]
    );

    fprintf( stderr, "midpoint of v%d (%g %g %g/%g %g %g)-v%d (%g %g %g/%g %g %g) = (%g %g %g/%g %g %g)\n\n",
      2, v2[0], v2[1], v2[2], v2[3], v2[4], v2[5],
      0, v0[0], v0[1], v0[2], v0[3], v0[4], v0[5],
         midpt2[0], midpt2[1], midpt2[2], midpt2[3], midpt2[4], midpt2[5]
    );
    }
#endif // DBG_MIDPTS

  switch (edgeCode)
    {
    // No edges to subdivide
  case 0:
    Callback2( v0, v1, v2, this->Algorithm, this->PrivateData, this->ConstPrivateData );
    break ;

    // One edge to subdivide
  case 1:
    this->AdaptivelySample2Facet( v0, midpt0, v2, maxDepth, move | 2 );
    this->AdaptivelySample2Facet( midpt0, v1, v2, maxDepth, move | 4 );
    break;
  case 2:
    this->AdaptivelySample2Facet( v0, v1, midpt1, maxDepth, move | 4 );
    this->AdaptivelySample2Facet( v0, midpt1, v2, maxDepth, move | 1 );
    break;
  case 4:
    this->AdaptivelySample2Facet( v0, v1, midpt2, maxDepth, move | 2 );
    this->AdaptivelySample2Facet( midpt2, v1, v2, maxDepth, move | 1 );
    break;

    // Two edges to subdivide
  case 3:
    this->AdaptivelySample2Facet( midpt0, v1, midpt1, maxDepth, move | 4 );
    if ( compareHopfCrossStringDist( v2, midpt0, v0, midpt1 ) )
      {
      this->AdaptivelySample2Facet( midpt0, midpt1,   v2  , maxDepth, move | 5 );
      this->AdaptivelySample2Facet(   v0,   midpt0,   v2  , maxDepth, move | 2 );
      }
    else
      {
      this->AdaptivelySample2Facet(   v0  , midpt0, midpt1, maxDepth, move | 6 );
      this->AdaptivelySample2Facet(   v0,   midpt1,   v2  , maxDepth, move | 1 );
      }
    break;
  case 5:
    this->AdaptivelySample2Facet( v0, midpt0, midpt2, maxDepth, move | 2 );
    if ( compareHopfCrossStringDist( v2, midpt0, v1, midpt2 ) )
      {
      this->AdaptivelySample2Facet( midpt0,   v1,     v2  , maxDepth, move | 4 );
      this->AdaptivelySample2Facet( midpt2, midpt0,   v2  , maxDepth, move | 3 );
      }
    else
      {
      this->AdaptivelySample2Facet( midpt0,   v1,   midpt2, maxDepth, move | 6 );
      this->AdaptivelySample2Facet( midpt2,   v1,     v2,   maxDepth, move | 1 );
      }
    break;
  case 6:
    this->AdaptivelySample2Facet( midpt2, midpt1, v2, maxDepth, move | 1 );
    if ( compareHopfCrossStringDist( v0, midpt1, v1, midpt2 ) )
      {
      this->AdaptivelySample2Facet(   v0,   midpt1, midpt2, maxDepth, move | 3 );
      this->AdaptivelySample2Facet(   v0,     v1,   midpt1, maxDepth, move | 4 );
      }
    else
      {
      this->AdaptivelySample2Facet(   v0,     v1,   midpt2, maxDepth, move | 2 );
      this->AdaptivelySample2Facet( midpt2,   v1,   midpt1, maxDepth, move | 5 );
      }
    break;

    // Three edges to subdivide
  case 7:
    this->AdaptivelySample2Facet( midpt0, midpt1, midpt2, maxDepth, 7 );
    this->AdaptivelySample2Facet(   v0  , midpt0, midpt2, maxDepth, move | 2 );
    this->AdaptivelySample2Facet( midpt0,   v1  , midpt1, maxDepth, move | 4 );
    this->AdaptivelySample2Facet( midpt2, midpt1,   v2  , maxDepth, move | 1 );
    break;
    }
}

void vtkStreamingTessellator::AdaptivelySample3Facet( double* v0, double* v1, double* v2, double* v3, int maxDepth, int move ) const
{
  int edgeCode = 0;

  double midpt0[11+vtkStreamingTessellator::MaxFieldSize];
  double midpt1[11+vtkStreamingTessellator::MaxFieldSize];
  double midpt2[11+vtkStreamingTessellator::MaxFieldSize];
  double midpt3[11+vtkStreamingTessellator::MaxFieldSize];
  double midpt4[11+vtkStreamingTessellator::MaxFieldSize];
  double midpt5[11+vtkStreamingTessellator::MaxFieldSize];
#ifdef ALLOW_TET_INTERIOR_PT
  double midpt6[11+vtkStreamingTessellator::MaxFieldSize];
#endif // ALLOW_TET_INTERIOR_PT

#if 1
  // Make valgrind happy
  vtkstd::fill(midpt0,midpt0+this->PointDimension[3],0.);
  vtkstd::fill(midpt1,midpt1+this->PointDimension[3],0.);
  vtkstd::fill(midpt2,midpt2+this->PointDimension[3],0.);
  vtkstd::fill(midpt3,midpt3+this->PointDimension[3],0.);
  vtkstd::fill(midpt4,midpt4+this->PointDimension[3],0.);
  vtkstd::fill(midpt5,midpt5+this->PointDimension[3],0.);
#ifdef ALLOW_TET_INTERIOR_PT
  vtkstd::fill(midpt6,midpt6+this->PointDimension[3],0.);
#endif // ALLOW_TET_INTERIOR_PT
#endif // 0

  double edgeLength2[6];
  if ( maxDepth-- > 0 )
    {
    for ( int i=0; i<this->PointDimension[3]; i++ )
      {
      midpt0[i] = (v0[i] + v1[i])/2.;
      midpt1[i] = (v1[i] + v2[i])/2.;
      midpt2[i] = (v2[i] + v0[i])/2.;
      midpt3[i] = (v0[i] + v3[i])/2.;
      midpt4[i] = (v1[i] + v3[i])/2.;
      midpt5[i] = (v2[i] + v3[i])/2.;
#ifdef ALLOW_TET_INTERIOR_PT
      midpt6[i] = (v1[i]+v2[i]-2*v0[i])/4. + (v3[i]-v0[i])/3. + v0[i];
#endif // ALLOW_TET_INTERIOR_PT
      }

    if ( (move &  1) && Algorithm->EvaluateEdge( v0, midpt0, v1, 3+this->EmbeddingDimension[3] ) )
      edgeCode |=  1;
    if ( (move &  2) && Algorithm->EvaluateEdge( v1, midpt1, v2, 3+this->EmbeddingDimension[3] ) )
      edgeCode |=  2;
    if ( (move &  4) && Algorithm->EvaluateEdge( v2, midpt2, v0, 3+this->EmbeddingDimension[3] ) )
      edgeCode |=  4;

    if ( (move &  8) && Algorithm->EvaluateEdge( v0, midpt3, v3, 3+this->EmbeddingDimension[3] ) )
      edgeCode |=  8;
    if ( (move & 16) && Algorithm->EvaluateEdge( v1, midpt4, v3, 3+this->EmbeddingDimension[3] ) )
      edgeCode |= 16;
    if ( (move & 32) && Algorithm->EvaluateEdge( v2, midpt5, v3, 3+this->EmbeddingDimension[3] ) )
      edgeCode |= 32;

    edgeLength2[0] = edgeLength2[1] = edgeLength2[2] = edgeLength2[3]
      = edgeLength2[4] = edgeLength2[5] = 0;
    for ( int c=0; c<3; ++c )
      {
      double tmp;
      tmp = v1[c] - v0[c];
      edgeLength2[0] += tmp*tmp;
      tmp = v2[c] - v1[c];
      edgeLength2[1] += tmp*tmp;
      tmp = v2[c] - v0[c];
      edgeLength2[2] += tmp*tmp;
      tmp = v3[c] - v0[c];
      edgeLength2[3] += tmp*tmp;
      tmp = v3[c] - v1[c];
      edgeLength2[4] += tmp*tmp;
      tmp = v3[c] - v2[c];
      edgeLength2[5] += tmp*tmp;
      }

#ifdef RUPRECHT
    // Get a full ordering of edge lengths?
#endif // RUPRECHT

#ifdef ALLOW_TET_INTERIOR_PT
    // Find the longest/shortest edges
    double shortest = edgeLength2[0];
    double longest = edgeLength2[0];
    for ( int e=1; e<6; e++ )
      {
      if ( edgeLength2[e] < shortest )
        shortest = edgeLength2[e];
      if ( edgeLength2[e] > longest )
        longest = edgeLength2[e];
      }
    // Divide at center if aspect ratio is > 4:1 (remember these are squares of edge lengths):
    if ( shortest / longest < 1./16. )
      edgeCode |= 64;
#endif // ALLOW_TET_INTERIOR_PT
#if 0
    // Dunno if we need to have the subdivision algorithm evaluate it, but I suspect
    // it wouldn't be a bad idea.
    if ( (move & 64) && Algorithm->EvaluatePoint( midpt6 ) )
      edgeCode |= 64;
#endif // 0
    }

  if ( edgeCode == 0 )
    {
    // No edges to subdivide
    Callback3( v0, v1, v2, v3, this->Algorithm, this->PrivateData, this->ConstPrivateData );
    }
  else
    {
    // Do the subdivision
#ifdef ALLOW_TET_INTERIOR_PT
    double* vertices[11] =
    {
      v0, v1, v2, v3,     midpt0, midpt1, midpt2, midpt3, midpt4, midpt5, midpt6
    };
#else // ALLOW_TET_INTERIOR_PT
    double* vertices[10] =
    {
      v0, v1, v2, v3,     midpt0, midpt1, midpt2, midpt3, midpt4, midpt5
    };
#endif // ALLOW_TET_INTERIOR_PT

#ifdef RUPRECHT

    // Generate tetrahedra that are compatible except when edge
    // lengths are equal on indeterminately subdivided faces.
    double* permuted[10];
    double permlen[6]; // permuted edge lengths
    int C = edgecodes_to_case_plus_permutation[ edgeCode ][0];
    int P = edgecodes_to_case_plus_permutation[ edgeCode ][1];
    int i, j;

    // 1. Permute the tetrahedron into our canonical configuration
    for ( i=0; i<4; ++i )
      {
      permuted[i] = vertices[ permutations_from_index[P][i] ];
      }
    for ( i=4; i<10; ++i )
      {
      // permute the edge lengths, too
      permuted[i] = vertices[ permutations_from_index[P][i] ];
      permlen[i-4]  = edgeLength2[ permutations_from_index[P][i] - 4 ];
      }
    int permove = move; // FIXME: Need to permute the bits.

    // 2. Generate tetrahedra based on the configuration.
    //    Note that case 0 is handled above (edgeCode == 0).
    switch (C)
      {
    case 1:  // Case 1
      this->AdaptivelySample3Facet( permuted[0], permuted[4], permuted[2], permuted[3], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[4], permuted[1], permuted[2], permuted[3], maxDepth, permove );
      break;
    case 2:  // Case 2a
      this->AdaptivelySample3Facet( permuted[3], permuted[4], permuted[5], permuted[1], maxDepth, permove );
      if ( permlen[1] > permlen[0] )
        {
        this->AdaptivelySample3Facet( permuted[0], permuted[4], permuted[5], permuted[3], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[0], permuted[5], permuted[2], permuted[3], maxDepth, permove );
        }
      else // if ( permlen[0] > permlen[1] )
        {
        this->AdaptivelySample3Facet( permuted[0], permuted[4], permuted[2], permuted[3], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[4], permuted[5], permuted[2], permuted[3], maxDepth, permove );
        } // else { insert code to handle permlen[0] == permlen[1] }
      break;
    case 3:  // Case 2b
      this->AdaptivelySample3Facet( permuted[0], permuted[4], permuted[9], permuted[3], maxDepth, permove |  2 );
      this->AdaptivelySample3Facet( permuted[4], permuted[1], permuted[9], permuted[3], maxDepth, permove |  4 );
      this->AdaptivelySample3Facet( permuted[0], permuted[4], permuted[2], permuted[9], maxDepth, permove | 16 );
      this->AdaptivelySample3Facet( permuted[4], permuted[1], permuted[2], permuted[9], maxDepth, permove |  8 );
      break;
    case 4:  // Case 3a
      this->AdaptivelySample3Facet( permuted[4], permuted[7], permuted[6], permuted[0], maxDepth, permove );
      // Sort edges 0, 2, 3 by length, ignoring the possibility that the lengths may be equivalent
      if ( permlen[0] > permlen[2] ) {
        if ( permlen[0] > permlen[3] ) {
          this->AdaptivelySample3Facet( permuted[1], permuted[3], permuted[2], permuted[4], maxDepth, permove );
          if ( permlen[2] > permlen[3] ) {
            // 0, 2, 3
            this->AdaptivelySample3Facet( permuted[4], permuted[6], permuted[3], permuted[2], maxDepth, permove );
            this->AdaptivelySample3Facet( permuted[4], permuted[6], permuted[7], permuted[3], maxDepth, permove );
          } else {
            // 0, 3, 2
            this->AdaptivelySample3Facet( permuted[7], permuted[4], permuted[2], permuted[3], maxDepth, permove );
            this->AdaptivelySample3Facet( permuted[7], permuted[4], permuted[6], permuted[2], maxDepth, permove );
          }
        } else {
          // 3, 0, 2
          this->AdaptivelySample3Facet( permuted[1], permuted[3], permuted[2], permuted[7], maxDepth, permove );
          this->AdaptivelySample3Facet( permuted[7], permuted[4], permuted[2], permuted[1], maxDepth, permove );
          this->AdaptivelySample3Facet( permuted[7], permuted[4], permuted[6], permuted[2], maxDepth, permove );
        }
      } else {
        if ( permlen[2] > permlen[3] ) {
          this->AdaptivelySample3Facet( permuted[1], permuted[3], permuted[2], permuted[6], maxDepth, permove );
          if ( permlen[0] > permlen[3] ) {
            // 2, 0, 3
            this->AdaptivelySample3Facet( permuted[4], permuted[6], permuted[3], permuted[1], maxDepth, permove );
            this->AdaptivelySample3Facet( permuted[4], permuted[6], permuted[7], permuted[3], maxDepth, permove );
          } else {
            // 2, 3, 0
            this->AdaptivelySample3Facet( permuted[6], permuted[7], permuted[1], permuted[3], maxDepth, permove );
            this->AdaptivelySample3Facet( permuted[6], permuted[7], permuted[4], permuted[1], maxDepth, permove );
          }
        } else {
          // 3, 2, 0
          this->AdaptivelySample3Facet( permuted[1], permuted[3], permuted[2], permuted[7], maxDepth, permove );
          this->AdaptivelySample3Facet( permuted[6], permuted[7], permuted[1], permuted[2], maxDepth, permove );
          this->AdaptivelySample3Facet( permuted[6], permuted[7], permuted[4], permuted[1], maxDepth, permove );
        }
      }
      break;
    case 5:  // Case 3b
      this->AdaptivelySample3Facet( permuted[0], permuted[7], permuted[4], permuted[2], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[4], permuted[7], permuted[8], permuted[2], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[4], permuted[8], permuted[1], permuted[2], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[7], permuted[3], permuted[8], permuted[2], maxDepth, permove );
      break;
    case 6:  // Case 3c
      this->AdaptivelySample3Facet( permuted[3], permuted[5], permuted[7], permuted[2], maxDepth, permove | 2 );
      if ( permlen[0] > permlen[1] )
        {
        this->AdaptivelySample3Facet( permuted[4], permuted[2], permuted[7], permuted[5], maxDepth, permove | 32 );
        this->AdaptivelySample3Facet( permuted[4], permuted[2], permuted[0], permuted[7], maxDepth, permove );
        }
      else
        {
        this->AdaptivelySample3Facet( permuted[0], permuted[5], permuted[2], permuted[7], maxDepth, permove | 16 );
        this->AdaptivelySample3Facet( permuted[0], permuted[5], permuted[7], permuted[4], maxDepth, permove | 2 );
        }
      if ( permlen[0] > permlen[3] )
        {
        this->AdaptivelySample3Facet( permuted[4], permuted[3], permuted[1], permuted[5], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[4], permuted[3], permuted[5], permuted[7], maxDepth, permove | 32 );
        }
      else
        {
        this->AdaptivelySample3Facet( permuted[7], permuted[1], permuted[4], permuted[5], maxDepth, permove | 8 );
        this->AdaptivelySample3Facet( permuted[7], permuted[1], permuted[5], permuted[3], maxDepth, permove | 4 );
        }
      break;
    case 7:  // Case 3d (originally lumped into 3c)
      this->AdaptivelySample3Facet( permuted[2], permuted[3], permuted[6], permuted[8], maxDepth, permove | 32 );
      if ( permlen[0] > permlen[4] )
        {
        this->AdaptivelySample3Facet( permuted[4], permuted[3], permuted[6], permuted[0], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[4], permuted[3], permuted[8], permuted[6], maxDepth, permove | 32 );
        }
      else
        {
        this->AdaptivelySample3Facet( permuted[8], permuted[0], permuted[6], permuted[4], maxDepth, permove | 4 );
        this->AdaptivelySample3Facet( permuted[8], permuted[0], permuted[3], permuted[6], maxDepth, permove | 8 );
        }
      if ( permlen[0] > permlen[2] )
        {
        this->AdaptivelySample3Facet( permuted[4], permuted[2], permuted[8], permuted[1], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[4], permuted[2], permuted[6], permuted[8], maxDepth, permove | 32 );
        }
      else
        {
        this->AdaptivelySample3Facet( permuted[6], permuted[1], permuted[8], permuted[4], maxDepth, permove | 4 );
        this->AdaptivelySample3Facet( permuted[6], permuted[1], permuted[2], permuted[8], maxDepth, permove | 8 );
        }
      break;
    case 8:  // Case 4a
      this->AdaptivelySample3Facet( permuted[7], permuted[8], permuted[9], permuted[3], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[7], permuted[9], permuted[8], permuted[6], maxDepth, permove | 32 );
      if ( permlen[3] > permlen[4] )
        {
        this->AdaptivelySample3Facet( permuted[7], permuted[1], permuted[6], permuted[8], maxDepth, permove | 32 );
        this->AdaptivelySample3Facet( permuted[7], permuted[1], permuted[0], permuted[6], maxDepth, permove );
        }
      else
        {
        this->AdaptivelySample3Facet( permuted[8], permuted[0], permuted[6], permuted[1], maxDepth, permove | 4 );
        this->AdaptivelySample3Facet( permuted[8], permuted[0], permuted[7], permuted[6], maxDepth, permove | 8 );
        }
      if ( permlen[4] > permlen[5] )
        {
        this->AdaptivelySample3Facet( permuted[8], permuted[2], permuted[6], permuted[9], maxDepth, permove | 4 );
        this->AdaptivelySample3Facet( permuted[8], permuted[2], permuted[1], permuted[6], maxDepth, permove | 8);
        }
      else
        {
        this->AdaptivelySample3Facet( permuted[9], permuted[1], permuted[6], permuted[2], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[9], permuted[1], permuted[8], permuted[6], maxDepth, permove | 32 );
        }
      break;
    case 9:  // Case 4b
      // Construct a bit vector describing which edges are longer than others
      i = ( permlen[1] > permlen[4] ? 1 : 0 ) | ( permlen[2] > permlen[3] ? 2 : 0 ) |
          ( permlen[1] > permlen[2] ? 4 : 0 ) | ( permlen[4] > permlen[3] ? 8 : 0 );
      // Use our bit vector to figure out whether the quadrilateral 5-6-7-8 should
      // be split on the 5-7 diagonal (case j=0) or the 6-8 diagonal (case j=1).
      if ( (i&1) ^ (i&2) )
        j = i & 1 ? 0 : 1;
      else // we know (i&4) ^ (i&8) is true from the results of the paper
        j = i & 4 ? 0 : 1;
      {
      // prisms to tessellate
      static int prisms[3][6] = {
        {  3,  2,  5,  8,  7,  6 },
        {  0,  1,  5,  6,  7,  8 },
        { -1, -1, -1, -1, -1, -1 } };
      int* p= &prisms[0][0];
      while ( p[0] != -1 )
        {
        int c = i & 3;
        switch (c)
          {
        case 0: // base diagonal determined by j, side diagonals are p[1]-p[4] & p[1]-p[3]
          if ( j )
            { // p[3]-p[5] diagonal
            this->AdaptivelySample3Facet( permuted[p[1]], permuted[p[3]], permuted[p[5]], permuted[p[2]], maxDepth, permove |  2 );
            this->AdaptivelySample3Facet( permuted[p[1]], permuted[p[3]], permuted[p[4]], permuted[p[5]], maxDepth, permove | 16 );
            this->AdaptivelySample3Facet( permuted[p[1]], permuted[p[3]], permuted[p[0]], permuted[p[4]], maxDepth, permove );
            }
          else
            { // p[2]-p[4] diagonal
            this->AdaptivelySample3Facet( permuted[p[1]], permuted[p[4]], permuted[p[5]], permuted[p[2]], maxDepth, permove | 16 );
            this->AdaptivelySample3Facet( permuted[p[1]], permuted[p[4]], permuted[p[2]], permuted[p[3]], maxDepth, permove |  2 );
            this->AdaptivelySample3Facet( permuted[p[1]], permuted[p[4]], permuted[p[3]], permuted[p[0]], maxDepth, permove );
            }
          break;
        case 1: // base diagonal p[2]-p[4], sides are p[0]-p[2] and p[1]-p[4]
          this->AdaptivelySample3Facet( permuted[p[2]], permuted[p[4]], permuted[p[1]], permuted[p[5]], maxDepth, permove | 1 );
          this->AdaptivelySample3Facet( permuted[p[2]], permuted[p[4]], permuted[p[0]], permuted[p[1]], maxDepth, permove | 1 );
          this->AdaptivelySample3Facet( permuted[p[2]], permuted[p[4]], permuted[p[3]], permuted[p[0]], maxDepth, permove | 1 );
          break;
        case 2: // base diagonal p[3]-p[5], sides are p[1]-p[3] and p[5]-p[0]
          this->AdaptivelySample3Facet( permuted[p[3]], permuted[p[5]], permuted[p[1]], permuted[p[2]], maxDepth, permove | 1 );
          this->AdaptivelySample3Facet( permuted[p[3]], permuted[p[5]], permuted[p[0]], permuted[p[1]], maxDepth, permove | 1 );
          this->AdaptivelySample3Facet( permuted[p[3]], permuted[p[5]], permuted[p[4]], permuted[p[0]], maxDepth, permove | 1 );
          break;
        case 3: // base diagonal determined by j, side diagonals are p[0]-p[2] & p[0]-p[5]
          if ( j )
            { // p[3]-p[5] diagonal
            this->AdaptivelySample3Facet( permuted[p[0]], permuted[p[5]], permuted[p[3]], permuted[p[4]], maxDepth, permove |  2 );
            this->AdaptivelySample3Facet( permuted[p[0]], permuted[p[5]], permuted[p[2]], permuted[p[3]], maxDepth, permove | 16 );
            this->AdaptivelySample3Facet( permuted[p[0]], permuted[p[5]], permuted[p[1]], permuted[p[2]], maxDepth, permove );
            }
          else
            { // p[2]-p[4] diagonal
            this->AdaptivelySample3Facet( permuted[p[0]], permuted[p[2]], permuted[p[5]], permuted[p[1]], maxDepth, permove );
            this->AdaptivelySample3Facet( permuted[p[0]], permuted[p[2]], permuted[p[4]], permuted[p[5]], maxDepth, permove |  2 );
            this->AdaptivelySample3Facet( permuted[p[0]], permuted[p[2]], permuted[p[3]], permuted[p[4]], maxDepth, permove | 16 );
            }
          break;
          }
        i >>= 2;
        p += 6;
        }
      }
      break;
    case 10: // Case 5
      // Lop off vertices 2 & 3, leaving a prism (0-1-5-6-7-8) and a pyramid (5-6-7-8-9)
      this->AdaptivelySample3Facet( permuted[5], permuted[6], permuted[9], permuted[2], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[7], permuted[8], permuted[9], permuted[3], maxDepth, permove );
      // Take care of the prism. Record the configuration in i so that the pyramid and prism are compatible
      if ( permlen[1] > permlen[2] )
        {
        if ( permlen[3] > permlen[4] )
          {
          this->AdaptivelySample3Facet( permuted[5], permuted[7], permuted[1], permuted[8], maxDepth, permove | 1 );
          this->AdaptivelySample3Facet( permuted[5], permuted[7], permuted[0], permuted[1], maxDepth, permove | 1 );
          this->AdaptivelySample3Facet( permuted[5], permuted[7], permuted[6], permuted[0], maxDepth, permove | 1 );
          i = 1;
          }
        else
          {
          // don't really care what i is, but aspect ratio better if:
          //    ||5-6|| + ||6-7|| > ||8-5|| + ||5-6|| => i=1
          //    ||5-6|| + ||6-7|| < ||8-5|| + ||5-6|| => i=0
          // If we choose i this way, we must generate compatible tets here, too
          this->AdaptivelySample3Facet( permuted[0], permuted[5], permuted[6], permuted[7], maxDepth, permove | 16 );
          this->AdaptivelySample3Facet( permuted[0], permuted[5], permuted[7], permuted[8], maxDepth, permove |  2 );
          this->AdaptivelySample3Facet( permuted[0], permuted[5], permuted[8], permuted[1], maxDepth, permove );
          i = 1;
          }
        }
      else
        // assume permlen[2] > permlen[1]
        {
        if ( permlen[3] > permlen[4] )
          {
          // don't really care what i is, but aspect ratio better if:
          //    ||5-6|| + ||6-7|| > ||8-5|| + ||5-6|| => i=1
          //    ||5-6|| + ||6-7|| < ||8-5|| + ||5-6|| => i=0
          // If we choose i this way, we must generate compatible tets here, too
          this->AdaptivelySample3Facet( permuted[1], permuted[6], permuted[8], permuted[5], maxDepth, permove |  2 );
          this->AdaptivelySample3Facet( permuted[1], permuted[6], permuted[7], permuted[8], maxDepth, permove | 16 );
          this->AdaptivelySample3Facet( permuted[1], permuted[6], permuted[0], permuted[7], maxDepth, permove );
          i = 0;
          }
        else
          {
          this->AdaptivelySample3Facet( permuted[6], permuted[8], permuted[1], permuted[5], maxDepth, permove | 1 );
          this->AdaptivelySample3Facet( permuted[6], permuted[8], permuted[0], permuted[1], maxDepth, permove | 1 );
          this->AdaptivelySample3Facet( permuted[6], permuted[8], permuted[7], permuted[0], maxDepth, permove | 1 );
          i = 0;
          }
        }
      // Take care of the pyramid based on the prism's tessellation
      if ( i )
        {
        this->AdaptivelySample3Facet( permuted[5], permuted[6], permuted[7], permuted[9], maxDepth, permove | 4 );
        this->AdaptivelySample3Facet( permuted[5], permuted[7], permuted[8], permuted[9], maxDepth, permove | 1 );
        }
      else
        {
        this->AdaptivelySample3Facet( permuted[6], permuted[7], permuted[8], permuted[9], maxDepth, permove | 4 );
        this->AdaptivelySample3Facet( permuted[6], permuted[8], permuted[5], permuted[9], maxDepth, permove | 1 );
        }
      break;
    case 11: // Case 6
      // Test interior diagonals to find decomposition w/ best aspect ratio
      for ( i=0; i<3; ++i )
        permlen[i] = 0.;
      for ( i=0; i<3; ++i )
        {
        double d;
        d = permuted[4][i] - permuted[9][i];
        permlen[0] += d*d;
        d = permuted[5][i] - permuted[7][i];
        permlen[1] += d*d;
        d = permuted[6][i] - permuted[8][i];
        permlen[2] += d*d;
        }
      i = permlen[0] < permlen[1] ? ( permlen[0] < permlen[2] ? 0 : 2 )
                                  : ( permlen[1] < permlen[2] ? 1 : 2 );
      // i = 0 -> interior diagonal 4-9 is shortest
      //   = 1 -> interior diagonal 5-7 is shortest
      //   = 2 -> interior diagonal 6-8 is shortest

      // First, output the 4 tets that "lop off" the corner vertices
      this->AdaptivelySample3Facet( permuted[4], permuted[7], permuted[6], permuted[0], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[4], permuted[5], permuted[8], permuted[1], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[5], permuted[6], permuted[9], permuted[2], maxDepth, permove );
      this->AdaptivelySample3Facet( permuted[7], permuted[8], permuted[9], permuted[3], maxDepth, permove );

      // Now use our shortest diagonal to break the remaining octahedron into 4 tets
      switch (i)
        {
      case 0:
        this->AdaptivelySample3Facet( permuted[4], permuted[9], permuted[7], permuted[8], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[4], permuted[9], permuted[6], permuted[7], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[4], permuted[9], permuted[5], permuted[6], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[4], permuted[9], permuted[8], permuted[5], maxDepth, permove );
        break;
      case 1:
        this->AdaptivelySample3Facet( permuted[5], permuted[7], permuted[8], permuted[9], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[5], permuted[7], permuted[4], permuted[8], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[5], permuted[7], permuted[6], permuted[4], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[5], permuted[7], permuted[9], permuted[6], maxDepth, permove );
        break;
      case 2:
        this->AdaptivelySample3Facet( permuted[6], permuted[8], permuted[9], permuted[7], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[6], permuted[8], permuted[7], permuted[4], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[6], permuted[8], permuted[4], permuted[5], maxDepth, permove );
        this->AdaptivelySample3Facet( permuted[6], permuted[8], permuted[5], permuted[9], maxDepth, permove );
        break;
        }
      break;
      }

#else // RUPRECHT

    // Use the tetrahedra cases with no concern for compatibility
    int* conn = new_tetrahedra[ new_tetrahedra_offsets[ edgeCode ] ];
    for ( int tet = 0; tet < new_tetrahedra_lengths[ edgeCode ]; ++tet )
      {
      int new_move = move;
#ifdef UGLY_ASPECT_RATIO_HACK
      // We only allow recursive interpolated midpoints on edges of
      // this tetrahedron that go toward vertex 10 (the midpoint of
      // this tetrahedron) because all others are on faces that may
      // be coincident with other (unrefined) tetrahedra.
      if ( conn[0] == 10 )
        new_move = move | (1+4+8+64);
      else if ( conn[1] == 10 )
        new_move = move | (1+2+16+64);
      else if ( conn[2] == 10 )
        new_move = move | (2+4+32+64);
      else if ( conn[3] == 10 )
        new_move = move | (8+16+32+64);
#endif // UGLY_ASPECT_RATIO_HACK

      this->AdaptivelySample3Facet( vertices[ conn[0] ], vertices[ conn[1] ], vertices[ conn[2] ], vertices[ conn[3] ], maxDepth, new_move );
      conn += 4;
      }
#endif // RUPRECHT
    }
}


