/*=========================================================================

   Library: iMSTK

   Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
   & Imaging in Medicine, Rensselaer Polytechnic Institute.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0.txt

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

=========================================================================*/

#pragma once

#include "imstkDeviceClient.h"
#include "imstkLoopThreadObject.h"

#include <vrpn_Configure.h>
#include <vrpn_Analog.h>

namespace imstk
{
///
/// \class VRPNDeviceClient
/// \brief Subclass of DeviceClient using VRPN
///
class VRPNArduinoDeviceClient : public DeviceClient, public LoopThreadObject
{
public:

    ///
    /// \brief Constructor
    ///
    VRPNArduinoDeviceClient(const std::string& deviceName, const std::string& ip) :
        DeviceClient(deviceName, ip),
        LoopThreadObject(deviceName + "@" + ip),
        m_ypr(Vec3d::Zero()),
        m_accel(Vec3d::Zero()),
        m_roll(0) {}

    ///
    /// \brief Destructor
    ///
    virtual ~VRPNArduinoDeviceClient() override = default;

    ///
    /// \brief Get YPR
    ///
    Vec3d& getYPR() { return m_ypr; }

    ///
    /// \brief Get acceleration
    ///
    Vec3d& getAcceleration() { return m_accel; }

    ///
    /// \brief Get the roll
    ///
    float getRoll() { return m_roll; }

protected:
    ///
    /// \brief Initialize device client module
    ///
    void initThread() override;

    ///
    /// \brief Run the device client
    ///
    void updateThread() override;

    ///
    /// \brief Clean the device client
    ///
    void stopThread() override;

private:

    ///
    /// \brief VRPN call back for position and orientation data
    /// \param userData Pointer to this to allow updating
    /// internal data
    /// \param b VRPN callback structure containing new position and
    /// orientation data
    ///
    static void VRPN_CALLBACK analogChangeHandler(void* userData, const _vrpn_ANALOGCB a);

    std::shared_ptr<vrpn_Analog_Remote> m_vrpnAnalog;           //!< VRPN position/orientation interface
    Vec3d m_ypr;
    Vec3d m_accel;
    float m_roll;
};
}
