// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate clap;
use self::clap::{App, AppSettings, Arg};

extern crate either;
use self::either::Either;

extern crate ghostflow;
use self::ghostflow::actions::check;
use self::ghostflow::actions::merge::MergeActionResult;
use self::ghostflow::actions::test;
use self::ghostflow::host::{CheckStatus, MergeRequest};
use self::ghostflow::utils::TrailerRef;

extern crate git_workarea;
use self::git_workarea::{CommitId, GitContext};

extern crate itertools;
use self::itertools::Itertools;

extern crate json_job_dispatch;
use self::json_job_dispatch::HandlerResult;
use self::json_job_dispatch::Result as JobResult;

extern crate serde_json;
use self::serde_json::{Map, Value};

use config::{Branch, Host, CheckAction, MergeAction, Project, ReformatAction, StageAction,
             StageUpdatePolicy, TestAction, TestBackend};
use handlers::common::data::{MergeRequestInfo, MergeRequestNoteInfo};
use handlers::common::handlers::commands::Commands;
use handlers::common::handlers::error::Error;
use handlers::common::handlers::utils;

use std::iter::FromIterator;

/// Handle an update to a merge request.
pub fn handle_merge_request_update(data: &Value, host: &Host, mr: MergeRequestInfo)
                                   -> JobResult<HandlerResult> {
    let (project, branch) = try_action!(utils::get_branch(host,
                                                          &mr.merge_request.target_repo.name,
                                                          &mr.merge_request.target_branch));

    let mut comment_notes = vec![];

    if let Some(test_action) = branch.test() {
        handle_test_on_update(data, test_action, &mr);
    }

    if mr.was_merged && mr.merge_request.remove_source_branch {
        remove_source_branch(&project.context, &mr.merge_request);
    }

    if !mr.is_open {
        if let Some(stage_action) = branch.stage() {
            handle_stage_on_close(host, stage_action, &mr);
        }

        return Ok(HandlerResult::Reject(format!("the merge request for the branch {} has been \
                                                 closed",
                                                mr.merge_request.source_branch)));
    }

    // TODO: If the MR has been edited to change the target branch, we may need to remove it from
    // other branches, but we don't have the old branch name :/ .

    if !mr.is_source_branch_deleted() {
        let fetch_res = host.service.fetch_mr(&project.context, &mr.merge_request);
        if let Err(err) = fetch_res {
            error!(target: "ghostflow-director/handler",
                   "failed to fetch from the {} repository: {:?}",
                   mr.merge_request.source_repo.url,
                   err);

            comment_notes.push(format!("Failed to fetch from the repository: {}.", err));
        }

        let reserve_res = project.context
            .reserve_ref(&format!("mr/{}", mr.merge_request.id),
                         &mr.merge_request.commit.id);
        if let Err(err) = reserve_res {
            error!(target: "ghostflow-director/handler",
                   "failed to reserve ref {} for merge request {}: {:?}",
                   mr.merge_request.commit.id,
                   mr.merge_request.url,
                   err);

            comment_notes.push(format!("Failed to reserve ref {} for the merge request: `{}`.",
                                       mr.merge_request.commit.id,
                                       err));
        }
    }

    let is_ok = if mr.is_source_branch_deleted() {
        // A merge request with a deleted source branch is never ok.
        false
    } else {
        match mr.check_status(host.service.as_ref()) {
            Ok(status) =>  {
                if status.is_checked() {
                    status.is_ok()
                } else {
                    match check_mr(project, branch, &mr.merge_request) {
                        Ok(b) => b,
                        Err(err) => {
                            error!(target: "ghostflow-director/handler",
                                   "failed to run checks for merge request {}: {:?}",
                                   mr.merge_request.url,
                                   err);

                            comment_notes.push(format!("Failed to run the checks: `{}`.", err));

                            false
                        },
                    }
                }
            },
            Err(err) => {
                error!(target: "ghostflow-director/handler",
                       "failed to determine the check status for merge request {}: {:?}",
                       mr.merge_request.url,
                       err);

                comment_notes.push(format!("Failed to determine the check status: `{}`.", err));

                false
            },
        }
    };
    let is_wip = mr.merge_request.work_in_progress;

    if let Some(stage_action) = branch.stage() {
        handle_stage_on_update(is_ok, is_wip, stage_action, &mr, &mut comment_notes);
    }

    Ok(utils::handle_result(comment_notes,
                            false,
                            |comment| host.service.post_mr_comment(&mr.merge_request, comment)))
}

fn remove_source_branch(ctx: &GitContext, mr: &MergeRequest) {
    info!(target: "ghostflow-director/handler",
          "removing the source branch for {}",
          mr.url);

    let push_res = ctx.git()
        .arg("push")
        .arg("--atomic")
        .arg("--porcelain")
        .arg(format!("--force-with-lease=refs/heads/{}:{}",
                     mr.source_branch,
                     mr.commit.id))
        .arg(&mr.source_repo.url)
        .arg(format!(":refs/heads/{}", mr.source_branch))
        .output();
    match push_res {
        Ok(push) => {
            if !push.status.success() {
                warn!(target: "ghostflow-director/handler",
                      "failed to remove the source branch of {} from the remote server: {}",
                      mr.url,
                      String::from_utf8_lossy(&push.stderr));
            }
        },
        Err(err) => {
            error!(target: "ghostflow-director/handler",
                   "failed to construct push command: {:?}",
                   err);
        },
    }
}

/// Handle the test action when a merge request is updated.
fn handle_test_on_update(data: &Value, action: &TestAction, mr: &MergeRequestInfo) {
    match *action.test() {
        TestBackend::Jobs(ref test_jobs) => {
            let job_data = utils::test_job("mr_update", data);
            if let Err(err) = test_jobs.test_update(job_data) {
                error!(target: "ghostflow-director/handler",
                        "failed to drop a job file for an update to {}: {:?}",
                        mr.merge_request.url,
                        err);
            }
        },
        TestBackend::Refs(ref test_refs) => {
            if !mr.is_open {
                if let Err(err) = test_refs.untest_mr(&mr.merge_request) {
                    error!(target: "ghostflow-director/handler",
                            "failed to remove the test ref for an update to {}: {:?}",
                            mr.merge_request.url,
                            err);
                }
            }

            // TODO: Anything else here? Forcefully untest the MR?
        },
    }
}

/// Handle removal of a merge request from the stage when it closes.
fn handle_stage_on_close(host: &Host, action: &StageAction, mr: &MergeRequestInfo) {
    let res = action.stage()
        .unstage_update_merge_request(&mr.merge_request, "because it has been closed");

    if let Err(err) = res {
        error!(target: "ghostflow-director/handler",
                "failed to unstage a closed merge request {}: {:?}",
                mr.merge_request.url,
                err);

        let notes = vec![
            format!("Error occurred when unstaging a closed merge request ({}): `{}`.",
                    host.maintainers.join(" "),
                    err),
        ];
        utils::handle_result(notes,
                             false,
                             |comment| host.service.post_mr_comment(&mr.merge_request, comment));
    }
}

/// Handle the stage policy when a merge request is updated.
fn handle_stage_on_update(is_ok: bool, is_wip: bool, action: &StageAction, mr: &MergeRequestInfo,
                          notes: &mut Vec<String>) {
    let mut stage = action.stage();

    let (policy, reason) = if !is_ok {
        (StageUpdatePolicy::Unstage, "since it is failing its checks")
    } else if is_wip {
        (StageUpdatePolicy::Unstage, "since it is marked as work-in-progress")
    } else if let Some(staged_topic) =
        stage.stager().find_topic_by_id(mr.merge_request.id) {
        if staged_topic.topic.commit == mr.merge_request.commit.id {
            (StageUpdatePolicy::Ignore, "it is already on the stage as-is")
        } else {
            (action.policy, "as per policy")
        }
    } else {
        (StageUpdatePolicy::Ignore, "it is not on the stage currently")
    };

    let (what, res) = match policy {
        StageUpdatePolicy::Ignore => ("ignore", Ok(())),
        StageUpdatePolicy::Restage => {
            ("restage",
             stage.stage_merge_request_named(&mr.merge_request,
                                             mr.topic_name(),
                                             &mr.merge_request.author.identity(),
                                             &mr.date))
        },
        StageUpdatePolicy::Unstage => {
            ("unstage", stage.unstage_update_merge_request(&mr.merge_request, reason))
        },
    };

    if let Err(err) = res {
        error!(target: "ghostflow-director/handler",
                "failed to {} merge request {}: {:?}",
                what,
                mr.merge_request.url,
                err);

        notes.push(format!("Failed to {}: `{}`.", what, err));
    }
}

/// Check a merge request against the target branch's checks.
fn check_mr(project: &Project, branch: &Branch, mr: &MergeRequest) -> check::Result<bool> {
    project.check_mr(branch, mr)
        .map(|status| if let check::CheckStatus::Pass = status {
            true
        } else {
            false
        })
}

/// Handle a note on a merge request.
pub fn handle_merge_request_note(data: &Value, host: &Host, mr_note: MergeRequestNoteInfo)
                                 -> JobResult<HandlerResult> {
    let mr = &mr_note.merge_request;
    let note = &mr_note.note;

    if !mr.is_open {
        return Ok(HandlerResult::Reject(format!("the merge request for the branch {} has been \
                                                 closed",
                                                mr.merge_request.source_branch)));
    }

    let already_handled = match host.service.get_mr_comment_awards(&mr.merge_request, note) {
        Ok(awards) => {
            awards.into_iter()
                .any(|award| {
                    award.name == "robot" && award.author.id == host.service.service_user().id
                })
        },
        Err(err) => {
            error!(target: "ghostflow-director/handler",
                   "failed to get the awards for a note at {}: {:?}",
                   mr.merge_request.url,
                   err);

            false
        },
    };

    if already_handled {
        return Ok(HandlerResult::Reject("this note has already been handled".to_string()));
    }

    let (project, branch) = try_action!(utils::get_branch(host,
                                                          &mr.merge_request.target_repo.name,
                                                          &mr.merge_request.target_branch));

    let mut commands = Commands::new(branch);

    let trailers = TrailerRef::extract(&mr_note.note.content);

    if trailers.is_empty() {
        return Ok(HandlerResult::Reject("no trailers present".to_string()));
    }

    let access = project.access_level(&note.author);
    let is_submitter = mr.merge_request.author.id == note.author.id;
    commands.add_from_trailers(access, is_submitter, &trailers);

    if commands.is_empty() {
        return Ok(HandlerResult::Reject("no commands present".to_string()));
    }

    let mut comment_notes = vec![];

    let fetch_res = host.service.fetch_mr(&project.context, &mr.merge_request);
    if let Err(err) = fetch_res {
        return Ok(HandlerResult::Fail(Error::chain(err,
                                                   format!("failed to fetch source for {}",
                                                           mr.merge_request.url))));
    }

    let errors = commands.error_messages(&note.author);
    let defer = if errors.is_empty() {
        handle_commands(data,
                        host,
                        &commands,
                        project,
                        branch,
                        &mr_note,
                        &mut comment_notes)
    } else {
        comment_notes.extend(errors);

        false
    };

    if !defer {
        // Award the command comment to indicate that we handled it.
        if let Err(err) = host.service.award_mr_comment(&mr.merge_request, note, "robot") {
            error!(target: "ghostflow-director/handler",
                   "failed to mark the comment on {} as seen: {:?}",
                   mr.merge_request.url,
                   err);

            comment_notes.push(format!("Failed to mark the comment as seen: `{}`.", err));
        }
    }

    Ok(utils::handle_result(comment_notes,
                            defer,
                            |comment| host.service.post_mr_comment(&mr.merge_request, comment)))
}

struct CommandData<'a> {
    data: &'a Value,
    host: &'a Host,
    project: &'a Project,
    branch: &'a Branch,
    mr_note: &'a MergeRequestNoteInfo,
    notes: &'a mut Vec<String>,
}

enum CommandResult {
    Continue,
    Defer,
    Stop,
}

macro_rules! try_command {
    ( $action:expr ) => {
        match $action {
            Either::Right(res) => res,
            Either::Left(command_result) => return command_result,
        }
    }
}

impl<'a> CommandData<'a> {
    fn check_status(&mut self) -> Either<CommandResult, CheckStatus> {
        let mr = &self.mr_note.merge_request;
        match mr.check_status(self.host.service.as_ref()) {
            Ok(status) => Either::Right(status),
            Err(err) => {
                error!(target: "ghostflow-director/handler",
                       "failed to determine the check status of {}: {:?}",
                       mr.merge_request.url,
                       err);

                self.notes.push(format!("Failed to determine the check status: `{}`.", err));

                Either::Left(CommandResult::Stop)
            },
        }
    }
}

macro_rules! handle_command_requests {
    ( $defer:ident, $stop:ident, $commands:ident, $branch:ident, $command_data:expr,
      $( $action:ident -> $action_member:ident => $action_impl:ident, )* ) => {
        $( if !$stop && $commands.$action.requested() {
            $branch.$action_member()
                .as_ref()
                .map(|action| {
                    let res = $action_impl($command_data,
                                           action,
                                           $commands.$action.arguments());

                    match res {
                        CommandResult::Continue => (),
                        CommandResult::Defer => $defer = true,
                        CommandResult::Stop => $stop = false,
                    }
                });
        } )*
    };
}

/// Handle commands in a merge request note.
fn handle_commands(data: &Value, host: &Host, commands: &Commands, project: &Project,
                   branch: &Branch, mr_note: &MergeRequestNoteInfo, notes: &mut Vec<String>)
                   -> bool {
    let mut command_data = CommandData {
        data: data,
        host: host,
        project: project,
        branch: branch,
        mr_note: mr_note,
        notes: notes,
    };

    let mut stop = false;
    let mut defer = false;

    handle_command_requests!(defer, stop, commands, branch, &mut command_data,
        check -> check => handle_check_command,
        stage -> stage => handle_stage_command,
        unstage -> stage => handle_unstage_command,
        test -> test => handle_test_command,
        merge -> merge => handle_merge_command,
        reformat -> reformat => handle_reformat_command,
    );

    defer
}

fn command_app(name: &'static str) -> App<'static, 'static> {
    App::new(name)
        .setting(AppSettings::ColorNever)
        .setting(AppSettings::DisableVersion)
        .setting(AppSettings::NoBinaryName)
}

/// Handle the `Do: check` command.
fn handle_check_command(data: &mut CommandData, _: &CheckAction, arguments: &[String])
                        -> CommandResult {
    let mr = &data.mr_note.merge_request;

    if arguments.is_empty() {
        match check_mr(data.project, data.branch, &mr.merge_request) {
            Ok(true) => (),
            Ok(false) => return CommandResult::Stop,
            Err(err) => {
                error!(target: "ghostflow-director/handler",
                       "failed to run checks on {}: {:?}",
                       mr.merge_request.url,
                       err);

                data.notes.push(format!("failed to run checks: `{}`", err));
            },
        }
    } else {
        data.notes.push(format!("Unrecognized `check` arguments: `{}`.",
                                arguments.iter().join("`, `")));
    }

    CommandResult::Continue
}

/// Handle the `Do: stage` command.
fn handle_stage_command(data: &mut CommandData, action: &StageAction, arguments: &[String])
                        -> CommandResult {
    let mr = &data.mr_note.merge_request;
    let note = &data.mr_note.note;
    let mut stage = action.stage();

    let check_status = try_command!(data.check_status());
    if !check_status.is_checked() {
        data.notes.push("Refusing to stage; topic is missing the checks.".to_string())
    } else if !check_status.is_ok() {
        data.notes.push("Refusing to stage; topic is failing the checks.".to_string())
    } else if arguments.is_empty() {
        let res = stage.stage_merge_request_named(&mr.merge_request,
                                                  mr.topic_name(),
                                                  &note.author.identity(),
                                                  &note.created_at);

        if let Err(err) = res {
            error!(target: "ghostflow-director/handler",
                   "failed during the stage action on {}: {:?}",
                   mr.merge_request.url,
                   err);

            data.notes.push(format!("Error occurred during stage action ({}): `{}`",
                                    data.host.maintainers.join(" "),
                                    err));
        }
    } else {
        data.notes.push(format!("Unrecognized `stage` arguments: `{}`.",
                                arguments.iter().join("`, `")));
    }

    CommandResult::Continue
}

/// Handle the `Do: unstage` command.
fn handle_unstage_command(data: &mut CommandData, action: &StageAction, arguments: &[String])
                          -> CommandResult {
    let mr = &data.mr_note.merge_request;
    let mut stage = action.stage();

    if arguments.is_empty() {
        let res = stage.unstage_merge_request(&mr.merge_request);

        if let Err(err) = res {
            error!(target: "ghostflow-director/handler",
                   "failed during the unstage action on {}: {:?}",
                   mr.merge_request.url,
                   err);

            data.notes.push(format!("Error occurred during unstage action ({}): `{}`",
                                    data.host.maintainers.join(" "),
                                    err));
        }
    } else {
        data.notes.push(format!("Unrecognized `unstage` arguments: `{}`.",
                                arguments.iter().join("`, `")));
    }

    CommandResult::Continue
}

/// Handle the `Do: test` command.
fn handle_test_command(data: &mut CommandData, action: &TestAction, arguments: &[String])
                       -> CommandResult {
    let mr = &data.mr_note.merge_request;
    let test = action.test();

    let (arguments, res): (Vec<String>, Option<test::Result<_>>) = match *test {
        TestBackend::Jobs(ref test_jobs) => {
            let args = Value::Array(arguments.iter()
                .map(|arg| Value::String(arg.clone()))
                .collect());
            let job_data = utils::test_job("test_action",
                                           &Value::Object(Map::from_iter(vec![
                ("arguments".to_string(), args),
                ("merge_request".to_string(), data.data.clone())
            ])));
            (vec![],
             Some(test_jobs.test_mr(&mr.merge_request, job_data)
                 .map_err(Into::into)))
        },
        TestBackend::Refs(ref test_refs) => {
            let mut stop = false;

            let arguments = arguments.iter()
                .filter_map(|arg| {
                    if arg == "--stop" {
                        stop = true;
                        None
                    } else {
                        Some(arg.to_string())
                    }
                })
                .collect::<Vec<_>>();

            let res = if arguments.is_empty() {
                let res = if stop {
                    test_refs.untest_mr(&mr.merge_request)
                } else {
                    test_refs.test_mr(&mr.merge_request)
                };

                Some(res.map_err(Into::into))
            } else {
                None
            };

            (arguments, res)
        },
    };

    if arguments.is_empty() != res.is_some() {
        error!(target: "ghostflow-director/handler",
               "a test action occurred even though some arguments were unparsed");
    }

    if arguments.is_empty() {
        if let Some(Err(err)) = res {
            data.notes.push(format!("Error occurred during test action ({}): `{:?}`",
                                    data.host.maintainers.join(" "),
                                    err));
        }
    } else {
        data.notes.push(format!("Unrecognized `test` arguments: `{}`.",
                                arguments.iter().join("`, `")));
    }

    CommandResult::Continue
}

/// Handle the `Do: merge` command.
fn handle_merge_command(data: &mut CommandData, action: &MergeAction, arguments: &[String])
                        -> CommandResult {
    let mr = &data.mr_note.merge_request;
    let note = &data.mr_note.note;

    let check_status = try_command!(data.check_status());
    if check_status.is_ok() {
        let matches = command_app("merge-action")
            .arg(Arg::with_name("TOPIC")
                .short("t")
                .long("topic")
                .takes_value(true))
            .get_matches_from_safe(arguments);

        match matches {
            Ok(matches) => {
                let merge = action.merge();

                let topic_name = matches.value_of("TOPIC")
                    .unwrap_or_else(|| mr.topic_name())
                    .to_string();

                if data.project.branch_names.contains(&topic_name) {
                    data.notes
                        .push(format!("The name of the topic ({}) is the same as a branch in the \
                                       project. Use the `--topic` option to merge with a different \
                                       name.",
                                      topic_name));
                    return CommandResult::Continue;
                }

                let res = merge.merge_mr_named(&mr.merge_request,
                                               topic_name,
                                               &note.author.identity(),
                                               &note.created_at);

                match res {
                    Ok(MergeActionResult::Success) |
                    Ok(MergeActionResult::Failed) => (),
                    // Defer the action if merging failed.
                    Ok(MergeActionResult::PushFailed) => {
                        data.notes.push("Failed to push the resulting merge".to_string());
                        return CommandResult::Defer;
                    },
                    Err(err) => {
                        error!(target: "ghostflow-director/handler",
                               "failed during the merge action on {}: {:?}",
                               mr.merge_request.url,
                               err);

                        data.notes.push(format!("Error occurred during merge action ({}): \
                                                 `{}`",
                                                data.host.maintainers.join(" "),
                                                err));
                    },
                }
            },
            Err(err) => {
                data.notes.push(format!("Unrecognized `merge` arguments: `{}`.",
                                        err.message.lines().next().unwrap()));
            },
        }
    } else if check_status.is_checked() {
        data.notes.push("Refusing to merge; topic is failing the checks.".to_string())
    } else {
        data.notes.push("Refusing to merge; topic is missing the checks.".to_string())
    }

    CommandResult::Continue
}

/// Handle the `Do: reformat` command.
fn handle_reformat_command(data: &mut CommandData, action: &ReformatAction, arguments: &[String])
                           -> CommandResult {
    let mr = &data.mr_note.merge_request;
    let reformat = action.reformat();

    if arguments.is_empty() {
        let res = reformat.reformat_mr(&CommitId::new(&data.branch.name), &mr.merge_request);

        if let Err(err) = res {
            error!(target: "ghostflow-director/handler",
                   "failed during the reformat action on {}: {:?}",
                   mr.merge_request.url,
                   err);

            data.notes.push(format!("Error occurred during reformat action ({}): `{}`",
                                    data.host.maintainers.join(" "),
                                    err));
        }
    } else {
        data.notes.push(format!("Unrecognized `reformat` arguments: `{}`.",
                                arguments.iter().join("`, `")));
    }

    CommandResult::Continue
}
