// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Command parsing and checking.
//!
//! Comments may contain commands for the robot to perform. This handles parsing out those commands
//! from a set of trailers.

extern crate ghostflow;
use self::ghostflow::host::User;
use self::ghostflow::utils::TrailerRef;

use config::Branch;

/// The various state command requests may be in.
pub enum CommandState {
    /// The command was requested and is allowed.
    Requested(Vec<String>),
    /// The command is not available for the branch.
    Unavailable,
    /// The command was not requested.
    Unrequested,
    /// The command was requested, but the user lacked permissions.
    Disallowed,
}

impl CommandState {
    /// Returns the arguments to the command.
    pub fn arguments(&self) -> &[String] {
        match *self {
            CommandState::Requested(ref args) => args,
            CommandState::Unavailable |
            CommandState::Unrequested |
            CommandState::Disallowed => &[],
        }
    }

    /// Returns `true` if the command was present.
    pub fn present(&self) -> bool {
        match *self {
            CommandState::Requested(_) |
            CommandState::Unavailable |
            CommandState::Disallowed => true,
            CommandState::Unrequested => false,
        }
    }

    /// Returns `true` if the command was requested, but unavailable.
    pub fn unavailable(&self) -> bool {
        match *self {
            CommandState::Unavailable => true,
            CommandState::Requested(_) |
            CommandState::Unrequested |
            CommandState::Disallowed => false,
        }
    }

    /// Returns `true` if the command was requested and allowed.
    pub fn requested(&self) -> bool {
        match *self {
            CommandState::Requested(_) => true,
            CommandState::Unavailable |
            CommandState::Unrequested |
            CommandState::Disallowed => false,
        }
    }

    /// Returns `true` if the command was requested but disallowed.
    pub fn disallowed(&self) -> bool {
        match *self {
            CommandState::Disallowed => true,
            CommandState::Requested(_) |
            CommandState::Unavailable |
            CommandState::Unrequested => false,
        }
    }
}

/// Structure for the state of requested commands.
pub struct Commands<'a> {
    /// The branch the commands apply to.
    branch: &'a Branch,

    /// The `stage` command.
    pub stage: CommandState,
    /// The `unstage` command.
    pub unstage: CommandState,
    /// The `merge` command.
    pub merge: CommandState,
    /// The `check` command.
    pub check: CommandState,
    /// The `test` command.
    pub test: CommandState,
    /// The `reformat` command.
    pub reformat: CommandState,

    /// List of unrecognized commands.
    pub unrecognized: Vec<String>,
}

macro_rules! check_action {
    // Actions with permission checks.
    ( $args:expr, $access:expr, $is_submitter:expr,
      ($action:expr, $branch_action:expr, $allow_submitter:expr) ) => {
        $action = if let Some(action) = $branch_action {
            if action.access_level <= $access || ($allow_submitter && $is_submitter) {
                CommandState::Requested($args.iter()
                    .map(|a| a.to_string())
                    .collect())
            } else {
                CommandState::Disallowed
            }
        } else {
            CommandState::Unavailable
        };
    };
    // Unconditional actions.
    ( $args:expr, $access:expr, $is_submitter:expr,
      ($action:expr) ) => {
        $action = CommandState::Requested($args.iter()
            .map(|a| a.to_string())
            .collect());
    };
}

macro_rules! check_actions {
    ( $command:expr, $args:expr, $access:expr, $is_submitter:expr, $unrecognized:expr,
      $( $name:expr => $action_spec:tt, )* ) => {
        match *$command {
            $( $name => { check_action!($args, $access, $is_submitter, $action_spec) }, )*
            unrecognized => $unrecognized.push(unrecognized.to_string()),
        }
    };
}

macro_rules! check_action_ok {
    { $method:ident, $( $action:expr => $name:expr, )* } => {
        let mut results = vec![];

        $( if $action.$method() {
            results.push($name);
        } )*

        results
    };
}

macro_rules! check_action_consistency {
    { $( ($action1:expr, $action2:expr) => ($name1:expr, $name2:expr), )* } => {
        let mut results = vec![];

        $( if $action1.requested() && $action2.requested() {
            results.push(concat!($name1, "` and `", $name2));
        } )*

        results
    };
}

impl<'a> Commands<'a> {
    /// Create a new commands parser for a branch.
    pub fn new(branch: &'a Branch) -> Self {
        Commands {
            branch: branch,

            check: CommandState::Unrequested,
            stage: CommandState::Unrequested,
            unstage: CommandState::Unrequested,
            merge: CommandState::Unrequested,
            test: CommandState::Unrequested,
            reformat: CommandState::Unrequested,

            unrecognized: vec![],
        }
    }

    /// Error messages from the commands.
    pub fn error_messages(&self, author: &User) -> Vec<String> {
        let mut messages = vec![];

        if !self.unrecognized.is_empty() {
            messages.push(format!("@{}: the following commands are not recognized at all: `{}`.",
                                  author.handle,
                                  self.unrecognized.join("`, `")));
        }

        let unavailable = self.unavailable_commands();
        if !unavailable.is_empty() {
            messages.push(format!("@{}: the following commands are not supported: `{}`.",
                                  author.handle,
                                  unavailable.join("`, `")));
        }

        let disallowed = self.disallowed_commands();
        if !disallowed.is_empty() {
            messages.push(format!("@{}: insufficient privileges for the commands: `{}`.",
                                  author.handle,
                                  disallowed.join("`, `")));
        }

        let consistency = self.consistency_messages();
        if !consistency.is_empty() {
            messages.push(format!("@{}: inconsistent commands: `{}`.",
                                  author.handle,
                                  consistency.join("`; `")));
        }

        messages
    }

    /// The set of unavailable commands.
    fn unavailable_commands(&self) -> Vec<&'static str> {
        check_action_ok! { unavailable,
            self.check => "check",
            self.stage => "stage",
            self.unstage => "unstage",
            self.merge => "merge",
            self.test => "test",
            self.reformat => "reformat",
        }
    }

    /// The set of disallowed commands.
    fn disallowed_commands(&self) -> Vec<&'static str> {
        check_action_ok! { disallowed,
            self.check => "check",
            self.stage => "stage",
            self.unstage => "unstage",
            self.merge => "merge",
            self.test => "test",
            self.reformat => "reformat",
        }
    }

    /// Checks whether the set of commands requested is consistent or not.
    ///
    /// Returns a list of messages about problems with the requested commands.
    fn consistency_messages(&self) -> Vec<&'static str> {
        check_action_consistency! {
            (self.check, self.stage) => ("check", "stage"),
            (self.check, self.unstage) => ("check", "unstage"),
            (self.check, self.merge) => ("check", "merge"),
            (self.check, self.reformat) => ("check", "reformat"),
            (self.stage, self.unstage) => ("stage", "unstage"),
            (self.stage, self.merge) => ("stage", "merge"),
            (self.stage, self.reformat) => ("stage", "reformat"),
            (self.unstage, self.reformat) => ("unstage", "reformat"),
            (self.test, self.merge) => ("test", "merge"),
            (self.test, self.reformat) => ("test", "reformat"),
            (self.merge, self.reformat) => ("merge", "reformat"),
        }
    }

    /// Parse commands from a set of trailers.
    pub fn add_from_trailers(&mut self, access: u64, is_submitter: bool, trailers: &[TrailerRef])
                             -> &mut Self {
        for trailer in trailers.iter() {
            match trailer.token {
                "do" | "Do" => {
                    // TODO: handle quoting
                    let values = trailer.value
                        .split_whitespace()
                        .collect::<Vec<_>>();
                    let (command, args) = values.split_first()
                        .expect("the regular expression should require a non-whitespace value");
                    check_actions!(command, args, access, is_submitter, self.unrecognized,
                        "check" => (self.check),
                        "stage" => (self.stage, self.branch.stage(), false),
                        "unstage" => (self.unstage, self.branch.stage(), true),
                        "merge" => (self.merge, self.branch.merge(), false),
                        "test" => (self.test, self.branch.test(), false),
                        "reformat" => (self.reformat, self.branch.reformat(), true),
                    );
                },
                // Ignore other trailers.
                _ => (),
            }
        }

        self
    }

    /// Returns `true` if there are no commands which were processed.
    pub fn is_empty(&self) -> bool {
        self.unrecognized.is_empty() &&
            !self.check.present() &&
            !self.stage.present() &&
            !self.unstage.present() &&
            !self.merge.present() &&
            !self.test.present() &&
            !self.reformat.present()
    }
}
