// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Common data structures required for the workflow.
//!
//! This module contains structures necessary to carry out the actual actions wanted for the
//! requested branches.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate ghostflow;
use self::ghostflow::actions::check;
use self::ghostflow::host::{CheckStatus, Comment, Commit, HostingService, Membership, MergeRequest,
                            Repo, User};
use self::ghostflow::utils::TrailerRef;

extern crate git_workarea;
use self::git_workarea::{ErrorKind, Result, ResultExt};

/// Information required to handle a push event.
#[derive(Debug, Clone)]
pub struct PushInfo {
    /// The commit being pushed.
    pub commit: Commit,
    /// The user who pushed the commit.
    pub author: User,
    /// When the push occurred.
    pub date: DateTime<UTC>,
}

/// Information required to handle a merge request update.
#[derive(Debug, Clone)]
pub struct MergeRequestInfo {
    /// The merge request information.
    pub merge_request: MergeRequest,
    /// Indicates whether the MR was just merged or not.
    pub was_merged: bool,
    /// Whether the merge request is open or not.
    pub is_open: bool,
    /// When the merge request was last updated.
    pub date: DateTime<UTC>,
}

impl MergeRequestInfo {
    fn trailers(&self) -> Vec<TrailerRef> {
        TrailerRef::extract(&self.merge_request.description)
    }

    /// Returns `true` if it appears that the source branch has been deleted.
    pub fn is_source_branch_deleted(&self) -> bool {
        self.merge_request.commit.id.as_str().is_empty()
    }

    /// The name of the topic for the merge request according to its description.
    pub fn topic_rename(&self) -> Option<&str> {
        self.trailers()
            .into_iter()
            .filter_map(|trailer| {
                if trailer.token == "Topic-rename" {
                    Some(trailer.value)
                } else {
                    None
                }
            })
            .next()
    }

    /// The name of the topic for the merge request for use in actions.
    pub fn topic_name(&self) -> &str {
        self.topic_rename()
            .unwrap_or(&self.merge_request.source_branch)
    }

    /// Determine the check status of a merge request.
    pub fn check_status(&self, service: &HostingService) -> Result<CheckStatus> {
        let service_user_id = service.service_user().id;
        Ok(service.get_commit_statuses(&self.merge_request.commit)
            .chain_err(|| ErrorKind::Msg(format!("failed to fetch commit statuses for the mr")))?
            .into_iter()
            // Only look at statuses posted by the current user.
            .filter(|status| status.author.id == service_user_id)
            // Only check the statuses we require.
            .find(|status| status.name == check::STATUS_NAME)
            .map_or(CheckStatus::Unchecked, |status| status.state.into()))
    }
}

/// Information required to handle a note on a merge request.
#[derive(Debug, Clone)]
pub struct MergeRequestNoteInfo {
    /// The merge request.
    pub merge_request: MergeRequestInfo,
    /// The note.
    pub note: Comment,
}

/// Information required to handle the addition of a user to a project.
#[derive(Debug, Clone)]
pub struct MembershipAdditionInfo {
    /// The repository.
    pub repo: Repo,
    /// The membership.
    pub membership: Membership,
}

/// Information required to handle the removal of a user from a project.
#[derive(Debug, Clone)]
pub struct MembershipRemovalInfo {
    /// The repository.
    pub repo: Repo,
    /// The user.
    pub user: User,
}
