// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate git_checks;
use self::git_checks::{BranchCheck, Check};
use self::git_checks::checks::*;

extern crate git_workarea;
use self::git_workarea::Identity;

extern crate serde;
// The code generator uses absolute paths.
use self::serde::{Deserialize, Deserializer};
use self::serde::de::Error as SerdeError;
use self::serde::de::Unexpected;

extern crate serde_json;
use self::serde_json::Value;

use config::defaults;
use error::*;

macro_rules! checks {
    { $req_kind:expr, $req_config:expr, $( $kind:expr => $config:ty, $dest:ident, )+ } => {
        match $req_kind {
            $( $kind => Ok($dest.push(serde_json::from_value::<$config>($req_config)
                                      .chain_err(|| format!("failed to parse {} config",
                                                            $req_kind))?
                                      .into_check())), )*
            _ => Err(format!("unknown check: {}", $req_kind).into()),
        }
    }
}

pub fn create_check(kind: &str, config: Value, checks: &mut Vec<Box<Check>>,
                    branch_checks: &mut Vec<Box<BranchCheck>>)
                    -> Result<()> {
    checks! {
        kind, config,
        "allow_robot" => AllowRobotConfig, branch_checks,
        "bad_commits" => BadCommitsConfig, checks,
        "check_end_of_line" => CheckEndOfLineConfig, checks,
        "check_executable_permissions" => CheckExecutablePermissionsConfig, checks,
        "check_size" => CheckSizeConfig, checks,
        "check_whitespace" => CheckWhitespaceConfig, checks,
        "commit_subject" => CommitSubjectConfig, checks,
        "formatting" => FormattingConfig, checks,
        "invalid_paths" => InvalidPathsConfig, checks,
        "invalid_utf8" => InvalidUtf8Config, checks,
        "reject_merges" => RejectMergesConfig, checks,
        "reject_separate_root" => RejectSeparateRootConfig, checks,
        "reject_symlinks" => RejectSymlinksConfig, checks,
        "release_branch" => ReleaseBranchConfig, branch_checks,
        "restricted_path" => RestrictedPathConfig, checks,
        "submodule_available" => SubmoduleAvailableConfig, checks,
        "submodule_rewind" => SubmoduleRewindConfig, checks,
        "submodule_watch" => SubmoduleWatchConfig, checks,
        "third_party" => ThirdPartyConfig, checks,
        "valid_name" => ValidNameConfig, checks,
    }
}

#[derive(Deserialize, Debug)]
struct AllowRobotConfig {
    name: String,
    email: String,
}

impl AllowRobotConfig {
    fn into_check(self) -> Box<BranchCheck> {
        let identity = Identity::new(&self.name, &self.email);
        Box::new(AllowRobot::new(identity))
    }
}

#[derive(Deserialize, Debug)]
struct BadCommitsConfig {
    bad_commits: Vec<String>,
}

impl BadCommitsConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(BadCommits::new(&self.bad_commits))
    }
}

#[derive(Deserialize, Debug)]
struct CheckEndOfLineConfig {
}

impl CheckEndOfLineConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(CheckEndOfLine::new())
    }
}

#[derive(Deserialize, Debug)]
struct CheckExecutablePermissionsConfig {
    extensions: Vec<String>,
}

impl CheckExecutablePermissionsConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(CheckExecutablePermissions::new(&self.extensions))
    }
}

fn default_max_size() -> usize {
    1 << 20
}

#[derive(Deserialize, Debug)]
struct CheckSizeConfig {
    #[serde(default="default_max_size")]
    max_size: usize,
}

impl CheckSizeConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(CheckSize::new(self.max_size))
    }
}

#[derive(Deserialize, Debug)]
struct CheckWhitespaceConfig {
}

impl CheckWhitespaceConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(CheckWhitespace::new())
    }
}

fn default_min_subject_length() -> usize {
    8
}

fn default_max_subject_length() -> usize {
    78
}

#[derive(Deserialize, Debug)]
struct CommitSubjectConfig {
    #[serde(default="default_min_subject_length")]
    min_length: usize,
    #[serde(default="default_max_subject_length")]
    max_length: usize,

    #[serde(default="defaults::default_true")]
    check_wip: bool,
    #[serde(default="defaults::default_true")]
    check_rebase_commands: bool,
}

impl CommitSubjectConfig {
    fn into_check(self) -> Box<Check> {
        let mut check = CommitSubject::new();

        check
            .with_summary_limits(self.min_length, self.max_length)
            .check_work_in_progress(self.check_wip)
            .check_rebase_commands(self.check_rebase_commands);

        Box::new(check)
    }
}

#[derive(Deserialize, Debug)]
struct FormattingConfig {
    kind: String,
    formatter: String,
    #[serde(default)]
    config_files: Vec<String>,
    #[serde(default)]
    fix_message: Option<String>,
}

impl FormattingConfig {
    fn into_check(self) -> Box<Check> {
        let mut check = Formatting::new(self.kind, self.formatter);

        check.add_config_files(self.config_files);

        self.fix_message.map(|message| check.with_fix_message(message));

        Box::new(check)
    }
}

fn default_invalid_characters() -> String {
    r#"<>:"|?*"#.to_string()
}

#[derive(Deserialize, Debug)]
struct InvalidPathsConfig {
    #[serde(default="default_invalid_characters")]
    invalid_characters: String,
}

impl InvalidPathsConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(InvalidPaths::new(self.invalid_characters))
    }
}

#[derive(Deserialize, Debug)]
struct InvalidUtf8Config {
}

impl InvalidUtf8Config {
    fn into_check(self) -> Box<Check> {
        Box::new(InvalidUtf8::new())
    }
}

#[derive(Deserialize, Debug)]
struct RejectMergesConfig {
}

impl RejectMergesConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(RejectMerges::new())
    }
}

#[derive(Deserialize, Debug)]
struct RejectSeparateRootConfig {
}

impl RejectSeparateRootConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(RejectSeparateRoot::new())
    }
}

#[derive(Deserialize, Debug)]
struct RejectSymlinksConfig {
}

impl RejectSymlinksConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(RejectSymlinks::new())
    }
}

#[derive(Deserialize, Debug)]
struct ReleaseBranchConfig {
    #[serde(default="defaults::default_release_branch")]
    branch: String,
    disallowed_commit: String,
    #[serde(default="defaults::default_false")]
    required: bool,
}

impl ReleaseBranchConfig {
    fn into_check(self) -> Box<BranchCheck> {
        let mut check = ReleaseBranch::new(self.branch, self.disallowed_commit);

        check.set_required(self.required);

        Box::new(check)
    }
}

#[derive(Deserialize, Debug)]
struct RestrictedPathConfig {
    restricted_path: String,
    #[serde(default="defaults::default_true")]
    required: bool,
}

impl RestrictedPathConfig {
    fn into_check(self) -> Box<Check> {
        let mut check = RestrictedPath::new(&self.restricted_path);

        check.required(self.required);

        Box::new(check)
    }
}

#[derive(Deserialize, Debug)]
struct SubmoduleAvailableConfig {
    #[serde(default="defaults::default_false")]
    require_first_parent: bool,
}

impl SubmoduleAvailableConfig {
    fn into_check(self) -> Box<Check> {
        let mut check = SubmoduleAvailable::new();

        check.require_first_parent(self.require_first_parent);

        Box::new(check)
    }
}

#[derive(Deserialize, Debug)]
struct SubmoduleRewindConfig {
}

impl SubmoduleRewindConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(SubmoduleRewind::new())
    }
}

#[derive(Deserialize, Debug)]
struct SubmoduleWatchConfig {
    #[serde(default="defaults::default_false")]
    reject_additions: bool,
    #[serde(default="defaults::default_false")]
    reject_removals: bool,
}

impl SubmoduleWatchConfig {
    fn into_check(self) -> Box<Check> {
        let mut check = SubmoduleWatch::new();

        check
            .reject_additions(self.reject_additions)
            .reject_removals(self.reject_removals);

        Box::new(check)
    }
}

#[derive(Deserialize, Debug)]
struct ThirdPartyConfig {
    name: String,
    path: String,
    root: String,
    script: String,
}

impl ThirdPartyConfig {
    fn into_check(self) -> Box<Check> {
        Box::new(ThirdParty::new(self.name, self.path, self.root, self.script))
    }
}

#[derive(Debug, Clone, Copy)]
enum ValidNameFullNamePolicyIO {
    Required,
    Preferred,
    Optional,
}
enum_serialize!(ValidNameFullNamePolicyIO -> "full name policy",
    Required => "required",
    Preferred => "preferred",
    Optional => "optional",
);

impl Default for ValidNameFullNamePolicyIO {
    fn default() -> Self {
        ValidNameFullNamePolicyIO::Required
    }
}

impl From<ValidNameFullNamePolicyIO> for ValidNameFullNamePolicy {
    fn from(policy: ValidNameFullNamePolicyIO) -> Self {
        match policy {
            ValidNameFullNamePolicyIO::Required => ValidNameFullNamePolicy::Required,
            ValidNameFullNamePolicyIO::Preferred => ValidNameFullNamePolicy::Preferred,
            ValidNameFullNamePolicyIO::Optional => ValidNameFullNamePolicy::Optional,
        }
    }
}

#[derive(Deserialize, Debug)]
struct ValidNameConfig {
    #[serde(default)]
    full_name_policy: ValidNameFullNamePolicyIO,
}

impl ValidNameConfig {
    fn into_check(self) -> Box<Check> {
        let mut check = ValidName::new();

        check
            .set_full_name_policy(self.full_name_policy.into());

        Box::new(check)
    }
}
