// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The Kitware robot
//!
//! This program watches a directory for JSON jobs in order to manage the Kitware workflow for
//! repositories.

#![warn(missing_docs)]

#[macro_use]
extern crate clap;
use clap::{Arg, App};

extern crate json_job_dispatch;
use json_job_dispatch::{Director, DirectorWatchdog, Handler, RunResult};

#[macro_use]
extern crate lazy_static;

#[macro_use]
extern crate log;
use log::LogLevel;

#[macro_use]
extern crate quick_error;

extern crate systemd;
use systemd::journal::JournalLog;

pub mod config;
use config::{Config, ConfigRead};

pub mod handlers;
use handlers::{connect_to_host, create_handler};

use std::error::Error;
use std::path::Path;

fn run_director(config_path: &Path) -> Result<RunResult, Box<Error>> {
    let mut config = try!(Config::from_path(&config_path, connect_to_host));

    let watchdog = DirectorWatchdog {};
    let mut handlers = vec![];

    for (name, host) in config.hosts.drain().into_iter() {
        handlers.push(try!(create_handler(host, &name)))
    }

    let mut director = try!(Director::new(&config.archive_dir));

    try!(watchdog.add_to_director(&mut director));
    for handler in &handlers {
        try!(handler.add_to_director(&mut director));
    }

    Ok(try!(director.watch_directory(&config.queue_dir)))
}

fn try_main() -> Result<(), Box<Error>> {
    let matches = App::new("kwrobot")
        .version(crate_version!())
        .author("Ben Boeckel <ben.boeckel@kitware.com>")
        .about("Watch a directory for project events and perform workflow actions on them")
        .arg(Arg::with_name("CONFIG")
            .short("c")
            .long("config")
            .help("Path to the configuration file")
            .required(true)
            .takes_value(true))
        .arg(Arg::with_name("DEBUG")
            .short("d")
            .long("debug")
            .help("Increase verbosity")
            .multiple(true))
        .arg(Arg::with_name("VERIFY")
            .short("v")
            .long("verify")
            .help("Check the configuration file and exit"))
        .get_matches();

    let log_level = match matches.occurrences_of("DEBUG") {
            0 => LogLevel::Error,
            1 => LogLevel::Warn,
            2 => LogLevel::Info,
            3 => LogLevel::Debug,
            4 | _ => LogLevel::Trace,
        };
    try!(JournalLog::init_with_level(log_level.to_log_level_filter()));

    let config_path = Path::new(matches.value_of("CONFIG").unwrap());

    if matches.is_present("VERIFY") {
        try!(ConfigRead::from_path(&config_path));
    } else {
        while let RunResult::Restart = try!(run_director(&config_path)) { }
    }

    Ok(())
}

fn main() {
    if let Err(err) = try_main() {
        panic!("{:?}", err);
    }
}
