// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate gitlab;
use self::gitlab::Gitlab;
use self::gitlab::webhooks;

extern crate json_job_dispatch;
use self::json_job_dispatch::{Director, Handler, HandlerResult};
use self::json_job_dispatch::Error as HandlerError;

extern crate kitware_workflow;
use self::kitware_workflow::host::{GitlabService, HostingService};

extern crate serde;
use self::serde::Deserialize;

extern crate serde_json;
use self::serde_json::{from_value, Value};

use super::traits::*;
use super::super::super::super::config::Host;
use super::super::super::common::handlers::*;

use std::rc::Rc;
use std::error::Error;

pub fn connect_to_host(url: &Option<String>, secrets: Value)
                       -> Result<Rc<HostingService>, Box<Error>> {
    let host = url.as_ref().map_or("gitlab.com", |u| u.as_str());
    let token = match secrets.pointer("/token").and_then(|t| t.as_string()) {
        Some(t) => t,
        None => return Err(unimplemented!()),
    };

    Ok(try!(Gitlab::new(host, token)
        .map(GitlabService::new)
        .map(|service| Rc::new(service) as Rc<HostingService>)))
}

struct GitlabHandler {
    host: Host,
    name: String,

    push_name: String,
    mr_name: String,
    note_name: String,
}

macro_rules! try_action {
    ( $action:expr ) => {
        match $action {
            Ok(res) => res,
            Err(handler_result) => return Ok(handler_result),
        }
    }
}

impl GitlabHandler {
    fn new(host: Host, name: String) -> Self {
        GitlabHandler {
            host: host,

            push_name: Self::push_handler_name(&name),
            mr_name: Self::mr_handler_name(&name),
            note_name: Self::note_handler_name(&name),

            name: name,
        }
    }

    fn read_hook<T: Deserialize>(object: Value) -> Result<T, HandlerResult> {
        from_value::<T>(object)
            .map_err(|err| {
                HandlerResult::Reject(format!("failed to parse event: {:?}", err))
            })
    }

    fn handle_push(&self, object: &Value) -> Result<HandlerResult, HandlerError> {
        let push = try_action!(Self::read_hook::<webhooks::PushHook>(object.clone()));

        handle_push(object, &self.host, GitlabPush::new(push))
    }

    fn handle_mr(&self, object: &Value) -> Result<HandlerResult, HandlerError> {
        let mr = try_action!(Self::read_hook::<webhooks::MergeRequestHook>(object.clone()));

        handle_merge_request_update(object, &self.host, GitlabMergeRequest::new(mr.object_attributes))
    }

    fn handle_note(&self, object: &Value) -> Result<HandlerResult, HandlerError> {
        let note = try_action!(Self::read_hook::<webhooks::NoteHook>(object.clone()));

        if let Some(mr_attrs) = note.merge_request {
            let mr_note = GitlabMergeRequestNote::new(mr_attrs, note.object_attributes);
            handle_merge_request_note(object, &self.host, mr_note)
        } else {
            Ok(HandlerResult::Reject("unhandled note object".to_string()))
        }
    }

    fn handler_name(name: &str, kind: &str) -> String {
        format!("gitlab/{}/{}", name, kind)
    }

    fn push_handler_name(name: &str) -> String {
        Self::handler_name(name, "push")
    }

    fn mr_handler_name(name: &str) -> String {
        Self::handler_name(name, "merge_request")
    }

    fn note_handler_name(name: &str) -> String {
        Self::handler_name(name, "note")
    }
}

impl Handler for GitlabHandler {
    fn name(&self) -> &str {
        &self.name
    }

    fn add_to_director<'a>(&'a self, director: &mut Director<'a>) -> Result<(), HandlerError> {
        try!(director.add_handler(&self.push_name, self));
        try!(director.add_handler(&self.mr_name, self));
        try!(director.add_handler(&self.note_name, self));

        Ok(())
    }

    fn handle(&self, kind: &str, object: &Value) -> Result<HandlerResult, HandlerError> {
        if kind == self.push_name {
            self.handle_push(object)
        } else if kind == self.mr_name {
            self.handle_mr(object)
        } else if kind == self.note_name {
            self.handle_note(object)
        } else {
            Ok(HandlerResult::Reject(format!("gitlab received an unhandled {} job", kind)))
        }
    }
}

pub fn create_handler(host: Host, name: &str) -> Result<Box<Handler>, Box<Error>> {
    Ok(Box::new(GitlabHandler::new(host, format!("gitlab/{}", name))))
}
