// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Core handler traits for kitware workflow actions.
//!
//! This module contains traits necessary to carry out the actual actions wanted for the requested
//! branches.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate kitware_git;
use self::kitware_git::{CommitId, Identity};

extern crate kitware_workflow;
use self::kitware_workflow::host::{HostedCommit, HostedMergeRequest, HostedRepo};

/// Trait representing information required to handle a push event.
pub trait ProjectInfo {
    /// A `HostedRepo` for the project.
    fn as_hosted(&self) -> &HostedRepo;

    /// The ID of the project.
    fn id(&self) -> u64;
    /// The name of the project.
    fn name(&self) -> &str;
    /// The root of the project's fork tree.
    fn fork_root(&self) -> &str;
}

/// Trait representing information required to handle a push event.
pub trait PushInfo {
    /// A `HostedCommit` for the push.
    fn as_hosted(&self) -> &HostedCommit;

    /// The project pushed into.
    fn project(&self) -> &ProjectInfo;
    /// The user who pushed the commit.
    fn author(&self) -> &Identity;
    /// The name of the ref pushed into.
    fn refname(&self) -> &str;
    /// When the push occurred.
    fn date(&self) -> &DateTime<UTC>;
}

/// The status of the hooks having been run on a merge request.
pub enum CheckStatus {
    /// The checks passed.
    Pass,
    /// The checks failed.
    Fail,
    /// The checks have not been run.
    Unchecked,
}

impl CheckStatus {
    /// Whether the checks have been run or not.
    pub fn is_checked(&self) -> bool {
        match *self {
            CheckStatus::Pass | CheckStatus::Fail => true,
            CheckStatus::Unchecked => false,
        }
    }

    /// Whether the checks need to be run or not.
    pub fn is_ok(&self) -> bool {
        match *self {
            CheckStatus::Pass => true,
            CheckStatus::Fail | CheckStatus::Unchecked => false,
        }
    }
}

/// Trait representing information required to handle a merge request update.
pub trait MergeRequestInfo {
    /// A `HostedMergeRequest` for the merge request.
    fn as_hosted(&self) -> &HostedMergeRequest;

    /// Whether the merge request is open or not.
    fn is_open(&self) -> bool;
    /// The ID of the merge request.
    fn id(&self) -> u64;
    /// The hook check status of the merge request.
    fn check_status(&self) -> CheckStatus;
    /// The author of the merge request.
    fn author(&self) -> &Identity;
    /// The access level of the author of the merge request to the target project.
    fn author_access(&self) -> u64;

    /// The source project.
    fn source_project(&self) -> &ProjectInfo;
    /// The name of the source branch.
    fn source_branch(&self) -> &str;
    /// The previous revision of the source branch (if available).
    fn source_branch_oldsha(&self) -> &Option<CommitId>;
    /// The current revision of the source branch.
    fn source_branch_sha(&self) -> &CommitId;
    /// When the merge request was last updated.
    fn date(&self) -> &DateTime<UTC>;

    /// The source project.
    fn target_project(&self) -> &ProjectInfo;
    /// The name of the target branch.
    fn target_branch(&self) -> &str;
}

/// Trait representing information required to handle a note.
pub trait NoteInfo {
    /// The content of the note.
    fn content(&self) -> &str;
    /// The creation date of the note.
    fn date(&self) -> &DateTime<UTC>;
    /// The author of the note.
    fn author(&self) -> &Identity;
    /// The author of the note's username.
    fn username(&self) -> &str;
    /// The access level of the author of the note relevant project.
    fn author_access(&self) -> u64;
}

/// Trait representing information required to handle a note on a merge request.
pub trait MergeRequestNoteInfo {
    /// The merge request.
    fn merge_request(&self) -> &MergeRequestInfo;
    /// The note.
    fn note(&self) -> &NoteInfo;
}
