// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Trailer extraction functions.
//!
//! Trailers are key/value pairs of strings at the end of commit messages which provide metadata
//! about people involved with the commit and/or branch such as those who reported the issue fixed
//! in the commit, reviewers, copyright notices, etc.

extern crate itertools;
use self::itertools::Itertools;

extern crate regex;
use self::regex::Regex;

lazy_static! {
    static ref TRAILER_RE: Regex =
        Regex::new("^\
                    (?P<token>[[:alpha:]-]+)\
                    :\\s+\
                    (?P<value>.+?)\
                    \\s*\
                    $").unwrap();
}

#[derive(Debug, PartialEq, Eq)]
/// A trailer from a commit message.
pub struct Trailer<'a> {
    /// The name of the trailer.
    pub token: &'a str,
    /// The value for the trailer.
    pub value: &'a str,
}

impl<'a> Trailer<'a> {
    fn new(token: &'a str, value: &'a str) -> Self {
        Trailer {
            token: token,
            value: value,
        }
    }

    /// Extract trailers from a commit message.
    pub fn extract(content: &'a str) -> Vec<Self> {
        let mut trailers = content.lines()
            .rev()
            .skip_while(|line| line.is_empty())
            .map(|line| TRAILER_RE.captures(line))
            .while_some()
            .map(|trailer| {
                Self::new(trailer.name("token").unwrap(),
                          trailer.name("value").unwrap())
            })
            .collect::<Vec<_>>();

        trailers.reverse();

        trailers
    }
}

#[cfg(test)]
mod test {
    use super::Trailer;

    fn check_content(content: &str, expected: &[(&str, &str)]) {
        assert_eq!(Trailer::extract(content),
                   expected.iter()
                    .map(|trailer| {
                        let &(token, value) = trailer;
                        Trailer::new(token, value)
                    })
                    .collect::<Vec<_>>());
    }

    #[test]
    fn test_trailers_extract_no_trailers() {
        let content = "\
            Some simple content.\
            ";
        let expected = &[];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_simple() {
        let content = "\
            Some simple content.\n\
            \n\
            Token: value\
            ";
        let expected = &[
                ("Token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_extra_whitespace_between() {
        let content = "\
            Some simple content.\n\
            \n\
            Token:   value\
            ";
        let expected = &[
                ("Token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_extra_whitespace_trailing() {
        let content = "\
            Some simple content.\n\
            \n\
            Token: value  \
            ";
        let expected = &[
                ("Token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_extra_whitespace_trailing_newline() {
        let content = "\
            Some simple content.\n\
            \n\
            Token: value  \n\
            ";
        let expected = &[
                ("Token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_extra_whitespace_both() {
        let content = "\
            Some simple content.\n\
            \n\
            Token:   value  \
            ";
        let expected = &[
                ("Token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_trailers_trailing_newline() {
        let content = "\
            Some simple content.\n\
            \n\
            Token: value\n\
            ";
        let expected = &[
                ("Token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_trailers_trailing_whitespace_line() {
        let content = "\
            Some simple content.\n\
            \n\
            Token: value\n
            ";
        let expected = &[];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_multiple_trailers() {
        let content = "\
            Some simple content.\n\
            \n\
            Token: value\n\
            Other-token: value\n\
            ";
        let expected = &[
                ("Token", "value"),
                ("Other-token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_handle_blank_lines_mid() {
        let content = "\
            Some simple content.\n\
            \n\
            Token: value\n\
            \n\
            Other-token: value\n\
            ";
        let expected = &[
                ("Other-token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_trailing_blank_line() {
        let content = "\
            Some simple content.\n\
            \n\
            Token: value\n\
            \n\
            ";
        let expected = &[
                ("Token", "value"),
            ];

        check_content(content, expected);
    }

    #[test]
    fn test_trailers_extract_bogus() {
        let content = "\
            Some simple content.\n\
            \n\
            Missed: value\n\
            \n\
            Token: value\
            ";
        let expected = &[
                ("Token", "value"),
            ];

        check_content(content, expected);
    }
}
