// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Core handler logic for kitware workflow actions.
//!
//! This module contains functions which carry out the actual actions wanted for the requested
//! branches.

extern crate itertools;
use self::itertools::Itertools;

extern crate json_job_dispatch;
use self::json_job_dispatch::{Error, HandlerResult};

extern crate kitware_git;
use self::kitware_git::CommitId;

extern crate kitware_workflow;
use self::kitware_workflow::host::HostError;

extern crate serde_json;
use self::serde_json::Value;

use super::commands::Commands;
use super::trailers::Trailer;
use super::traits::{MergeRequestInfo, MergeRequestNoteInfo, PushInfo};
use super::super::super::config::{Branch, Host, Project, StageUpdatePolicy};

macro_rules! try_action {
    ( $action:expr ) => {
        match $action {
            Ok(res) => res,
            Err(handler_result) => return Ok(handler_result),
        }
    }
}

fn get_project<'a>(host: &'a Host, project_name: &str) -> Result<&'a Project, HandlerResult> {
    host.projects.get(project_name)
        .ok_or_else(|| HandlerResult::Reject(format!("unwatched project {}", project_name)))
}

fn get_branch<'a>(host: &'a Host, project_name: &str, branch_name: &str)
                  -> Result<(&'a Project, &'a Branch), HandlerResult> {
    get_project(host, project_name)
        .and_then(|project| {
            project.branches.get(branch_name)
                .map(|branch| (project, branch))
                .ok_or_else(|| HandlerResult::Reject(format!("unwatched branch {} for project {}", branch_name, project_name)))
        })
}

fn handle_result<F>(notes: Vec<String>, post_comment: F) -> Result<HandlerResult, Error>
    where F: Fn(&str) -> Result<(), HostError>
{
    if notes.is_empty() {
        Ok(HandlerResult::Accept)
    } else {
        let md_comment = notes.join("  \n");
        let txt_comment = notes.join("\n");

        if let Err(err) = post_comment(&md_comment) {
            error!(target: "kwrobot/handler",
                   "Failed to post comment:\n'{}'\n'{}'.",
                   md_comment, err);
        }

        Ok(HandlerResult::Reject(txt_comment))
    }
}

/// Handle a push to a repository.
pub fn handle_push<P: PushInfo>(_: &Value, host: &Host, push: P) -> Result<HandlerResult, Error> {
    let refs_heads_prefix = "refs/heads/";
    if !push.refname().starts_with(refs_heads_prefix) {
        return Ok(HandlerResult::Reject(format!("non-branch ref push {}", push.refname())));
    }

    let branch = &push.refname()[refs_heads_prefix.len()..];
    let (project, branch) = try_action!(get_branch(host, push.project().name(), branch));

    let mut comment_notes = vec![];

    let fetch_res = project.context.fetch_into("origin", push.refname(), push.refname());
    if let Err(err) = fetch_res {
        comment_notes.push(format!("Failed to fetch from the repository: {:?}.",
                                   err));
    }

    if let Some(stage_action) = branch.stage() {
        let mut stage = stage_action.stage();
        let res = stage.base_branch_update(push.as_hosted(), push.author(), push.date());

        if let Err(err) = res {
            comment_notes.push(format!("Error occurred when updating the base branch ({}): `{:?}`",
                                       host.maintainers.join(" "),
                                       err));
        }
    }

    handle_result(comment_notes, |comment| {
            host.service.post_commit_comment(push.as_hosted(), comment)
        })
}

/// Handle an update to a merge request.
pub fn handle_merge_request_update<M: MergeRequestInfo>(_: &Value, host: &Host, mr: M) -> Result<HandlerResult, Error> {
    if !mr.is_open() {
        return Ok(HandlerResult::Reject(format!("the MR for the branch {} has been closed", mr.source_branch())));
    }

    let (project, branch) = try_action!(get_branch(host, mr.target_project().name(), mr.target_branch()));

    let mut comment_notes = vec![];

    let hosted_mr = mr.as_hosted();
    let source_project = hosted_mr.source_repo();
    let fetch_res = project.context.fetch(source_project.url(),
                                          &[hosted_mr.source_branch()]);
    if let Err(err) = fetch_res {
        comment_notes.push(format!("Failed to fetch from the repository: {:?}.",
                                   err));
    }

    let reserve_res = project.context.reserve_ref(&format!("refs/mr/{}", hosted_mr.id()),
                                                  &CommitId::new(hosted_mr.commit().id()));
    if let Err(err) = reserve_res {
        comment_notes.push(format!("Failed to reserve ref {} for the merge request: {:?}",
                                   hosted_mr.commit().id(),
                                   err));
    }

    let is_ok = mr.check_status().is_ok();
    let is_wip = hosted_mr.work_in_progress();

    if !is_ok {
        let hook_result = branch.check_branch(mr.source_branch_sha(), &format!("mr/{}", mr.id()), mr.author());
        // TODO: add the status
        // TODO: create comment (if necessary)
    }

    if let Some(stage_action) = branch.stage() {
        let (policy, reason) = if !is_ok {
                (StageUpdatePolicy::Unstage, "since it is failing its checks")
            } else if is_wip {
                (StageUpdatePolicy::Unstage, "since it is marked as work-in-progress")
            } else {
                (stage_action.policy, "as per policy")
            };

        let mut stage = stage_action.stage();
        let (what, res) = match policy {
                StageUpdatePolicy::Ignore => ("ignore", Ok(())),
                StageUpdatePolicy::Restage => {
                    ("restage", stage.stage_merge_request(mr.as_hosted(), mr.author(), mr.date()))
                },
                StageUpdatePolicy::Unstage => {
                    ("unstage", stage.unstage_update_merge_request(mr.as_hosted(), reason))
                },
            };

        if let Err(err) = res {
            comment_notes.push(format!("Failed to {}: `{:?}`",
                                        what, err));
        }
    }

    handle_result(comment_notes, |comment| {
            host.service.post_mr_comment(mr.as_hosted(), comment)
        })
}

/// Handle a note on a merge request.
pub fn handle_merge_request_note<M: MergeRequestNoteInfo>(_: &Value, host: &Host, mr_note: M) -> Result<HandlerResult, Error> {
    let mr = mr_note.merge_request();
    let note = mr_note.note();
    let (_, branch) = try_action!(get_branch(host, mr.target_project().name(), mr.target_branch()));

    let mut commands = Commands::new(branch);

    let trailers = Trailer::extract(mr_note.note().content());
    commands.add_from_trailers(note.author_access(), &trailers);

    let mut comment_notes = vec![];

    let disallowed = commands.disallowed_commands();
    let consistency = commands.consistency_messages();
    if disallowed.is_empty() && consistency.is_empty() {
        if commands.stage.requested() {
            if let Some(stage_action) = branch.stage() {
                let mut stage = stage_action.stage();

                let mut do_unstage = false;
                let mut unrecognized = vec![];

                for arg in commands.stage.arguments() {
                    match arg.as_str() {
                        "--unstage" => do_unstage = true,
                        _ => unrecognized.push(arg),
                    };
                }

                if unrecognized.is_empty() {
                    let (what, res) = if do_unstage {
                            ("unstage", stage.unstage_merge_request(mr.as_hosted()))
                        } else {
                            ("stage", stage.stage_merge_request(mr.as_hosted(), mr.author(), note.date()))
                        };

                    if let Err(err) = res {
                        comment_notes.push(format!("Error occurred during {} action ({}): `{:?}`",
                                                   what,
                                                   host.maintainers.join(" "),
                                                   err));
                    }
                } else {
                    comment_notes.push(format!("Unrecognized `stage` arguments: `{}`.",
                                               unrecognized.iter().join("`, `")));
                }
            }
        }
    } else {
        if !disallowed.is_empty() {
            comment_notes.push(format!("@{}: insufficient privileges for the commands: `{}`.",
                                       note.username(),
                                       disallowed.join("`, `")));
        }

        if !consistency.is_empty() {
            comment_notes.push(format!("@{}: inconsistent commands: `{}`.",
                                       note.username(),
                                       consistency.join("`, `")));
        }
    }

    handle_result(comment_notes, |comment| {
            host.service.post_mr_comment(mr.as_hosted(), comment)
        })
}
