// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate git_hooks;
use self::git_hooks::{BranchHook, GitHookConfiguration, Hook, HookResult};
use self::git_hooks::{list_refs, run_hooks};
use self::git_hooks::Error as HookError;

extern crate kitware_git;
use self::kitware_git::{CommitId, GitContext, Identity, SubmoduleInfo, SubmoduleMap};

extern crate kitware_stager;
use self::kitware_stager::Stager;

extern crate kitware_workflow;
use self::kitware_workflow::actions::{clone, stage};
use self::kitware_workflow::host::{HostedProject, HostingService};

extern crate serde_json;
use self::serde_json::Value;

use super::hooks::create_hook;
use super::io;

use std::cell::{RefCell, RefMut};
use std::collections::hash_map::HashMap;
use std::collections::hash_set::HashSet;
use std::error::Error;
use std::path::{Path, PathBuf};
use std::rc::Rc;

pub use super::io::StageUpdatePolicy;

/// Main configuration object for the robot.
pub struct Config {
    /// The archive directory.
    pub archive_dir: PathBuf,
    /// The directory to watch for new jobs.
    pub queue_dir: PathBuf,
    /// Host configuration.
    pub hosts: HashMap<String, Host>,
}

impl Config {
    /// Read the configuration from a path.
    ///
    /// The `connect_to_host` function constructs `HostingService` instances given an API and a URL
    /// to communicate with.
    pub fn from_path<P, F>(path: P, connect_to_host: F) -> Result<Self, Box<Error>>
        where P: AsRef<Path>,
              F: Fn(&str, &Option<String>, Value) -> Result<Rc<HostingService>, Box<Error>>,
    {
        io::Config::from_path(path)
            .and_then(|config| {
                let archive_dir = config.archive_dir().to_path_buf();
                let queue_dir = config.queue_dir().to_path_buf();
                let workdir = PathBuf::from(config.workdir);
                let hosts = try!(config.hosts.into_iter()
                    .map(|(name, host)| {
                        let host_workdir = workdir.join(&name);
                        let secrets = try!(host.secrets());
                        let service = try!(connect_to_host(&host.host_api, &host.host_url, secrets));
                        let host = try!(Host::new(host, service, host_workdir));

                        Ok((name, host))
                    })
                    .collect::<Result<HashMap<_, _>, Box<Error>>>());

                Ok(Config {
                    archive_dir: archive_dir,
                    queue_dir: queue_dir,
                    hosts: hosts,
                })
            })
    }
}

/// Configuration for a host.
pub struct Host {
    /// The API used to communicate with the host.
    pub api: String,
    /// The `HostingService` instance to communicate with the host.
    pub service: Rc<HostingService>,
    /// The users to notify when errors occur with the workflow.
    pub maintainers: Vec<String>,
    /// Projects which have workflow actions configured.
    pub projects: HashMap<String, Project>,
    /// A directory which may be used for on-disk scratchspace.
    pub workdir: PathBuf,
}

quick_error! {
    #[derive(Debug)]
    enum HostError {
        NoMaintainers {
            display("no maintainers given")
        }
    }
}

impl Host {
    fn new(host: io::Host, service: Rc<HostingService>, workdir: PathBuf) -> Result<Self, Box<Error>> {
        let project_workdir = workdir.join("projects");
        let projects = try!(host.projects.into_iter()
            .map(|(name, project)| {
                let project_workdir = project_workdir.join(&name);
                let project = try!(Project::new(&name, project, service.clone(), project_workdir));

                Ok((name, project))
            })
            .collect::<Result<HashMap<_, _>, Box<Error>>>());

        if host.maintainers.is_empty() {
            return Err(Box::new(HostError::NoMaintainers));
        }

        Ok(Host {
            api: host.host_api,
            service: service,
            maintainers: host.maintainers,
            projects: projects,
            workdir: workdir,
        })
    }
}

/// Configuration for a project.
pub struct Project {
    /// The `HostingService` instance to communicate with the host.
    pub service: Rc<HostingService>,
    /// A context for working with the git repository of the project.
    pub context: GitContext,
    /// Branches which have workflow actions configured.
    pub branches: HashMap<String, Branch>,
}

impl Project {
    fn new(name: &str, project: io::Project, service: Rc<HostingService>, workdir: PathBuf) -> Result<Self, Box<Error>> {
        let hosted_project = HostedProject {
                name: name.to_string(),
                service: service.clone(),
            };
        let clone = clone::Clone_::new(&workdir, hosted_project);
        let context = try!(clone.clone_watched_repo());
        let identity = Identity::new(project.name.clone(), project.email.clone());
        let submodules = project.submodules.iter()
            .map(|(name, submodule)| {
                let submodule_info = SubmoduleInfo::new(&submodule.gitdir, &CommitId::new(&submodule.branch));
                (name.into(), submodule_info)
            })
            .collect::<SubmoduleMap>();

        let branches = try!(project.branches.iter()
            .map(|(name, branch)| {
                let hooks = try!(Hooks::new(branch.hooks.iter().chain(project.hooks.iter())));
                let merge_hooks = try!(Hooks::new(branch.merge_hooks.iter().chain(project.hooks.iter())));

                let mut cbranch = try!(Branch::new(&name, hooks, merge_hooks, context.clone(),
                                                   submodules.clone(), identity.clone()));

                if let Some(ref stage) = branch.stage {
                    let hosted_project = HostedProject {
                            name: name.clone(),
                            service: service.clone(),
                        };

                    try!(cbranch.add_stage(&stage, hosted_project));
                }

                Ok((name.to_string(), cbranch))
            })
            .collect::<Result<HashMap<_, _>, Box<Error>>>());

        Ok(Project {
            service: service,
            context: context,
            branches: branches,
        })
    }
}

struct Hooks {
    hooks: Vec<Box<Hook>>,
    branch_hooks: Vec<Box<BranchHook>>,
}

impl Hooks {
    fn new<'a, I: Iterator<Item=(&'a String, &'a io::HookConfig)>>(hook_configs: I) -> Result<Self, Box<Error>> {
        let mut hooks = vec![];
        let mut branch_hooks = vec![];
        let mut seen = HashSet::new();

        for (name, hook) in hook_configs {
            if !seen.insert(name) {
                continue;
            }

            if let Some(ref name) = hook.name {
                try!(create_hook(name, hook.config.clone(), &mut hooks, &mut branch_hooks));
            }
        }

        Ok(Hooks {
            hooks: hooks,
            branch_hooks: branch_hooks,
        })
    }

    fn add_hooks<'a>(&'a self, conf: &mut GitHookConfiguration<'a>) {
        for hook in &self.hooks {
            conf.add_hook(hook.as_ref());
        }

        for hook in &self.branch_hooks {
            conf.add_branch_hook(hook.as_ref());
        }
    }
}

/// Representation of a stage action for a branch.
pub struct StageAction {
    stage: RefCell<stage::Stage>,
    /// The policy for updating branches on this stage.
    pub policy: StageUpdatePolicy,
    /// Access level requirement to be able to use the stage action for a branch.
    pub access_level: u64,
}

impl StageAction {
    /// A mutable reference to the stage workflow action.
    pub fn stage(&self) -> RefMut<stage::Stage> {
        self.stage.borrow_mut()
    }
}

/// Configuration for a branch within a project.
pub struct Branch {
    /// The name of the branch.
    pub name: String,

    hooks: Hooks,
    merge_hooks: Hooks,

    context: GitContext,
    submodules: SubmoduleMap,
    identity: Identity,

    stage: Option<StageAction>,
}

impl Branch {
    fn new(name: &str, hooks: Hooks, merge_hooks: Hooks, context: GitContext,
           submodules: SubmoduleMap, identity: Identity)
           -> Result<Self, Box<Error>>
    {
        Ok(Branch {
            name: name.to_string(),

            hooks: hooks,
            merge_hooks: merge_hooks,

            context: context,
            submodules: submodules,
            identity: identity,

            stage: None,
        })
    }

    /// Returns `true` if the branch has a staging branch.
    pub fn has_stage(&self) -> bool {
        self.stage.is_some()
    }

    fn add_stage(&mut self, stage_conf: &io::Stage, project: HostedProject) -> Result<&mut Self, stage::Error> {
        let branch = &stage_conf.integration_branch;
        let stager = try!(Stager::from_branch(&self.context, CommitId::new(&self.name),
                                              CommitId::new(&branch),
                                              self.identity.clone(), self.submodules.clone()));

        let mut stage = stage::Stage::new(stager, branch, project);

        if stage_conf.quiet {
            stage.quiet();
        }

        if stage_conf.keep_topics {
            stage.keep_topics();
        }

        self.stage = Some(StageAction {
                stage: RefCell::new(stage),
                policy: stage_conf.update_policy,
                access_level: stage_conf.required_access_level,
            });
        Ok(self)
    }

    fn git_hook_configuration<'a>(&'a self, hooks: &'a Hooks) -> GitHookConfiguration<'a> {
        let mut conf = GitHookConfiguration::new(None, &self.submodules);

        hooks.add_hooks(&mut conf);

        conf
    }

    /// Run hooks against a branch which is expected to be merged into this branch.
    pub fn check_branch(&self, topic: &CommitId, reason: &str, who: &Identity) -> Result<HookResult, HookError> {
        let refs = try!(list_refs(&self.context, reason, &self.name, topic.as_str()));
        let conf = self.git_hook_configuration(&self.hooks);

        run_hooks(&self.context, &conf, &refs, who)
    }

    /// Run hooks against a proposed merge commit into this branch.
    pub fn check_merge(&self, merge_commit: &CommitId, who: &Identity) -> Result<HookResult, HookError> {
        let conf = self.git_hook_configuration(&self.merge_hooks);

        run_hooks(&self.context, &conf, &[merge_commit.as_str()], who)
    }

    /// Get the stage action for the branch.
    pub fn stage(&self) -> Option<&StageAction> {
        self.stage.as_ref()
    }
}
