// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Command parsing and checking.
//!
//! Comments may contain commands for the robot to perform. This handles parsing out those commands
//! from a set of trailers.

use super::trailers::Trailer;
use super::super::super::config::Branch;

/// The various state command requests may be in.
pub enum CommandState {
    /// The command was requested and is allowed.
    Requested(Vec<String>),
    /// The command is not available for the branch.
    Unavailable,
    /// The command was not requested.
    Unrequested,
    /// The command was requested, but the user lacked permissions.
    Disallowed,
}

impl CommandState {
    /// Returns the arguments to the command.
    pub fn arguments(&self) -> &[String] {
        match *self {
            CommandState::Requested(ref args) => args,
            CommandState::Unavailable |
            CommandState::Unrequested |
            CommandState::Disallowed => &[],
        }
    }

    /// Returns `true` if the command was requested and allowed.
    pub fn requested(&self) -> bool {
        match *self {
            CommandState::Requested(_) => true,
            CommandState::Unavailable |
            CommandState::Unrequested |
            CommandState::Disallowed => false,
        }
    }

    /// Returns `true` if the command was requested but disallowed.
    pub fn disallowed(&self) -> bool {
        match *self {
            CommandState::Disallowed => true,
            CommandState::Requested(_) |
            CommandState::Unavailable |
            CommandState::Unrequested => false,
        }
    }
}

/// Structure for the state of requested commands.
pub struct Commands<'a> {
    /// The branch the commands apply to.
    branch: &'a Branch,

    /// The `stage` command.
    pub stage: CommandState,

    /// List of unrecognized commands.
    pub unrecognized: Vec<String>,
}

impl<'a> Commands<'a> {
    pub fn new(branch: &'a Branch) -> Self {
        Commands {
            branch: branch,

            stage: CommandState::Unrequested,

            unrecognized: vec![],
        }
    }

    /// The set of disallowed commands.
    pub fn disallowed_commands(&self) -> Vec<&'static str> {
        let mut disallowed = vec![];

        if self.stage.disallowed() {
            disallowed.push("stage");
        }

        disallowed
    }

    /// Checks whether the set of commands requested is consistent or not.
    ///
    /// Returns a list of messages about problems with the requested commands.
    pub fn consistency_messages(&self) -> Vec<&'static str> {
        vec![]
    }

    /// Parse commands from a set of trailers.
    pub fn add_from_trailers<'b>(&mut self, access: u64, trailers: &[Trailer<'b>]) -> &mut Self {
        for trailer in trailers.iter() {
            match trailer.token {
                "do" | "Do" => {
                    // TODO: handle quoting
                    let values = trailer.value
                        .split_whitespace()
                        .collect::<Vec<_>>();
                    let (command, args) = values
                        .split_first()
                        .unwrap(); // Trailers require a non-whitespace value.
                    match *command {
                        "stage" => {
                            self.stage = if let Some(stage_action) = self.branch.stage() {
                                    if access < stage_action.access_level {
                                        CommandState::Disallowed
                                    } else {
                                        CommandState::Requested(args.iter()
                                                                    .map(|a| a.to_string())
                                                                    .collect())
                                    }
                                } else {
                                    CommandState::Unavailable
                                };
                        },
                        unrecognized => {
                            self.unrecognized.push(unrecognized.to_string())
                        },
                    }
                },
                _ => {
                    // Ignore other trailers.
                    ()
                },
            }
        }

        self
    }
}
