// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Common data structures required for the workflow.
//!
//! This module contains structures necessary to carry out the actual actions wanted for the
//! requested branches.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate ghostflow;
use self::ghostflow::host::{Comment, Commit, Membership, MergeRequest, Repo, User};
use self::ghostflow::utils::TrailerRef;

extern crate git_workarea;
use self::git_workarea::CommitId;

/// Information required to handle a push event.
#[derive(Debug, Clone)]
pub struct PushInfo {
    /// The commit being pushed.
    pub commit: Commit,
    /// The user who pushed the commit.
    pub author: User,
    /// When the push occurred.
    pub date: DateTime<UTC>,
}

/// Information required to handle a merge request update.
#[derive(Debug, Clone)]
pub struct MergeRequestInfo {
    /// The merge request information.
    pub merge_request: MergeRequest,
    /// Indicates whether the MR was just merged or not.
    pub was_merged: bool,
    /// Whether the merge request is open or not.
    pub is_open: bool,
    /// When the merge request was last updated.
    pub date: DateTime<UTC>,
}

/// Information about backporting a merge request into multiple branches.
pub struct BackportInfo {
    /// The branch to backport into.
    branch: String,
    /// The commit to backport.
    commit: Option<CommitId>,
}

impl BackportInfo {
    /// Backport information for the main target of a merge request.
    pub fn main_target<B>(branch: B) -> Self
        where B: ToString,
    {
        BackportInfo {
            branch: branch.to_string(),
            commit: None,
        }
    }

    /// Parse backport information from a string description.
    ///
    /// The format is `branch[:commit]`. Without the `commit` part, the last commit of the merge
    /// request's source topic is implied.
    pub fn from_str<S>(value: S) -> Self
        where S: AsRef<str>,
    {
        let value = value.as_ref();
        let (branch, commit) = match value.bytes().position(|c| c == b':') {
            Some(loc) => {
                let (branch, commit) = value.split_at(loc);
                (branch, Some(commit))
            },
            None => (value, None),
        };

        BackportInfo {
            branch: branch.to_string(),
            commit: commit.map(CommitId::new),
        }
    }

    /// The commit to backport for the given merge request.
    pub fn commit_for_mr<'a>(&'a self, mr: &'a MergeRequest) -> &'a CommitId {
        self.commit.as_ref().unwrap_or(&mr.commit.id)
    }

    /// The name of the targeted backport branch.
    pub fn branch(&self) -> &str {
        &self.branch
    }

    /// The commit to backport.
    pub fn commit(&self) -> Option<&CommitId> {
        self.commit.as_ref()
    }
}

impl MergeRequestInfo {
    /// Returns `true` if it appears that the source branch has been deleted.
    pub fn is_source_branch_deleted(&self) -> bool {
        self.merge_request.commit.id.as_str().is_empty()
    }

    /// Backport information for the merge request.
    ///
    /// Backport information is stored in the description of the merge request using `Backport`
    /// trailers.
    pub fn backports(&self) -> Vec<BackportInfo> {
        let trailers = TrailerRef::extract(&self.merge_request.description);

        trailers.into_iter()
            .filter_map(|trailer| {
                if trailer.token == "Backport" {
                    Some(BackportInfo::from_str(trailer.value))
                } else {
                    None
                }
            })
            .collect()
    }
}

/// Information required to handle a note on a merge request.
#[derive(Debug, Clone)]
pub struct MergeRequestNoteInfo {
    /// The merge request.
    pub merge_request: MergeRequestInfo,
    /// The note.
    pub note: Comment,
}

/// Information required to handle the addition of a user to a project.
#[derive(Debug, Clone)]
pub struct MembershipAdditionInfo {
    /// The repository.
    pub repo: Repo,
    /// The membership.
    pub membership: Membership,
}

/// Information required to handle the removal of a user from a project.
#[derive(Debug, Clone)]
pub struct MembershipRemovalInfo {
    /// The repository.
    pub repo: Repo,
    /// The user.
    pub user: User,
}
