// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate itertools;
use self::itertools::Itertools;

extern crate serde;
use self::serde::{Deserialize, Deserializer};
use self::serde::de::Error as SerdeError;
use self::serde::de::Unexpected;

extern crate serde_json;
use self::serde_json::Value;

extern crate serde_yaml;

use config::checks;
use config::defaults;
use error::*;

use std::collections::hash_map::HashMap;
use std::fs::File;
use std::path::Path;

fn empty_object() -> Value {
    Value::Object(Default::default())
}

fn default_data_ref_namespace() -> String {
    "data".to_string()
}

#[derive(Deserialize, Debug, Clone)]
pub struct Data {
    #[serde(default="defaults::default_false")]
    /// Whether the keep data refs after handling data pushes.
    pub keep_refs: bool,

    #[serde(default="default_data_ref_namespace")]
    /// The namespace for data references.
    pub namespace: String,

    /// The destinations for the data.
    pub destinations: Vec<String>,
}

#[derive(Deserialize, Debug, Clone)]
/// Configuration for a check.
pub struct CheckConfig {
    #[serde(default)]
    /// The kind of check to perform.
    ///
    /// Skip to remove the check.
    pub kind: Option<String>,
    #[serde(default="empty_object")]
    /// Configuration object for the check.
    pub config: Value,
}

fn default_follow_ref_namespace() -> String {
    "follow".to_string()
}

#[derive(Deserialize, Debug)]
/// Configuration for the follow action for a branch.
pub struct Follow {
    #[serde(default="default_follow_ref_namespace")]
    /// The namespace for follow references.
    pub ref_namespace: String,
}

#[derive(Deserialize, Debug, Default, Clone, Copy)]
/// Policies for collecting information from a merge request before merging.
pub struct MergePolicy;

#[derive(Deserialize, Debug, Clone)]
/// Configuration for the merge action for a branch.
pub struct Merge {
    #[serde(default="defaults::default_false")]
    /// If true, only errors and essential comments will be made to the service.
    pub quiet: bool,
    /// The maximum number of commit summaries to show in the merge commit message.
    ///
    /// Set to `null` for `unlimited` and `0` for no logs.
    pub log_limit: Option<usize>,

    /// The access level required to be able to stage.
    pub required_access_level: u64,

    #[serde(default)]
    /// The policy to use for merging.
    pub policy: MergePolicy,

    #[serde(default)]
    /// Labels to add to issues which have been closed via a merged merge request.
    pub issue_labels: Vec<String>,
}

#[derive(Deserialize, Debug, Clone)]
pub struct Formatter {
    /// The "kind" of formatter.
    ///
    /// Used for attribute lookup.
    pub kind: String,
    /// The path to the formatter.
    pub formatter: String,

    #[serde(default)]
    /// Paths within the repository which contain configuration for the formatter.
    pub config_files: Vec<String>,
}

#[derive(Deserialize, Debug, Clone)]
/// Configuration for the stage action for a branch.
pub struct Reformat {
    /// The access level required to be able to stage.
    pub required_access_level: u64,

    /// The formatters to use for the action.
    pub formatters: Vec<Formatter>,
}

#[derive(Debug, Clone, Copy)]
/// Policies for updates to branches which have been merged into the stage.
pub enum StageUpdatePolicy {
    /// Updates should be ignored; the previous merge should be used instead.
    Ignore,
    /// The updated branch should be added to the stage.
    Restage,
    /// The branch should be removed from the stage when an update appears.
    Unstage,
}
enum_serialize!(StageUpdatePolicy -> "stage update policy",
    Ignore => "ignore",
    Restage => "restage",
    Unstage => "unstage",
);

impl Default for StageUpdatePolicy {
    fn default() -> Self {
        StageUpdatePolicy::Unstage
    }
}

#[derive(Deserialize, Debug, Clone, Copy)]
/// Configuration for the stage action for a branch.
pub struct Stage {
    #[serde(default="defaults::default_false")]
    /// If true, only errors and essential comments will be made to the service.
    pub quiet: bool,

    /// The access level required to be able to stage.
    pub required_access_level: u64,

    #[serde(default)]
    /// The update policy for merge requests which receive new code.
    pub update_policy: StageUpdatePolicy,
}

#[derive(Debug, Clone, Copy)]
/// Backends for performing tests for projects.
pub enum TestBackend {
    /// Job files will be created in a directory for test commands.
    Jobs,
    /// Refs will be pushed to the remote to indicate topics to test.
    Refs,
}
enum_serialize!(TestBackend -> "testing backend",
    Jobs => "jobs",
    Refs => "refs",
);

#[derive(Deserialize, Debug)]
pub struct TestJobsConfig {
    #[serde(default="defaults::default_false")]
    /// If true, only errors and essential comments will be made to the service.
    pub quiet: bool,

    /// The directory to drop job files into.
    pub queue: String,

    #[serde(default="defaults::default_true")]
    /// Whether to test updates to the branch itself or not.
    pub test_updates: bool,
}

fn default_test_ref_namespace() -> String {
    "test-topics".to_string()
}

#[derive(Deserialize, Debug)]
pub struct TestRefsConfig {
    #[serde(default="defaults::default_false")]
    /// If true, only errors and essential comments will be made to the service.
    pub quiet: bool,

    #[serde(default="default_test_ref_namespace")]
    /// The namespace for test references.
    pub namespace: String,
}

#[derive(Deserialize, Debug)]
/// Configuration for the test action for a branch.
pub struct Test {
    /// The access level required to be able to test.
    pub required_access_level: u64,

    /// The backend to use for testing topics for this branch.
    pub backend: TestBackend,

    #[serde(default="empty_object")]
    /// The configuration for the backend.
    pub config: Value,
}

#[derive(Deserialize, Debug)]
/// Configuration for a branch within a project.
pub struct Branch {
    #[serde(default)]
    /// The follow configuration.
    pub follow: Option<Follow>,
    #[serde(default)]
    /// The merge configuration.
    pub merge: Option<Merge>,
    #[serde(default)]
    /// The reformat configuration.
    pub reformat: Option<Reformat>,
    #[serde(default)]
    /// The stage configuration.
    pub stage: Option<Stage>,
    #[serde(default)]
    /// The test configuration.
    pub test: Option<Test>,

    #[serde(default)]
    /// Branch-specific checks.
    ///
    /// Checks which share the same name as the project override the project's check.
    pub checks: HashMap<String, CheckConfig>,
}

#[derive(Deserialize, Debug)]
/// Configuration for a project.
pub struct Project {
    #[serde(default)]
    /// Project-wide checks.
    pub checks: HashMap<String, CheckConfig>,
    #[serde(default)]
    /// The data configuration for the project.
    pub data: Option<Data>,

    /// The name to use when committing to this project.
    pub name: String,
    /// The email to use when committing to this project.
    pub email: String,

    /// The branches which should be watched.
    pub branches: HashMap<String, Branch>,

    #[serde(default)]
    /// The submodule git directories for the project.
    pub submodules: HashMap<String, String>,

    #[serde(default)]
    /// Project maintainers.
    pub maintainers: Vec<String>,
}

#[derive(Deserialize, Debug)]
/// Configuration for a host.
pub struct Host {
    /// The API the host uses for communication.
    pub host_api: String,
    /// The URL to the host.
    pub host_url: Option<String>,
    /// The list of maintainers on the host to notify when errors occur.
    pub maintainers: Vec<String>,
    /// The path to a JSON file containing private information to contact this host.
    pub secrets_path: String,
    /// The webhook URL to register for new projects.
    pub webhook_url: Option<String>,

    /// The projects to handle on this host.
    ///
    /// All events for other projects are ignored.
    pub projects: HashMap<String, Project>,
}

impl Host {
    /// The secrets object.
    pub fn secrets(&self) -> Result<Value> {
        let fin = File::open(&self.secrets_path)
            .chain_err(|| format!("failed to open secrets file '{}'",
                                  self.secrets_path))?;
        Ok(serde_json::from_reader(fin)
           .chain_err(|| format!("failed to read secrets from '{}'",
                                 self.secrets_path))?)
    }
}

#[derive(Deserialize, Debug)]
/// Main configuration object for the robot.
pub struct Config {
    /// A scratch space directory for the host.
    pub workdir: String,

    /// Host configuration.
    pub hosts: HashMap<String, Host>,

    archive: Option<String>,
    queue: String,
}

impl Config {
    /// Read the configuration from a path.
    pub fn from_path<P: AsRef<Path>>(path: P) -> Result<Self> {
        let fin = File::open(path.as_ref())
            .chain_err(|| format!("failed to open config file '{}'",
                                  path.as_ref().display()))?;
        Ok(serde_yaml::from_reader(fin)
           .chain_err(|| format!("failed to read config from '{}'",
                                 path.as_ref().display()))?)
    }

    /// The directory to read for input data.
    pub fn queue_dir(&self) -> &Path {
        Path::new(&self.queue)
    }

    /// The directory to place processed job files.
    pub fn archive_dir(&self) -> &Path {
        Path::new(self.archive.as_ref().unwrap_or(&self.queue))
    }

    /// Verify all of the check configuration blocks in the configuration file.
    pub fn verify_all_check_configs(self) -> Result<()> {
        // Check the checks in each host.
        self.hosts
            .into_iter()
            .map(|(_, host)| {
                // And in each project.
                host.projects
                    .into_iter()
                    .map(|(_, project)| {
                        // First get the project-wide checks.
                        let project_checks = project.checks
                            .values()
                            .cloned()
                            .collect_vec();

                        project_checks.into_iter()
                            // And merge them with the branch-specific checks.
                            .chain(project.branches
                                .into_iter()
                                .map(|(_, branch)| {
                                    branch.checks
                                        .into_iter()
                                        .map(|(_, check)| {
                                            check
                                        })
                                })
                                .flatten())
                    })
                    .flatten()
            })
            .flatten()
            // Split the check configuration into an owned kind and configuration block.
            .map(|check_config| {
                (check_config.kind.clone(), check_config.config)
            })
            // Parse each check configuration section out.
            .map(|(check_kind, config): (Option<String>, Value)| {
                check_kind.map(|kind| {
                    let mut checks = Vec::new();
                    let mut branch_check = Vec::new();
                    checks::create_check(&kind, config, &mut checks, &mut branch_check)
                })
            })
            // Skip any branch checks which are deleting a project-level check.
            .filter_map(|x| x)
            // Collect them up as results.
            .collect::<Result<Vec<_>>>()
            // And drop the actual check results.
            .map(|_| ())
    }
}
