/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */

#include "cmDepfileFormat.h"

#include <cassert>
#include <cctype>
#include "cm_jsoncpp_reader.h"
#include "cm_jsoncpp_writer.h"
#include "cm_utf8.h"
#include "cmsys/FStream.hxx"
#include <set>
#include <map>

#include "cmAlgorithms.h"
#include "cmGeneratedFileStream.h"
#include "cmSystemTools.h"

static int
IntFromHexDigit(char c) {
  if (c >= '0' && c <= '9') {
    return c - '0';
  }
  if (c >= 'a' && c <= 'f') {
    return c - 'a' + 0xA;
  }
  if (c >= 'A' && c <= 'F') {
    return c - 'A' + 0xA;
  }
  assert(false);
  return -1;
}

static bool
ParseFilename(Json::Value const& val, std::string& result)
{
  if (val.isObject()) {
    Json::Value const& format = val["format"];
    if (format == "raw8") {
      Json::Value const& data = val["data"];
      for (auto const& byte : data) {
        result.push_back(byte.asUInt());
      }
    } else if (format == "raw16") {
      // TODO: Implement.
      return false;
    } else {
      return false;
    }
  } else {
    result = val.asString();

    size_t pos = 0;
    while ((pos = result.find('%', pos)) != std::string::npos) {
      // Invalid escape.
      if (pos + 2 >= result.size()) {
        return false;
      }

      // Invalid hex.
      char high_nibble_hex = result[pos + 1];
      char low_nibble_hex = result[pos + 2];
      if (!std::isxdigit(high_nibble_hex) || !std::isxdigit(low_nibble_hex)) {
        return false;
      }

      int high_nibble = IntFromHexDigit(high_nibble_hex);
      int low_nibble = IntFromHexDigit(low_nibble_hex);

      // Embedded NUL.
      if (!high_nibble && !low_nibble) {
        return false;
      }

      result.replace(pos, 3, 1, static_cast<char>((high_nibble << 4)+ low_nibble));
      pos += 1;
    }

  }

  return true;
}

static Json::Value
EncodeFilename(std::string const& path)
{
  if (cm_utf8_is_valid(path.c_str())) {
    std::string result = path;
    size_t pos = 0;

    while ((pos = result.find('%', pos)) != std::string::npos) {
      result.replace(pos, 1, "%25"); // % is 0x25 in ASCII
      pos += 3;
    }

    return result;
  } else {
    Json::Value filepath = Json::objectValue;
    filepath["format"] = "raw8";
    Json::Value& data = filepath["data"] = Json::arrayValue;

    for (char c : path) {
      data.append(static_cast<unsigned int>(c));
    }

    return filepath;
  }
}

#define PARSE_FILENAME(val, res)                                      \
  do {                                                                \
    if (!ParseFilename(val, res)) {                                   \
      cmSystemTools::Error("-E cmake_ninja_depends failed to parse ", \
                            arg_pp.c_str(),                           \
                            ": invalid filename");                    \
      return false;                                                   \
    }                                                                 \
  } while (0)

bool cmDepfileFormat::ParseTrtbd(std::string const& arg_pp, cmSourceInfo* info)
{
  Json::Value ppio;
  Json::Value const& ppi = ppio;
  cmsys::ifstream ppf(arg_pp.c_str(), std::ios::in | std::ios::binary);
  {
    Json::Reader reader;
    if (!reader.parse(ppf, ppio, false)) {
      cmSystemTools::Error("-E cmake_ninja_depends failed to parse ",
                            arg_pp.c_str(),
                            reader.getFormattedErrorMessages().c_str());
      return false;
    }
  }

  Json::Value const& version = ppi["version"];
  if (version.asUInt() != 0) {
    cmSystemTools::Error("-E cmake_ninja_depends failed to parse ",
                         arg_pp.c_str(),
                         ": version ", version.asString().c_str());
    return false;
  }

  Json::Value const& sources = ppi["sources"];
  if (sources.isArray()) {
    if (sources.size() != 1) {
      cmSystemTools::Error("-E cmake_ninja_depends failed to parse ",
                           arg_pp.c_str(),
                           ": expected 1 source entry");
      return false;
    }

    for (auto const& source : sources) {
      Json::Value const& depends = source["depends"];
      if (depends.isArray()) {
        std::string depend_filename;
        for (auto const& depend : depends) {
          PARSE_FILENAME(depend, depend_filename);
          info->Includes.push_back(depend_filename);
        }
      }

      if (source.isMember("future-compile")) {
        Json::Value const& future_compile = source["future-compile"];

        if (future_compile.isMember("outputs")) {
          Json::Value const& outputs = future_compile["outputs"];
          if (outputs.isArray()) {
            if (outputs.size() < 1) {
              cmSystemTools::Error("-E cmake_ninja_depends failed to parse ",
                                   arg_pp.c_str(),
                                   ": expected at least one 1 output");
              return false;
            }

            PARSE_FILENAME(outputs[0], info->PrimaryOutput);
          }
        }

        if (future_compile.isMember("provides")) {
          Json::Value const& provides = future_compile["provides"];
          if (provides.isArray()) {
            for (auto const& provide : provides) {
              cmSourceReqInfo provide_info;

              Json::Value const& logical = provide["logical"];
              PARSE_FILENAME(logical, provide_info.Logical);

              if (provide.isMember("filepath")) {
                Json::Value const& filepath = provide["filepath"];
                PARSE_FILENAME(filepath, provide_info.Filepath);
              } else {
                provide_info.Filepath = provide_info.Logical + ".mod";
              }

              info->Provides.push_back(provide_info);
            }
          }
        }

        if (future_compile.isMember("requires")) {
          Json::Value const& requires = future_compile["requires"];
          if (requires.isArray()) {
            for (auto const& require : requires) {
              cmSourceReqInfo require_info;

              Json::Value const& logical = require["logical"];
              PARSE_FILENAME(logical, require_info.Logical);

              if (require.isMember("filepath")) {
                Json::Value const& filepath = require["filepath"];
                PARSE_FILENAME(filepath, require_info.Filepath);
              }

              info->Requires.push_back(require_info);
            }
          }
        }
      }
    }
  }

  return true;
}

bool cmDepfileFormat::WriteTrtbd(
    std::string const& path, std::string const& input,
    cmSourceInfo const& info)
{
  Json::Value ddi(Json::objectValue);
  ddi["version"] = 0;
  ddi["revision"] = 0;
  Json::Value& sources = ddi["sources"] = Json::arrayValue;

  Json::Value source(Json::objectValue);
  source["input"] = input;

  Json::Value& depends = source["depends"] = Json::arrayValue;
  for (auto const& include : info.Includes) {
    depends.append(EncodeFilename(include));
  }

  Json::Value& future_compile = source["future-compile"] = Json::objectValue;

  Json::Value& outputs = future_compile["outputs"] = Json::arrayValue;
  outputs.append(EncodeFilename(info.PrimaryOutput));

  Json::Value& provides = future_compile["provides"] = Json::arrayValue;
  for (auto const& provide : info.Provides) {
    Json::Value provide_obj(Json::objectValue);
    auto const encoded = EncodeFilename(provide.Logical);
    provide_obj["logical"] = encoded;
    if (provide.Filepath.empty()) {
      provide_obj["filepath"] = encoded;
    } else {
      provide_obj["filepath"] = EncodeFilename(provide.Filepath);
    }

    provides.append(provide_obj);
  }

  Json::Value& requires = future_compile["requires"] = Json::arrayValue;
  for (auto const& require : info.Requires) {
    Json::Value require_obj(Json::objectValue);
    auto const encoded = EncodeFilename(require.Logical);
    require_obj["logical"] = encoded;
    if (require.Filepath.empty()) {
      require_obj["filepath"] = encoded;
    } else {
      require_obj["filepath"] = EncodeFilename(require.Filepath);
    }

    requires.append(require_obj);
  }

  sources.append(source);

  cmGeneratedFileStream ddif(path);
  ddif << ddi;

  return !!ddif;
}
