/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmOptionCommand.h"

#include <string>

#include <cm/optional>
#include <cmext/string_view>

#include "cmArgumentParser.h"
#include "cmArgumentParserTypes.h"
#include "cmExecutionStatus.h"
#include "cmMakefile.h"
#include "cmMessageType.h"
#include "cmPolicies.h"
#include "cmRange.h"
#include "cmState.h"
#include "cmStateSnapshot.h"
#include "cmStateTypes.h"
#include "cmStringAlgorithms.h"
#include "cmValue.h"

namespace {
bool cmOptionFullCommand(std::vector<std::string> const& args,
                         cmExecutionStatus& status)
{
  std::string const& optionName = args[0];
  std::string const& docstring = args[1];
  std::string const& defaultValue = args[2];

  struct Arguments
  {
    cm::optional<std::string> Type;
    cm::optional<ArgumentParser::MaybeEmpty<std::vector<std::string>>>
      ValidValues;
    bool Advanced = false;
    bool Default = false;
  };

  static auto const parser = cmArgumentParser<Arguments>{}
                               .Bind("TYPE"_s, &Arguments::Type)
                               .Bind("VALID_VALUES"_s, &Arguments::ValidValues)
                               .Bind("ADVANCED"_s, &Arguments::Advanced)
                               .Bind("DEFAULT"_s, &Arguments::Default);

  std::vector<std::string> unparsed;
  Arguments const arguments =
    parser.Parse(cmMakeRange(args).advance(3), &unparsed);
  if (!unparsed.empty()) {
    status.SetError(
      cmStrCat("called with unrecognized arguments: ", cmJoin(unparsed, " ")));
    return false;
  }

  cmStateEnums::CacheEntryType type = cmStateEnums::CacheEntryType::BOOL;
  if (arguments.Type &&
      !cmState::StringToCacheEntryType(*arguments.Type, type)) {
    status.SetError(
      cmStrCat("called with invalid 'TYPE' value: '", *arguments.Type, '\''));
    return false;
  }
  if (type == cmStateEnums::CacheEntryType::UNINITIALIZED) {
    status.SetError("may not be specified with the 'UNINITIALIZED' type");
    return false;
  }
  if (arguments.ValidValues && type != cmStateEnums::CacheEntryType::STRING) {
    status.SetError(
      "only supports type 'STRING' when 'VALID_VALUES' is specified");
    return false;
  }

  // Do nothing if a local definition exists.
  if (status.GetMakefile().GetStateSnapshot().GetDefinition(optionName)) {
    return true;
  }

  cmState* state = status.GetMakefile().GetState();
  cmValue existingValue = state->GetCacheEntryValue(optionName);
  if (existingValue) {
    if (arguments.ValidValues) {
      state->SetCacheEntryProperty(optionName, "STRINGS",
                                   cmJoin(*arguments.ValidValues, ";"));
    } else {
      state->RemoveCacheEntryProperty(optionName, "STRINGS");
    }
    if (arguments.Advanced) {
      state->SetCacheEntryBoolProperty(optionName, "ADVANCED", true);
    } else {
      state->RemoveCacheEntryProperty(optionName, "ADVANCED");
    }
    if (arguments.Default) {
      state->SetCacheEntryProperty(optionName, "DEFAULT", defaultValue);
    } else {
      state->RemoveCacheEntryProperty(optionName, "DEFAULT");
    }
    if (state->GetCacheEntryType(optionName) != cmStateEnums::UNINITIALIZED) {
      state->SetCacheEntryProperty(optionName, "HELPSTRING", docstring);
    }
    state->SetCacheEntryProperty(optionName, "TYPE",
                                 cmState::CacheEntryTypeToString(type));
    return true;
  }

  status.GetMakefile().AddCacheDefinition(optionName, defaultValue, docstring,
                                          type);
  if (arguments.ValidValues) {
    state->SetCacheEntryProperty(optionName, "STRINGS",
                                 cmJoin(*arguments.ValidValues, ";"));
  }
  if (arguments.Advanced) {
    state->SetCacheEntryBoolProperty(optionName, "ADVANCED", true);
  }
  if (arguments.Default) {
    state->SetCacheEntryProperty(optionName, "DEFAULT", defaultValue);
    state->SetCacheEntryBoolProperty(optionName, "DEFAULTED", true);
  }

  return true;
}

bool cmOptionBoolCommand(std::vector<std::string> const& args,
                         cmExecutionStatus& status)
{
  // Determine the state of the option policy
  bool checkAndWarn = false;
  {
    auto policyStatus =
      status.GetMakefile().GetPolicyStatus(cmPolicies::CMP0077);
    const auto& existsBeforeSet =
      status.GetMakefile().GetStateSnapshot().GetDefinition(args[0]);
    switch (policyStatus) {
      case cmPolicies::WARN:
        checkAndWarn = (existsBeforeSet != nullptr);
        CM_FALLTHROUGH;
      case cmPolicies::OLD:
        // OLD behavior does not warn.
        break;
      case cmPolicies::REQUIRED_ALWAYS:
      case cmPolicies::REQUIRED_IF_USED:
      case cmPolicies::NEW: {
        // See if a local variable with this name already exists.
        // If so we ignore the option command.
        if (existsBeforeSet) {
          return true;
        }
      } break;
    }
  }

  // See if a cache variable with this name already exists
  // If so just make sure the doc state is correct
  cmState* state = status.GetMakefile().GetState();
  cmValue existingValue = state->GetCacheEntryValue(args[0]);
  if (existingValue &&
      (state->GetCacheEntryType(args[0]) != cmStateEnums::UNINITIALIZED)) {
    state->SetCacheEntryProperty(args[0], "HELPSTRING", args[1]);
    return true;
  }

  // Nothing in the cache so add it
  std::string initialValue = existingValue ? *existingValue : "Off";
  if (args.size() == 3) {
    initialValue = args[2];
  }
  bool init = cmIsOn(initialValue);
  status.GetMakefile().AddCacheDefinition(args[0], init ? "ON" : "OFF",
                                          args[1], cmStateEnums::BOOL);

  if (status.GetMakefile().GetPolicyStatus(cmPolicies::CMP0077) !=
        cmPolicies::NEW &&
      status.GetMakefile().GetPolicyStatus(cmPolicies::CMP0126) ==
        cmPolicies::NEW) {
    // if there was a definition then remove it
    status.GetMakefile().GetStateSnapshot().RemoveDefinition(args[0]);
  }

  if (checkAndWarn) {
    const auto& existsAfterSet =
      status.GetMakefile().GetStateSnapshot().GetDefinition(args[0]);
    if (!existsAfterSet) {
      status.GetMakefile().IssueMessage(
        MessageType::AUTHOR_WARNING,
        cmStrCat(cmPolicies::GetPolicyWarning(cmPolicies::CMP0077),
                 "\n"
                 "For compatibility with older versions of CMake, option "
                 "is clearing the normal variable '",
                 args[0], "'."));
    }
  }
  return true;
}
}

// cmOptionCommand
bool cmOptionCommand(std::vector<std::string> const& args,
                     cmExecutionStatus& status)
{
  const bool argError = args.size() < 2;
  if (argError) {
    std::string m = cmStrCat("called with incorrect number of arguments: ",
                             cmJoin(args, " "));
    status.SetError(m);
    return false;
  }

  if (args.size() > 3) {
    return cmOptionFullCommand(args, status);
  }

  return cmOptionBoolCommand(args, status);
}
