//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "smtk/session/aeva/operators/Export.h"

#include "smtk/session/aeva/Export_xml.h"
#include "smtk/session/aeva/Resource.h"
#include "smtk/session/aeva/Session.h"

#include "smtk/attribute/Resource.h"
#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/ComponentItem.h"
#include "smtk/attribute/FileItem.h"
#include "smtk/attribute/IntItem.h"
#include "smtk/attribute/ResourceItem.h"
#include "smtk/attribute/StringItem.h"

#include "smtk/model/Group.h"
#include "smtk/model/Model.h"
#include "smtk/model/Volume.h"
#include "smtk/model/Face.h"

#include "smtk/operation/MarkGeometry.h"

#include "smtk/resource/Manager.h"

#include "smtk/common/Paths.h"

#include "vtkImageData.h"
#include "vtkXMLImageDataWriter.h"

using ResourceArray = std::vector<smtk::session::aeva::Resource::Ptr>;

namespace smtk
{
namespace session
{
namespace aeva
{

Export::Result Export::operateInternal()
{
  smtk::attribute::FileItem::Ptr filenameItem = this->parameters()->findFile("filename");
  smtk::attribute::StringItem::Ptr filetypeItem = this->parameters()->findString("filetype");

  std::string filename = filenameItem->value();

  // Infer file type from name
  std::string ext = smtk::common::Paths::extension(filename);
  // Downcase the extension
  std::transform(ext.begin(), ext.end(), ext.begin(), ::tolower);
  if (ext == ".med")
    return this->exportMedMesh();

  // Internal storage is a VTK image, default to that:
  return this->exportVTKImage();
}

Export::Result Export::exportVTKImage()
{
  auto inputItem = this->parameters()->associations();
  auto inputValue = inputItem->valueAs<smtk::model::Entity>(0);
  smtk::session::aeva::Resource::Ptr resource;
  if (inputValue)
  {
    resource =
      std::static_pointer_cast<smtk::session::aeva::Resource>(inputValue->resource());
  }
  if (!resource)
  {
    smtkErrorMacro(this->log(), "No resources to export.");
    return this->createResult(smtk::operation::Operation::Outcome::FAILED);
  }

  auto& session = resource->session();
  auto image = vtkImageData::SafeDownCast(session->findStorage(inputValue->id()));
  if (!image)
  {
    smtkErrorMacro(this->log(), "Cannot retrieve image from model.");
    return this->createResult(smtk::operation::Operation::Outcome::FAILED);
  }

  vtkNew<vtkXMLImageDataWriter> writer;
  // force file extension to match what we're writing.
  std::string filename = smtk::common::Paths::replaceExtension(
    this->parameters()->findFile("filename")->value(), ".vti");

  writer->SetFileName(filename.c_str());
  writer->SetInputData(image);
  writer->Write();

  return this->createResult(Export::Outcome::SUCCEEDED);
}

Export::Result Export::exportMedMesh()
{
  smtkErrorMacro(log(), "Med files not yet supported.");
  return this->createResult(smtk::operation::Operation::Outcome::FAILED);
}

const char* Export::xmlDescription() const
{
  return Export_xml;
}

bool exportResource(const smtk::resource::ResourcePtr& resource)
{
  Export::Ptr exportResource = Export::create();
  exportResource->parameters()->findResource("resource")->setValue(resource);
  Export::Result result = exportResource->operate();
  return (result->findInt("outcome")->value() == static_cast<int>(Export::Outcome::SUCCEEDED));
}

} // namespace aeva
} //namespace session
} // namespace smtk
