/*=========================================================================

  Program:   ParaView
  Module:    vtkPVCoreApplication.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPVCoreApplication.h"

#include "vtkAPVVersion.h" // for APV_VERSION_*
#include "vtkCommand.h"
#include "vtkDummyController.h"
#include "vtkLogger.h"
#include "vtkObjectFactory.h"
#include "vtkPVCoreApplicationOptions.h"
#include "vtkPVServerApplication.h"
#include "vtkReflectionInitializer.h"
#include "vtkRemotingCoreUtilities.h"
#include "vtkRemotingServerManagerCoreLogVerbosity.h"
#include "vtkResourceFileLocator.h"
#include "vtkServicesEngine.h"
#include "vtkSmartPointer.h"
#include "vtkVersion.h"

#include <string>
#include <thread>
#include <vtksys/RegularExpression.hxx>
#include <vtksys/SystemTools.hxx>

// clang-format off
#include <vtk_fmt.h> // needed for `fmt`
#include VTK_FMT(fmt/core.h)
// clang-format on

#define APV_SOURCE_VERSION "paraview version " APV_VERSION_FULL

namespace
{
std::string GetSharedResourcesDirectory(const std::string& vtkNotUsed(appDir))
{
  // Look for where the function "GetVTKVersion." lives.
  auto vtk_libs = vtkGetLibraryPathForSymbol(GetVTKVersion);

  // Where docs might be in relation to the executable
  std::vector<std::string> prefixes = {
#if defined(_WIN32) || defined(__APPLE__)
    ".."
#else
    "share/paraview-" APV_VERSION
#endif
  };

  // Search for the docs directory
  vtkNew<vtkResourceFileLocator> locator;
  auto resource_dir = locator->Locate(vtk_libs, prefixes, "doc");
  if (!resource_dir.empty())
  {
    resource_dir = vtksys::SystemTools::CollapseFullPath(resource_dir);
  }

  return resource_dir;
}

}

class vtkPVCoreApplication::vtkInternals
{
public:
  const std::thread::id OwnerTID{ std::this_thread::get_id() };
  vtkSmartPointer<vtkServicesEngine> Engine;
  vtkSmartPointer<vtkMultiProcessController> Controller;
  vtkSmartPointer<vtkPVCoreApplicationOptions> Options;
  vtkPVCoreApplication::ApplicationTypes Type;
  std::string ApplicationName;
  std::string ApplicationDirPath;
  std::string ApplicationFilePath;
  std::string SharedResourcesDirectory;
  std::string ExampleFilesDirectory;
  std::string DocDirectory;

  bool Initialized{ false };
  int Rank{ 0 };
  int NumberOfRanks{ 1 };
  int ExitCode{ EXIT_SUCCESS };
  bool ExitCodeSet{ false };
  bool SymmetricMPIMode{ false };

  void SetupApplicationPaths(const std::string argv0)
  {
    // setup app dir/file paths.
    if (argv0.empty())
    {
      this->ApplicationDirPath = vtksys::SystemTools::GetCurrentWorkingDirectory();
      this->ApplicationFilePath = this->ApplicationDirPath + "/unknown_executable";
    }
    else
    {
      std::string errMsg;
      if (!vtksys::SystemTools::FindProgramPath(argv0.c_str(), this->ApplicationFilePath, errMsg))
      {
        // if FindProgramPath fails. We really don't have much of an alternative
        // here. Python module importing is going to fail.
        this->ApplicationFilePath = vtksys::SystemTools::CollapseFullPath(argv0);
      }
      this->ApplicationDirPath = vtksys::SystemTools::GetFilenamePath(this->ApplicationFilePath);
    }

    this->ApplicationName = vtksys::SystemTools::GetFilenameName(this->ApplicationFilePath);
    this->ApplicationName =
      vtksys::SystemTools::GetFilenameWithoutLastExtension(this->ApplicationName);

    this->SharedResourcesDirectory = ::GetSharedResourcesDirectory(this->ApplicationDirPath);
    this->ExampleFilesDirectory = this->SharedResourcesDirectory = "/examples";
    this->DocDirectory = this->SharedResourcesDirectory = "/doc";
  }

  std::string GetRankAnnotatedName(const std::string& name) const
  {
    return name.empty() || this->NumberOfRanks <= 1 ? name
                                                    : name + "." + std::to_string(this->Rank);
  }
};

vtkPVCoreApplication* vtkPVCoreApplication::Singleton;
//----------------------------------------------------------------------------
vtkPVCoreApplication::vtkPVCoreApplication(vtkPVCoreApplication::ApplicationTypes type)
  : Internals(new vtkPVCoreApplication::vtkInternals())
{
  auto& internals = (*this->Internals);
  internals.Type = type;
}

//----------------------------------------------------------------------------
vtkPVCoreApplication::~vtkPVCoreApplication()
{
  vtkLogIfF(ERROR, vtkPVCoreApplication::Singleton == this,
    "vtkPVCoreApplication singleton is non-null. Did you forget to call 'Finalize'?");
}

//----------------------------------------------------------------------------
vtkPVCoreApplication* vtkPVCoreApplication::GetInstance()
{
  return vtkPVCoreApplication::Singleton;
}

//----------------------------------------------------------------------------
vtkPVCoreApplication::ApplicationTypes vtkPVCoreApplication::GetApplicationType() const
{
  const auto& internals = (*this->Internals);
  return internals.Type;
}

//----------------------------------------------------------------------------
vtkMultiProcessController* vtkPVCoreApplication::GetController() const
{
  const auto& internals = (*this->Internals);
  vtkRemotingCoreUtilities::EnsureThread(internals.OwnerTID);
  return internals.Controller;
}

//----------------------------------------------------------------------------
int vtkPVCoreApplication::GetNumberOfRanks() const
{
  const auto& internals = (*this->Internals);
  return internals.NumberOfRanks;
}

//----------------------------------------------------------------------------
int vtkPVCoreApplication::GetRank() const
{
  const auto& internals = (*this->Internals);
  return internals.Rank;
}

//----------------------------------------------------------------------------
bool vtkPVCoreApplication::GetSymmetricMPIMode() const
{
  const auto& internals = (*this->Internals);
  return internals.SymmetricMPIMode;
}

//----------------------------------------------------------------------------
const std::string& vtkPVCoreApplication::GetApplicationName() const
{
  const auto& internals = (*this->Internals);
  return internals.ApplicationName;
}

//----------------------------------------------------------------------------
const std::string& vtkPVCoreApplication::GetApplicationDirPath() const
{
  const auto& internals = (*this->Internals);
  return internals.ApplicationDirPath;
}

//----------------------------------------------------------------------------
const std::string& vtkPVCoreApplication::GetApplicationFilePath() const
{
  const auto& internals = (*this->Internals);
  return internals.ApplicationFilePath;
}

//----------------------------------------------------------------------------
bool vtkPVCoreApplication::IsInitialized() const
{
  const auto& internals = (*this->Internals);
  return internals.Initialized;
}

//----------------------------------------------------------------------------
bool vtkPVCoreApplication::Initialize(const std::string& executable,
  rxcpp::observe_on_one_worker scheduler, vtkMultiProcessController* globalController)
{
  auto& internals = (*this->Internals);
  vtkRemotingCoreUtilities::EnsureThread(internals.OwnerTID);
  vtkLogger::SetThreadName(internals.ApplicationName);

  internals.ExitCodeSet = false;

  static bool reflection_initialized = false;
  if (!reflection_initialized)
  {
    vtkLogF(TRACE, "Initialize VTK-object reflection");
    auto* reflection = vtkReflectionInitializer::New();
    if (!reflection)
    {
      vtkLogF(ERROR, "Reflection has not been initialized correctly!");
      abort();
    }
    reflection->Initialize();
    reflection->Delete();
    reflection_initialized = true;
  }

  if (vtkPVCoreApplication::Singleton)
  {
    throw std::runtime_error("Cannot initialize multiple ParaView-based applications.");
  }

  internals.SetupApplicationPaths(executable);
  auto* options = this->GetOptions(); // ensure options object is setup.
  if (!options->GetReadOnly())
  {
    options->SetReadOnly(true);
  }

  if (globalController)
  {
    internals.Controller = globalController;
  }
  else
  {
    internals.Controller = vtk::TakeSmartPointer(vtkDummyController::New());
    internals.Controller->Initialize(nullptr, nullptr);
  }

  internals.Rank = internals.Controller->GetLocalProcessId();
  internals.NumberOfRanks = internals.Controller->GetNumberOfProcesses();
  vtkLogger::SetThreadName(internals.GetRankAnnotatedName(internals.ApplicationName));
  internals.SymmetricMPIMode = this->GetOptions()->GetSymmetricMPIMode();

  if (options->GetLogStdErrVerbosity() != vtkLogger::VERBOSITY_INVALID)
  {
    vtkLogger::SetStderrVerbosity(options->GetLogStdErrVerbosity());
  }

  for (const auto& log_pair : options->GetLogFiles())
  {
    vtkLogger::LogToFile(
      internals.GetRankAnnotatedName(log_pair.first).c_str(), vtkLogger::TRUNCATE, log_pair.second);
  }

  // TODO: vtkProcessModule::InitializePythonEnvironment()

  // Startup services engine.
  internals.Engine = vtk::TakeSmartPointer(vtkServicesEngine::New());
  // TODO: should we pass a duplicated controller here?
  internals.Engine->SetController(internals.Controller);
  internals.Engine->SetSymmetricMPIMode(internals.SymmetricMPIMode);
  internals.Engine->Initialize(this->GetEngineURL(), scheduler);
  vtkPVCoreApplication::Singleton = this;

  if (!this->InitializeInternal())
  {
    vtkPVCoreApplication::Singleton = nullptr;
    return false;
  }

  vtkLogF(TRACE, "application initialized");
  internals.Initialized = true;
  if (internals.Rank == 0 && internals.Type == vtkPVCoreApplication::SERVER &&
    !internals.ExitCodeSet)
  {
    std::string url = vtkPVCoreApplication::GetCSURL(internals.Engine->GetUrl());
    cout << "Connection URL: " << url << endl;
    cout << "Accepting connection(s): " << url.substr(url.find_first_of('/') + 2)
         << endl; // drop 'protocol://'
  }
  return true;
}

//----------------------------------------------------------------------------
void vtkPVCoreApplication::Finalize()
{
  auto& internals = (*this->Internals);
  vtkRemotingCoreUtilities::EnsureThread(internals.OwnerTID);
  if (vtkPVCoreApplication::Singleton == this)
  {
    this->InvokeEvent(vtkCommand::ExitEvent);
    this->FinalizeInternal();
    internals.Engine->Finalize();
    vtkPVCoreApplication::Singleton = nullptr;
  }
}

//----------------------------------------------------------------------------
vtkServicesEngine* vtkPVCoreApplication::GetServicesEngine() const
{
  const auto& internals = (*this->Internals);
  return internals.Engine;
}

//----------------------------------------------------------------------------
vtkPVCoreApplicationOptions* vtkPVCoreApplication::GetOptions() const
{
  auto& internals = (*this->Internals);
  if (internals.Options == nullptr)
  {
    vtkRemotingCoreUtilities::EnsureThread(internals.OwnerTID);
    internals.Options = this->CreateOptions();
  }
  return internals.Options;
}

//----------------------------------------------------------------------------
vtkSmartPointer<vtkPVCoreApplicationOptions> vtkPVCoreApplication::CreateOptions() const
{
  return vtk::TakeSmartPointer(vtkPVCoreApplicationOptions::New());
}

//----------------------------------------------------------------------------
std::string vtkPVCoreApplication::GetEngineURL() const
{
  if (this->GetServicesEngine()->IsA("vtkAsioServicesEngine") &&
    this->GetApplicationType() != ApplicationTypes::SERVER)
  {
    return "builtin+" + this->GetServicesEngine()->GetProtocol() + "://";
  }
  return this->GetServicesEngine()->GetProtocol() + "://";
}

//-----------------------------------------------------------------------------
void vtkPVCoreApplication::Exit(int exitCode)
{
  auto& internals = (*this->Internals);
  vtkRemotingCoreUtilities::EnsureThread(internals.OwnerTID);
  internals.ExitCodeSet = true;
  internals.ExitCode = exitCode;
}

//-----------------------------------------------------------------------------
int vtkPVCoreApplication::GetExitCode() const
{
  const auto& internals = (*this->Internals);
  vtkRemotingCoreUtilities::EnsureThread(internals.OwnerTID);
  return internals.ExitCodeSet ? internals.ExitCode : EXIT_SUCCESS;
}

//-----------------------------------------------------------------------------
bool vtkPVCoreApplication::WaitForExit(
  const rxcpp::schedulers::run_loop& rlp, int timeout /*=0*/) const
{
  return this->WaitForExit(rlp, std::chrono::milliseconds(timeout));
}

//-----------------------------------------------------------------------------
bool vtkPVCoreApplication::WaitForExit(
  const rxcpp::schedulers::run_loop& rlp, const std::chrono::milliseconds& duration) const
{
  const auto& internals = (*this->Internals);
  vtkRemotingCoreUtilities::EnsureThread(internals.OwnerTID);

  auto* controller = this->GetController();

  // On satellite ranks we need to able to wait on: -  ProcessRMIs for any
  // messages from rank 0 -  rlp.empty() for running any callbacks but also in
  // case a "run_on_main_thread" request comes or RunBlocking is called from
  // any of the services.  since ProcessRMIs is blocking until a new message
  // arrives we can run in deadlock if a service needs  a "run_on_main_thread"
  // request (or RunBlocking) while the main thread is blocked on ProcessRMIs.
  // To avoid the issue we process RMIs on a separate thread for the satellite
  // nodes. Note that this assumes that MPI calls can be assessed by any thread
  // .i.e. MPI_Init_thread(.. MPI_THREAD_MULTIPLE) is used. But we use this
  // already since RS and DS can do MPI calls.
  std::thread mpiWorker;
  if (internals.Rank > 0)
  {
    mpiWorker = std::thread([controller]() { controller->ProcessRMIs(1, /* dont_loop= */ 0); });
  }
  auto start = std::chrono::system_clock::now();
  while (!internals.ExitCodeSet)
  {
    while (!rlp.empty() && rlp.peek().when < rlp.now())
    {
      vtkVLogScopeF(VTKREMOTINGSERVERMANAGERCORE_LOG_VERBOSITY(), "Dispatch one event");
      rlp.dispatch();
      start = std::chrono::system_clock::now();
    }
    if (!(duration.count() <= 0) && (std::chrono::system_clock::now() - start) >= duration)
    {
      // timed out!
      break;
    }
    std::this_thread::sleep_for(std::chrono::microseconds(10));
  }
  if (internals.Rank == 0 && controller->GetNumberOfProcesses() > 0)
  {
    controller->TriggerBreakRMIs();
  }
  if (internals.Rank > 0)
  {
    mpiWorker.join();
  }
  controller->Barrier();
  return internals.ExitCodeSet;
}

//-----------------------------------------------------------------------------
void vtkPVCoreApplication::ProcessEvents(const vtkrxcpp::schedulers::run_loop& rlp) const
{
  const auto& internals = (*this->Internals);
  while (!rlp.empty() && !internals.ExitCodeSet && (rlp.peek().when < rlp.now()))
  {
    rlp.dispatch();
  }
}

//----------------------------------------------------------------------------
const char* vtkPVCoreApplication::GetParaViewSourceVersion()
{
  return APV_SOURCE_VERSION;
}

//----------------------------------------------------------------------------
int vtkPVCoreApplication::GetVersionMajor()
{
  return APV_VERSION_MAJOR;
}

//----------------------------------------------------------------------------
int vtkPVCoreApplication::GetVersionMinor()
{
  return APV_VERSION_MINOR;
}

//----------------------------------------------------------------------------
int vtkPVCoreApplication::GetVersionPatch()
{
  return APV_VERSION_PATCH;
}

//----------------------------------------------------------------------------
const std::string& vtkPVCoreApplication::GetSharedResourcesDirectory() const
{
  const auto& internals = (*this->Internals);
  return internals.SharedResourcesDirectory;
}

//----------------------------------------------------------------------------
const std::string& vtkPVCoreApplication::GetExampleFilesDirectory() const
{
  const auto& internals = (*this->Internals);
  return internals.ExampleFilesDirectory;
}

//----------------------------------------------------------------------------
const std::string& vtkPVCoreApplication::GetDocDirectory() const
{
  const auto& internals = (*this->Internals);
  return internals.DocDirectory;
}

//----------------------------------------------------------------------------
std::string vtkPVCoreApplication::GetCSURL(const std::string& engineURL)
{
  vtksys::RegularExpression regEx("^[^:]+://(.*)$");
  if (!regEx.find(engineURL))
  {
    vtkLogF(ERROR, "URL in unrecognized form ('%s'). Aborting.", engineURL.c_str());
    abort();
  }

  return fmt::format("cs://{}", regEx.match(1));
}

//----------------------------------------------------------------------------
std::string vtkPVCoreApplication::GetEngineURL(const std::string& csURL)
{
  vtksys::RegularExpression regEx("^cs://(.*)$");
  if (!regEx.find(csURL))
  {
    vtkLogF(ERROR, "URL in unrecognized form ('%s'). Aborting.", csURL.c_str());
    abort();
  }

  return fmt::format("{}", regEx.match(1));
}

//----------------------------------------------------------------------------
void vtkPVCoreApplication::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}
