/*=========================================================================

  Program:   ParaView
  Module:    vtkInformationHelpers.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkInformationHelpers.h"
#include "vtkAlgorithm.h"
#include "vtkExecutive.h"
#include "vtkInformation.h"
#include "vtkLogger.h"
#include "vtkNJson.h"
#include "vtkObjectWrapper.h"
#include "vtkReflection.h"
#include "vtkStreamingDemandDrivenPipeline.h"

namespace
{

/**
 * @brief Extract timesteps from a vtkObjectWrapper
 *
 * @return true on success, false otherwise
 * @note logic copied from vtkSITimeStepsProperty of paraview/paraview
 */
bool TimeStepsInformationHelper(const vtkObjectWrapper* wrapper, vtkNJson& values)
{

  // Get reference to the algorithm
  vtkAlgorithm* algo = vtkAlgorithm::SafeDownCast(wrapper->GetVTKObject());

  if (!algo)
  {
    return false;
  }

  vtkInformation* outInfo = algo->GetExecutive()->GetOutputInformation(0);

  // If no information just exit
  if (!outInfo)
  {
    return false;
  }

  // Else find out
  if (outInfo->Has(vtkStreamingDemandDrivenPipeline::TIME_STEPS()))
  {
    const double* timeSteps = outInfo->Get(vtkStreamingDemandDrivenPipeline::TIME_STEPS());
    int len = outInfo->Length(vtkStreamingDemandDrivenPipeline::TIME_STEPS());

    for (int i = 0; i < len; i++)
    {
      values.push_back(timeSteps[i]);
    }
    return true;
  }

  // No value does not mean failure. So just return true
  return true;
}

/**
 * @brief Extract ArraySelection information from a vtkObjectWrapper
 * @return true on success, false otherwise
 * @note takes advatage of the special handling for DataArraySelection that the wrappings implement
 */
bool ArraySelectionInformationHelper(
  const vtkObjectWrapper* wrapper, vtkNJson& values, const std::string& helperAttribute)
{

  // Get reference to the algorithm
  vtkAlgorithm* algo = vtkAlgorithm::SafeDownCast(wrapper->GetVTKObject());

  if (!algo)
  {
    return false;
  }
  std::vector<vtkVariant> args;

  std::string command = "Get" + helperAttribute + "DataArraySelection";
  bool success = vtkReflection::Get(algo, command, args);
  if (!success)
  {
    vtkLogF(
      WARNING, "ArraySelectionInformationHelper: Could not access %s property", command.c_str());
    return false;
  }
  // TODO for reader without Get*DataArraySelection method we need to
  // adapt the code to perform the steps that a Get*DataArraySelection method does in an *RXX.cxx
  // file. It will be much easier to update the reader in VTK to return a ArraySelection

  for (auto& value : args)
  {
    values.push_back(value.ToString());
  }

  return true;
}

}

bool GetValueUsingHelper(const std::string& helperName, const vtkObjectWrapper* wrapper,
  const std::string& helperAttribute, vtkNJson& values)
{
  if (helperName == "TimeStepsInformationHelper")
  {
    return TimeStepsInformationHelper(wrapper, values);
  }
  if (helperName == "ArraySelectionInformationHelper")
  {
    return ArraySelectionInformationHelper(wrapper, values, helperAttribute);
  }
  // TODO add more helpers
  else
  {
    vtkLogF(WARNING, "Support for '%s' is not yet implemented, values will be skipped",
      helperName.c_str());
  }
  return false;
}
