//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "smtk/session/aeva/operators/SelectionResponder.h"
#include "smtk/session/aeva/SelectionResponder_xml.h"
#include "smtk/session/aeva/vtk/vtkGlobalIdBooleans.h"

#include "smtk/extension/vtk/source/vtkResourceMultiBlockSource.h"

#include "smtk/view/Selection.h"

#include "smtk/session/aeva/CellSelection.h"
#include "smtk/session/aeva/Resource.h"

#include "smtk/model/Entity.h"
#include "smtk/model/Face.h"
#include "smtk/model/Model.h"

#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/ComponentItem.h"
#include "smtk/attribute/IntItem.h"
#include "smtk/attribute/StringItem.h"

#include "smtk/operation/Manager.h"
#include "smtk/operation/MarkGeometry.h"

#include "smtk/io/Logger.h"

#include "pqView.h"

#include "vtkAppendDataSets.h"
#include "vtkCellData.h"
#include "vtkCompositeDataIterator.h"
#include "vtkExtractSelection.h"
#include "vtkIdTypeArray.h"
#include "vtkInformation.h"
#include "vtkMultiBlockDataSet.h"
#include "vtkPolyData.h"
#include "vtkSelection.h"
#include "vtkSelectionNode.h"
#include "vtkUnsignedIntArray.h"

#ifdef ERROR
#undef ERROR
#endif

namespace smtk
{
namespace session
{
namespace aeva
{

namespace // anonymous
{

vtkDataObject* subtractByGlobalId(vtkSmartPointer<vtkDataObject> const& workpiece,
  vtkSmartPointer<vtkDataObject> const& tool)
{
  vtkNew<vtkGlobalIdBooleans> filter;
  filter->SetWorkpiece(workpiece);
  filter->SetTool(tool);
  filter->SetOperationToDifference();
  filter->Update();

  vtkDataObject* result = filter->GetOutputDataObject(0);
  result->Register(nullptr);
  return result;
}

} // anonmyous namespace

SelectionResponder::SelectionResponder() = default;

SelectionResponder::~SelectionResponder() = default;

bool SelectionResponder::transcribeCellIdSelection(Result& result)
{
  bool didModify = false;
  if (this->selectingBlocks())
  {
    return didModify;
  }

  auto assoc = this->parameters()->associations();
  smtk::resource::ResourcePtr resource = assoc->valueAs<smtk::resource::Resource>();
  smtk::session::aeva::Resource::Ptr aevaResource =
    std::dynamic_pointer_cast<smtk::session::aeva::Resource>(resource);
  if (!resource)
  {
    smtkErrorMacro(smtk::io::Logger::instance(), "No associated resource for aeva selection.");
    return false;
  }
  auto session = aevaResource->session();

  auto smtkSelection = this->smtkSelection();
  auto* geometry = this->vtkData();
  ::vtkSelection* selnBlock = this->vtkSelection();
  vtkNew<vtkExtractSelection> extractor;
  extractor->SetInputDataObject(0, geometry);
  extractor->SetInputDataObject(1, selnBlock);
  // extractor->PreserveTopologyOn(); // <-- defines a selection array as point-data or cell-data
  extractor->Update();
  auto* selectedStuff = vtkMultiBlockDataSet::SafeDownCast(extractor->GetOutputDataObject(0));

  auto created = result->findComponent("created");
  std::set<smtk::resource::ComponentPtr> selection;
  // Since both selectedStuff and geometry have the same structure,
  // we can apply the iterator to both.
  vtkCompositeDataIterator* iter = selectedStuff->NewIterator();
  // Users may have selected stuff from a side set. But aeva (at least for now)
  // only allows cell-selections from element blocks. So, instead of creating
  // a selection per iterator entry, we append all selected cells on a per-model
  // basis and create a "selection" per element-block entry.
  std::set<vtkSmartPointer<vtkDataObject> > selectedPieces;
  // Note that if a CellSelection instance exists, it must be in the selection.
  auto cellSelection = CellSelection::instance();
  bool addedCellSeln = false;

  // I. Collect pieces of the VTK selection from different model entities
  for (iter->InitTraversal(); !iter->IsDoneWithTraversal(); iter->GoToNextItem())
  {
    auto* seln = iter->GetCurrentDataObject();
    auto uuid = vtkResourceMultiBlockSource::GetDataObjectUUID(geometry->GetMetaData(iter));
    auto srcComp = aevaResource->findEntity(uuid);
    bool isCellSelection = (cellSelection && uuid == cellSelection->id());
    if (!srcComp && !isCellSelection)
    {
      continue;
    }
    if (this->modifier() == pqView::PV_SELECTION_SUBTRACTION && !isCellSelection)
    {
      // No need to add geometry we will ignore (cannot subtract things not in selection from selection).
      continue;
    }
    if (isCellSelection)
    {
      addedCellSeln = true;
      switch (this->modifier())
      {
        case pqView::PV_SELECTION_ADDITION:
          // Keep all the existing cells, not just what the user chose during this pick.
          selectedPieces.insert(aevaResource->session()->findStorage(cellSelection->id()));
          break;
        case pqView::PV_SELECTION_SUBTRACTION: // fall through
        case pqView::PV_SELECTION_TOGGLE:      // fall through
        case pqView::PV_SELECTION_DEFAULT:     // fall through
        default:
          selectedPieces.insert(seln);
          break;
      }
    }
    else
    {
      selectedPieces.insert(seln);
    }
  }
  iter->Delete();

  // II. Collect pieces of the SMTK selection (entire model entities), including the cell selection.
  switch (this->modifier())
  {
    case pqView::PV_SELECTION_ADDITION:
      if (cellSelection && !addedCellSeln)
      {
        selectedPieces.insert(aevaResource->session()->findStorage(cellSelection->id()));
      }
      break;
    default:
      // Do nothing here.
      break;
  }

  // TODO: if modifier() == SUBTRACT/TOGGLE, compute the difference between cellSelection's storage and seln?
  //       else filter out cells with duplicate global IDs.
  // III. Aggregate/append pieces together as appropriate.
  vtkSmartPointer<vtkDataObject> wholeSelection;
  if (!selectedPieces.empty())
  {
    if (selectedPieces.size() > 1)
    {
      vtkNew<vtkAppendDataSets> append;
      append->MergePointsOn();
      for (auto const& input : selectedPieces)
      {
        append->AddInputData(input);
      }
      append->Update();
      wholeSelection = append->GetOutputDataObject(0);
    }
    else
    {
      wholeSelection = *selectedPieces.begin();
    }

    switch (this->modifier())
    {
      case pqView::PV_SELECTION_SUBTRACTION: // fall through
      case pqView::PV_SELECTION_TOGGLE:
      {
        vtkDataObject* result;
        result = subtractByGlobalId(
          aevaResource->session()->findStorage(cellSelection->id()), wholeSelection);
        wholeSelection.TakeReference(result);
      }
      break;
      case pqView::PV_SELECTION_DEFAULT: // fall through
      case pqView::PV_SELECTION_ADDITION:
      default:
        break;
    }

    if (!cellSelection)
    {
      cellSelection = CellSelection::create(aevaResource, wholeSelection, this->manager());
      created->appendValue(cellSelection);
    }
    else
    {
      // Update the geometry of the existing selection
      cellSelection->replaceData(wholeSelection);
      auto modified = result->findComponent("modified");
      modified->appendValue(cellSelection);
    }
    selection.insert(cellSelection);
  }
  didModify |= smtkSelection->modifySelection(selection,
    m_smtkSelectionSource,
    m_smtkSelectionValue,
    view::SelectionAction::FILTERED_ADD,
    /* bitwise */ true,
    /* notify */ false);

  return didModify;
}

SelectionResponder::Result SelectionResponder::operateInternal()
{
  auto result = this->createResult(smtk::operation::Operation::Outcome::SUCCEEDED);
  bool worked = this->transcribeCellIdSelection(result);
  if (!worked)
  {
    result->findInt("outcome")->setValue(
      static_cast<int>(smtk::operation::Operation::Outcome::FAILED));
  }
  return result;
}

const char* SelectionResponder::xmlDescription() const
{
  return SelectionResponder_xml;
}

} // namespace aeva
} // namespace session
} // namespace smtk
