//=============================================================================
//
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//=============================================================================
#include "smtk/session/aeva/Session.h"

#include "vtkCellData.h"
#include "vtkDataObject.h"
#include "vtkDataSet.h"
#include "vtkGenerateGlobalIds.h"
#include "vtkIntArray.h"
#include "vtkPointData.h"

namespace smtk
{
namespace session
{
namespace aeva
{

std::string Session::defaultFileExtension(const smtk::model::Model& /*model*/) const
{
  return ".vti";
}

vtkSmartPointer<vtkDataObject> Session::findStorage(const smtk::common::UUID& uid) const
{
  auto it = m_storage.find(uid);
  if (it != m_storage.end())
  {
    return it->second;
  }
  return nullptr;
}

void Session::addStorage(const smtk::common::UUID& uid,
  vtkSmartPointer<vtkDataObject> const& storage)
{
  if (uid.isNull())
  {
    return;
  }

  if (!storage)
  {
    m_storage.erase(uid);
  }
  else
  {
    m_storage[uid] = storage;
  }
}

#if 0
Session::ImageType::Pointer Session::findStorage(const smtk::common::UUID& uid) const
{
  auto it = m_storage.find(uid);
  if (it != m_storage.end())
  {
    return it->second;
  }
  return nullptr;
}

void Session::addStorage(const smtk::common::UUID& uid, ImageType::Pointer storage)
{
  if (uid.isNull())
  {
    return;
  }

  if (!storage)
  {
    m_storage.erase(uid);
  }
  else
  {
    m_storage[uid] = storage;
  }
}
#endif

bool Session::removeStorage(const smtk::common::UUID& uid)
{
  if (uid.isNull())
  {
    return false;
  }

  return m_storage.erase(uid) > 0;
}

smtk::model::SessionInfoBits Session::transcribeInternal(const smtk::model::EntityRef& /*entity*/,
  SessionInfoBits /*requestedInfo*/,
  int /*depth*/)
{
  return smtk::model::SESSION_EVERYTHING;
}

smtk::model::SessionIOPtr Session::createIODelegate(const std::string& /*format*/)
{
  // TODO: If we write any custom JSON to .smtk files, return a handler.
  return nullptr;
}

void Session::offsetGlobalIds(vtkDataSet* data,
  long pointIdOffset,
  long cellIdOffset,
  long& maxPointId,
  long& maxCellId)
{
  if (!data)
  {
    return;
  }
  vtkIntArray* pointDataNums = vtkIntArray::SafeDownCast(data->GetPointData()->GetGlobalIds());
  vtkIntArray* cellDataNums = vtkIntArray::SafeDownCast(data->GetCellData()->GetGlobalIds());

  if (!pointDataNums || !cellDataNums)
  {
    // need to generate globalIds.
    vtkNew<vtkGenerateGlobalIds> generator;
    generator->SetInputDataObject(data);
    generator->Update();
    auto* outdata = vtkDataSet::SafeDownCast(generator->GetOutputDataObject(0));

    // transform point to ints, to matcch .med format
    if (!pointDataNums)
    {
      vtkSmartPointer<vtkIntArray> newPointIds = vtkIntArray::New();
      newPointIds->DeepCopy(outdata->GetPointData()->GetGlobalIds());
      data->GetPointData()->SetGlobalIds(newPointIds);
      pointDataNums = vtkIntArray::SafeDownCast(data->GetPointData()->GetGlobalIds());
    }

    if (!cellDataNums)
    {
      vtkSmartPointer<vtkIntArray> newCellIds = vtkIntArray::New();
      newCellIds->DeepCopy(outdata->GetCellData()->GetGlobalIds());
      data->GetCellData()->SetGlobalIds(newCellIds);
      cellDataNums = vtkIntArray::SafeDownCast(data->GetCellData()->GetGlobalIds());
    }
  }

  if (pointDataNums && pointDataNums->GetNumberOfValues() > 0)
  {
    bool doOffset = pointDataNums->GetValue(0) <= pointIdOffset;
    for (int i = 0; i < pointDataNums->GetNumberOfValues(); ++i)
    {
      if (doOffset)
      {
        pointDataNums->SetValue(i, pointDataNums->GetValue(i) + pointIdOffset);
      }
      maxPointId = std::max(maxPointId, static_cast<long>(pointDataNums->GetValue(i)));
    }
  }

  if (cellDataNums && cellDataNums->GetNumberOfValues() > 0)
  {
    bool doOffset = cellDataNums->GetValue(0) <= cellIdOffset;
    for (int i = 0; i < cellDataNums->GetNumberOfValues(); ++i)
    {
      if (doOffset)
      {
        cellDataNums->SetValue(i, cellDataNums->GetValue(i) + cellIdOffset);
      }
      maxCellId = std::max(maxCellId, static_cast<long>(cellDataNums->GetValue(i)));
    }
  }
}

void Session::setPrimary(smtk::resource::Component& c, bool isPrimary)
{
  if (isPrimary)
  {
    c.properties().get<long>()["primary"] = 1;
    c.properties().erase<long>("side-set");
  }
  else
  {
    c.properties().get<long>()["side-set"] = 1;
    c.properties().erase<long>("primary");
  }
}

bool Session::isPrimary(const smtk::resource::Component& c)
{
  if (c.properties().contains<long>("primary"))
  {
    return !!c.properties().at<long>("primary");
  }
  return false;
}

} // namespace aeva
} // namespace session
} // namespace smtk
