//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#ifndef vtkMedReader_h
#define vtkMedReader_h

#include "vtkIOGeometryModule.h" // For export macro
#include "vtkMultiBlockDataSetAlgorithm.h"
#include "vtkSmartPointer.h"
#include <vector>

class HdfNode;
class vtkCellArray;
class vtkFieldData;
class vtkIntArray;
class vtkMultiBlockDataSet;
class vtkPoints;
class vtkPolyData;
class vtkUnstructuredGrid;
class vtkStringArray;

enum class MedDataType
{
    NONE,
    UNSTRUCTUREDGRID,
    POLYDATA
};

class vtkMedReader : public vtkMultiBlockDataSetAlgorithm
{
public:
    static vtkMedReader* New();
    vtkTypeMacro(vtkMedReader, vtkMultiBlockDataSetAlgorithm);
    void PrintSelf(ostream& os, vtkIndent indent) override;

    /**
     * Get the output as vtkPolyData
     */
    vtkPolyData* GetPolyDataOutput(int port);

    /**
     * Get the output as vtkUnstructuredGrid
     */
    vtkUnstructuredGrid* GetUnstructuredGridOutput(int port);

    //@{
    /**
    * This file to open and read.
    */
    vtkSetStringMacro(FileName);
    vtkGetStringMacro(FileName);
    //@}

    /**
     * Informs on the type of each object present in the MultiBlockDataSet
     */
    MedDataType GetType(int port) { return Types[port]; }
    vtkIdType GetNumberOfBlocks() { return static_cast<vtkIdType>(Types.size()); }

protected:
    vtkMedReader();
    ~vtkMedReader() override;

    void InsertBlock(vtkMultiBlockDataSet* output,
        vtkPoints* points, vtkCellArray* cells,
        vtkIntArray* pointTags, vtkIntArray* cellTags,
        vtkFieldData* fieldData,
        std::string medCellType);

    int RequestData(vtkInformation* request, vtkInformationVector** inputVec, vtkInformationVector* outputVec) override;

    int FillInputPortInformation(int port, vtkInformation* info) override;

    char* FileName;
    std::vector<MedDataType> Types;

private:
    vtkMedReader(const vtkMedReader&) = delete;
    void operator=(const vtkMedReader&) = delete;
};

#endif
