//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "vtkMedReader.h"

#include "hdf5.h"
#include "vtkCellData.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMultiBlockDataSet.h"
#include "vtkPointData.h"
#include "vtkPolyData.h"
#include "vtkStringArray.h"
#include "vtkUnstructuredGrid.h"
#include "vtksys/SystemTools.hxx"

#include <list>

vtkStandardNewMacro(vtkMedReader);

class HdfNode
{
public:
    hid_t locId = -1;
    std::string name = "";
    std::string path = "";
    HdfNode* parent = nullptr;
    std::list<HdfNode*> children;

    HdfNode() = default;
    HdfNode(const HdfNode&) = default;
    HdfNode(hid_t id, const std::string& sname, const std::string& spath, HdfNode* p = nullptr)
      : locId(id), name(sname), path(spath), parent(p)
    {
    }

    HdfNode* findChild(std::string name)
    {
        for (std::list<HdfNode*>::iterator i = children.begin(); i != children.end(); i++)
        {
            if ((*i)->name == name)
                return *i;
        }
        return nullptr;
    }
};

static HdfNode* maiGroup = nullptr; // CellData
static HdfNode* noeGroup = nullptr; // PointData
static HdfNode* elemeGroup = nullptr; // Cell Tag Information
static HdfNode* noEudGroup = nullptr; // Point Tag Information

// Mappings from med string type to vertex count
static std::unordered_map<std::string, vtkIdType> vertexCount
{
    {"PO1", 1}, // Vertex
    {"SE2", 2}, // Line
    {"SE3", 3}, // Line 3
    {"TR3", 3}, // Triangle
    {"TR6", 6}, // Triangle 6
    {"QU4", 4}, // Quad
    {"QU8", 8}, // Quad 8
    {"TE4", 4}, // Tetra
    {"T10", 10}, // Tetra 10
    {"HE8", 8}, // Hexahedron
    {"HE20", 20}, // Hexahedron 20
    {"PY5", 5}, // Pyramid
    {"PY13", 13}, // Pyramid 13
    {"PE6", 6}, // Wedge
    {"P15", 15} // Wedge 15
};

// Recursive function to iterate throughout the hdf to locate objects and groups of interest
//----------------------------------------------------------------------------
static herr_t buildTree(hid_t locId, const char* name, const H5L_info_t* info, void* opData)
{
    const std::string strName = name;
    H5O_info_t infoBuffer;
    hid_t status = H5Oget_info_by_name(locId, name, &infoBuffer, H5P_DEFAULT);

    // Create a node in the tree for this, doubly link it
    HdfNode* parentNode = static_cast<HdfNode*>(opData);
    HdfNode* currNode = new HdfNode{locId, name, parentNode->path + name + '/'};
    parentNode->children.push_back(currNode);
    currNode->parent = parentNode;

    if (strName == "NOE")
        noeGroup = currNode;
    else if (strName == "MAI")
        maiGroup = currNode;
    else if (strName == "ELEME")
        elemeGroup = currNode;
    else if (strName == "NOEUD")
        noEudGroup = currNode;

    // Test what the link is pointing too
    if (infoBuffer.type == H5O_TYPE_GROUP)
        status = H5Literate_by_name(locId, name, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, buildTree, currNode, H5P_DEFAULT);
    // Other types for datasets

    return 0;
}

//----------------------------------------------------------------------------
static void deleteTree(HdfNode* node)
{
    for (std::list<HdfNode*>::iterator i = node->children.begin(); i != node->children.end(); i++)
    {
        deleteTree(*i);
    }
    delete node;
}


// Reads all the points/vertices/nodes
static bool readPoints(HdfNode* node, hid_t fileId, vtkSmartPointer<vtkPoints> points)
{
    // Load in the pts dataset using its path, named COO
    hid_t dataset = H5Dopen(fileId, node->path.c_str(), H5P_DEFAULT);

    // Read the number of points attribute NBR, NOE->COO->NBR
    hid_t attributeId = H5Aopen_name(dataset, "NBR");
    int numberOfPoints = -1;
    if (H5Aread(attributeId, H5T_NATIVE_INT, &numberOfPoints) != 0)
        return false;
    H5Aclose(attributeId);

    // Check the datatype
    hid_t datatype = H5Dget_type(dataset);
    hid_t type = H5Tget_class(datatype);
    if (type != H5T_FLOAT)
        return false;

    // Med vertices are not strided like VTKs, so we copy
    // (not x1, y1, z1, x2, y2, z2, ... but x1, x2, ..., y1, y2, ..., z1, z2, ...)
    float* pts = new float[numberOfPoints * 3];
    H5Dread(dataset, H5T_NATIVE_FLOAT, H5S_ALL, H5S_ALL, H5P_DEFAULT, pts);
    H5Dclose(dataset);
    points->SetNumberOfPoints(numberOfPoints);
    for (vtkIdType i = 0; i < numberOfPoints; i++)
    {
        points->SetPoint(i, pts[i], pts[i + numberOfPoints], pts[i + 2 * numberOfPoints]);
    }
    delete[] pts;
    return true;
}

// Reads all the cells
static bool readCells(HdfNode* node, hid_t fileId, std::string cellType, vtkSmartPointer<vtkCellArray> cells)
{
    // Load in the cells dataset using its path, named NOD
    const hid_t dataset = H5Dopen(fileId, node->path.c_str(), H5P_DEFAULT);

    // Read the number of cells attribute, MAI->CellTypeGroup->NOD->NBR
    const hid_t attributeId = H5Aopen_name(dataset, "NBR");
    int numberOfCells = -1;
    if (H5Aread(attributeId, H5T_NATIVE_INT, &numberOfCells) != 0)
        return false;
    H5Aclose(attributeId);

    // Check the datatype
    const hid_t datatype = H5Dget_type(dataset);
    const hid_t type = H5Tget_class(datatype);
    if (type != H5T_INTEGER)
        return false;

    // Read in the indices
    const vtkIdType cellVertexCount = vertexCount[cellType];
    int* indices = new int[numberOfCells * cellVertexCount];
    H5Dread(dataset, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT, indices);
    H5Dclose(dataset);

    // Copy into VTK cell datastruct
    // Like the vertices the cells are not strided either
    size_t iter = 0;
    for (vtkIdType i = 0; i < numberOfCells; i++)
    {
        iter = i;
        cells->InsertNextCell(cellVertexCount);
        for (vtkIdType j = 0; j < cellVertexCount; j++)
        {
            cells->InsertCellPoint(indices[iter] - 1);
            iter += numberOfCells;
        }
    }
    delete[] indices;
    return true;
}

// Reads all the strings of the families
static bool readTagGroup(HdfNode* node, hid_t fileId, vtkSmartPointer<vtkIntArray> familyIdArray, vtkSmartPointer<vtkStringArray> stringArray)
{
    for (std::list<HdfNode*>::iterator i = node->children.begin(); i != node->children.end(); i++)
    {
        HdfNode* currNode = *i;
        const hid_t subGroup = H5Gopen(fileId, currNode->path.c_str(), H5P_DEFAULT);
        int famId = 0;
        // Read the corresponding family
        const hid_t attributeId = H5Aopen_name(subGroup, "NUM");
        if (H5Aread(attributeId, H5T_NATIVE_INT, &famId) != 0)
            return false;
        H5Aclose(attributeId);
        H5Gclose(subGroup);

        // Now read in its data object
        HdfNode* groNode = currNode->findChild("GRO");
        if (groNode == nullptr)
            return false;
        HdfNode* nomNode = groNode->findChild("NOM");
        if (nomNode == nullptr)
            return false;
        const hid_t dataset = H5Dopen(fileId, nomNode->path.c_str(), H5P_DEFAULT);
        const hid_t fileType = H5Dget_type(dataset);
        // Get the datatype and its dimensions
        const hid_t arrayNdims = H5Tget_array_ndims(fileType);
        if (arrayNdims != 1)
            return false;
        hsize_t arrayDims;
        H5Tget_array_dims(fileType, &arrayDims);
        if (arrayDims != 80)
            return false;
        // Get dataspace and allocate memory for reading.
        // This is a nd dataset of 1d arrays of 80 ints
        const hid_t dataspace = H5Dget_space(dataset);
        const hid_t nDims = H5Sget_simple_extent_ndims(dataspace);
        if (nDims != 1)
            return false;
        hsize_t dim;
        H5Sget_simple_extent_dims(dataspace, &dim, NULL);

        // Finally allocate and read the data
        int* nomData = new int[arrayDims * dim];
        const hid_t memtype = H5Tarray_create(H5T_NATIVE_INT, 1, &arrayDims);
        H5Dread(dataset, memtype, H5S_ALL, H5S_ALL, H5P_DEFAULT, nomData);
        H5Dclose(dataset);
        for (hsize_t j = 0; j < dim; j++)
        {
            vtkStdString str;
            for (size_t k = 0; k < arrayDims; k++)
            {
                str += static_cast<char>(nomData[k + j * arrayDims]);
            }
            stringArray->InsertNextValue(str);
            familyIdArray->InsertNextValue(famId);
        }
        delete[] nomData;
    }
    return true;
}

// reads all the integer scalar fields on the cells
static bool readTagField(HdfNode* node, hid_t fileId, vtkSmartPointer<vtkIntArray> tags)
{
    // Load in the tag dataset
    const hid_t dataset = H5Dopen(fileId, node->path.c_str(), H5P_DEFAULT);

    // Read the number of points/cells attribute NBR, NOE->FAM->NBR
    const hid_t attributeId = H5Aopen_name(dataset, "NBR");
    int numVals = -1;
    if (H5Aread(attributeId, H5T_NATIVE_INT, &numVals) != 0)
        return false;
    H5Aclose(attributeId);

    // Check the datatype
    const hid_t datatype = H5Dget_type(dataset);
    const hid_t type = H5Tget_class(datatype);
    if (type != H5T_INTEGER)
        return false;

    // Read directly into the VTK data array
    tags->SetNumberOfValues(numVals);
    int* tagsPtr = static_cast<int*>(tags->GetVoidPointer(0));
    H5Dread(dataset, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT, tagsPtr);
    H5Dclose(dataset);
    return true;
}


//----------------------------------------------------------------------------
vtkMedReader::vtkMedReader()
{
    this->FileName = nullptr;
}

//----------------------------------------------------------------------------
vtkMedReader::~vtkMedReader()
{
    delete[] this->FileName;
}


//----------------------------------------------------------------------------
vtkPolyData* vtkMedReader::GetPolyDataOutput(int port)
{
    return vtkPolyData::SafeDownCast(this->GetOutput()->GetBlock(port));
}

//----------------------------------------------------------------------------
vtkUnstructuredGrid* vtkMedReader::GetUnstructuredGridOutput(int port)
{
    return vtkUnstructuredGrid::SafeDownCast(this->GetOutput()->GetBlock(port));
}

//----------------------------------------------------------------------------
void vtkMedReader::InsertBlock(vtkMultiBlockDataSet* output,
    vtkPoints* points, vtkCellArray* cells,
    vtkIntArray* pointTags, vtkIntArray* cellTags,
    vtkFieldData* fieldData,
    std::string medCellType)
{
    // Create a dataset in the multiblock set for this object
    vtkSmartPointer<vtkPointSet> pointSet = nullptr;
    if (medCellType == "PO1" ||
        medCellType == "SE2" || medCellType == "SE3" ||
        medCellType == "TR3" || medCellType == "TR6" ||
        medCellType == "QU4" || medCellType == "QU8")
    {
        vtkSmartPointer<vtkPolyData> results = vtkSmartPointer<vtkPolyData>::New();
        results->SetPolys(cells);
        pointSet = results;
        Types.push_back(MedDataType::POLYDATA);
    }
    else
    {
        vtkSmartPointer<vtkUnstructuredGrid> results = vtkSmartPointer<vtkUnstructuredGrid>::New();
        if (medCellType == "TE4" || medCellType == "T10")
            results->SetCells(VTK_TETRA, cells);
        else if (medCellType == "HE8" || medCellType == "HE20")
            results->SetCells(VTK_HEXAHEDRON, cells);
        else if (medCellType == "PY5" || medCellType == "PY13")
            results->SetCells(VTK_PYRAMID, cells);
        else if (medCellType == "PE6" || medCellType == "P15")
            results->SetCells(VTK_WEDGE, cells);
        else
        {
            vtkWarningMacro(<< "Unknown cell type: " + medCellType);
            return;
        }
        pointSet = results;
        Types.push_back(MedDataType::UNSTRUCTUREDGRID);
    }
    // Add the point and cell tags
    if (pointTags != NULL)
        pointSet->GetPointData()->AddArray(pointTags);
    if (cellTags != NULL)
        pointSet->GetCellData()->AddArray(cellTags);
    if (fieldData != NULL)
        pointSet->GetFieldData()->DeepCopy(fieldData);
    pointSet->SetPoints(points);
    const unsigned int numBlocks = output->GetNumberOfBlocks();
    output->SetNumberOfBlocks(numBlocks + 1);
    output->SetBlock(numBlocks, pointSet);
}


//----------------------------------------------------------------------------
int vtkMedReader::RequestData(
    vtkInformation* vtkNotUsed(request), vtkInformationVector** inputVec, vtkInformationVector* outputVec)
{
    // Get the output of this filter
    vtkInformation* outInfo = outputVec->GetInformationObject(0);
    vtkMultiBlockDataSet* output = vtkMultiBlockDataSet::SafeDownCast(outInfo->Get(vtkDataObject::DATA_OBJECT()));

    if (!vtksys::SystemTools::FileExists(FileName))
    {
        vtkWarningMacro(<< ("Failed to read file. File " + std::string(FileName) + " does not exist").c_str());
        return 1;
    }

    // Open up the HDF
    const hid_t fileId = H5Fopen(FileName, H5F_ACC_RDONLY, H5P_DEFAULT);
    const hid_t rootGroupId = H5Gopen(fileId, ".", H5P_DEFAULT);

    // Build our own tree from the HDF file for quicker movement, paths, and to segment out groups of interest
    // There should be no cicular dependencies in med files
    HdfNode* rootNode = new HdfNode{rootGroupId, ".", "./"};
    H5Literate(rootGroupId, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, buildTree, rootNode);

    // There should only be one "mesh ensemble" in the file
    size_t ensMaaCount = 0;
    for (std::list<HdfNode*>::iterator i = rootNode->children.begin(); i != rootNode->children.end(); i++)
    {
        if ((*i)->name == "ENS_MAA")
            ensMaaCount++;
    }
    if (ensMaaCount > 1)
    {
        vtkWarningMacro(<< "Failed to read file, there should only be one mesh in the file\n");
        return 1;
    }

    // Load in the point data (vertices/nodes)
    vtkSmartPointer<vtkPoints> points = vtkSmartPointer<vtkPoints>::New();
    HdfNode* cooNode = noeGroup->findChild("COO");
    if (cooNode == nullptr || !readPoints(cooNode, fileId, points))
    {
        points = NULL;
        vtkWarningMacro(<< "Failed to read point data from med file");
        return 1;
    }

    // Load the point FAM tags (families) in integer array, if it has any
    vtkSmartPointer<vtkIntArray> pointTags = vtkSmartPointer<vtkIntArray>::New();
    pointTags->SetName("Tags");
    HdfNode* pointFamNode = noeGroup->findChild("FAM");
    if (pointFamNode == nullptr || !readTagField(pointFamNode, fileId, pointTags))
        pointTags = NULL;

    // Create two arrays meant to be parallel
    vtkSmartPointer<vtkStringArray> stringArray = vtkSmartPointer<vtkStringArray>::New();
    stringArray->SetName("Strings");
    vtkSmartPointer<vtkIntArray> familyIdArray = vtkSmartPointer<vtkIntArray>::New();
    familyIdArray->SetName("Family Ids");

    // Under ELEME we have the information for each cell tag
    if (elemeGroup != nullptr)
        readTagGroup(elemeGroup, fileId, familyIdArray, stringArray);

    // Under NOEUD we have the information for each point/node tag
    if (noEudGroup != nullptr)
        readTagGroup(noEudGroup, fileId, familyIdArray, stringArray);

    vtkSmartPointer<vtkFieldData> fieldData = NULL;
    if (elemeGroup != nullptr || noEudGroup != nullptr)
    {
        fieldData = vtkSmartPointer<vtkFieldData>::New();
        // Sort the arrays by Id
        for (vtkIdType i = 0; i < familyIdArray->GetNumberOfValues(); i++)
        {
            const vtkIdType idi = familyIdArray->GetValue(i);
            for (vtkIdType j = i + 1; j < familyIdArray->GetNumberOfValues(); j++)
            {
                const vtkIdType idj = familyIdArray->GetValue(j);
                if (idj < idi)
                {
                    // Swap
                    familyIdArray->SetValue(j, idi);
                    familyIdArray->SetValue(i, idj);
                    const vtkStdString strj = stringArray->GetValue(j);
                    stringArray->SetValue(j, stringArray->GetValue(i));
                    stringArray->SetValue(i, strj);
                }
            }
        }
        fieldData->AddArray(familyIdArray);
        fieldData->AddArray(stringArray);
    }

    // Now load in Cells
    // For every cell type for this mesh (ie: Every group under MAI)
    for (std::list<HdfNode*>::iterator i = maiGroup->children.begin(); i != maiGroup->children.end(); i++)
    {
        // MAI stands for "maillage" which is french for "mesh"
        HdfNode* maiGroupChild = *i;

        // Find the NOD child
        vtkSmartPointer<vtkCellArray> cells = vtkSmartPointer<vtkCellArray>::New();
        HdfNode* nodNode = maiGroupChild->findChild("NOD");
        if (nodNode == nullptr || !readCells(nodNode, fileId, maiGroupChild->name, cells))
        {
            cells = NULL;
            vtkWarningMacro(<< "Failed to read cell data from med file");
            return 1;
        }

        // Load the point FAM tags (families) in integer array, if it has any
        vtkSmartPointer<vtkIntArray> cellTags = vtkSmartPointer<vtkIntArray>::New();
        cellTags->SetName("Tags");
        HdfNode* cellFamNode = maiGroupChild->findChild("FAM");
        if (cellFamNode == nullptr || !readTagField(cellFamNode, fileId, cellTags))
            pointTags = NULL;

        // Add the data to the output
        InsertBlock(output, points, cells, pointTags, cellTags, fieldData, maiGroupChild->name);
    }

    // Finally cleanup
    H5Gclose(rootGroupId);
    H5Fclose(fileId);
    deleteTree(rootNode);

    return 1;
}


//----------------------------------------------------------------------------
int vtkMedReader::FillInputPortInformation(int port, vtkInformation* info)
{
    if (port == 0)
    {
        info->Set(vtkAlgorithm::INPUT_IS_OPTIONAL(), 1);
        return 1;
    }
    return 0;
}

//----------------------------------------------------------------------------
void vtkMedReader::PrintSelf(ostream& os, vtkIndent indent)
{
    this->Superclass::PrintSelf(os, indent);

    os << indent << "FileName: " << this->FileName << "\n";
    os << indent << "Number of Meshes: " << this->Types.size() << "\n";
    for (size_t i = 0; i < Types.size(); i++)
    {
        if (this->Types[i] == MedDataType::POLYDATA)
            os << indent << "Port " + std::to_string(i) + " Type: POLYDATA\n";
        else if (this->Types[i] == MedDataType::UNSTRUCTUREDGRID)
            os << indent << "Port " + std::to_string(i) + " Type: UNSTRUCTUREDGRID\n";
    }
}
