//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "smtk/io/Logger.h"

#include "smtk/session/aeva/operators/Import.h"

#include "smtk/session/aeva/vtk/vtkMedReader.h"

#include "smtk/session/aeva/Import_xml.h"
#include "smtk/session/aeva/Resource.h"
#include "smtk/session/aeva/Session.h"

#include "smtk/attribute/Resource.h"
#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/ComponentItem.h"
#include "smtk/attribute/FileItem.h"
#include "smtk/attribute/IntItem.h"
#include "smtk/attribute/ReferenceItem.h"
#include "smtk/attribute/ResourceItem.h"
#include "smtk/attribute/StringItem.h"

#include "smtk/model/Group.h"
#include "smtk/model/Model.h"
#include "smtk/model/Volume.h"
#include "smtk/model/Face.h"

#include "smtk/operation/MarkGeometry.h"

#include "smtk/resource/Manager.h"



#include "smtk/common/Paths.h"

#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageToVTKImageFilter.h"

#include "vtkImageData.h"
#include "vtkPolyData.h"
#include "vtkUnstructuredGrid.h"
#include "vtkPointData.h"
#include "vtkDataArray.h"
#include "vtkLookupTable.h"
#include "vtkSmartPointer.h"
#include "vtkXMLImageDataReader.h"
#include "vtkXMLPImageDataReader.h"

SMTK_THIRDPARTY_PRE_INCLUDE
#include "boost/filesystem.hpp"
SMTK_THIRDPARTY_POST_INCLUDE

using namespace smtk::model;
using namespace smtk::common;
using namespace boost::filesystem;

namespace smtk
{
namespace session
{
namespace aeva
{

Import::Result Import::operateInternal()
{
  smtk::session::aeva::Resource::Ptr resource = nullptr;
  smtk::session::aeva::Session::Ptr session = nullptr;
  m_result = this->createResult(Import::Outcome::FAILED);

  auto assoc = this->parameters()->associations();
  if (assoc->isEnabled() && assoc->isSet(0))
  {
    resource = dynamic_pointer_cast<Resource>(assoc->objectValue(0));
    if (resource)
    {
      session = resource->session();
    }
  }

  smtk::attribute::FileItem::Ptr filenameItem = this->parameters()->findFile("filename");
  std::string filename = filenameItem->value();

  // Create a new resource for the import if needed.
  if (!resource)
  {
    auto manager = this->specification()->manager();
    if (manager)
    {
      resource = manager->create<smtk::session::aeva::Resource>();
    }
    else
    {
      resource = smtk::session::aeva::Resource::create();
    }
    m_result->findResource("resource")->setValue(resource);
  }
  if (!session)
  {
    session = smtk::session::aeva::Session::create();
    resource->setSession(session);
  }

  std::string potentialName = smtk::common::Paths::stem(filename);
  if (resource->name().empty() && !potentialName.empty())
  {
    resource->setName(potentialName);
  }

  if (smtk::common::Paths::extension(filename) == ".med")
  {
    return this->importMedMesh(resource);
  }
  else
  {
    return this->importITKImage(resource);
  }
}

Import::Result Import::importVTKImage(const Resource::Ptr& resource)
{
  auto& session = resource->session();
  smtk::attribute::FileItem::Ptr filenameItem = this->parameters()->findFile("filename");
  smtk::attribute::StringItem::Ptr labelItem = this->parameters()->findString("label map");
  std::string filename = filenameItem->value();

  vtkSmartPointer<vtkImageData> image;
  // TODO: This should be done *after* the image has been successfully loaded.
  //       However, to make the ITK/VTK ifdef below easier, we put it here.
  //       Move it later.
  auto model = resource->addModel(3, 3, filename);
  auto volume = resource->addVolume();
  model.addCell(volume);
  auto createdItems = m_result->findComponent("created");
  createdItems->appendValue(model.component());
  createdItems->appendValue(volume.component());

  // For now, load a surrogate VTK image
  vtkNew<vtkXMLImageDataReader> reader;
  reader->SetFileName(filename.c_str());
  reader->Update();
  image = reader->GetOutput();

  auto scalars = image->GetPointData()->GetScalars();
  if (scalars)
  {
    auto lkup = scalars->GetLookupTable();
    if (!lkup)
    {
      vtkNew<vtkLookupTable> greyscale;
      greyscale->SetSaturationRange(0, 0);
      greyscale->SetValueRange(0, 1);
      greyscale->Build();
      scalars->SetLookupTable(greyscale);
      lkup = greyscale;
    }
    lkup->SetTableRange(scalars->GetRange());
  }

  session->addStorage(volume.entity(), image);
  if (image->GetPointData()->GetScalars())
  {
    volume.setName(image->GetPointData()->GetScalars()->GetName());
  }
  operation::MarkGeometry(resource).markModified(volume.component());

  m_result->findInt("outcome")->setValue(0, static_cast<int>(Import::Outcome::SUCCEEDED));
  return m_result;
}

Import::Result Import::importITKImage(const Resource::Ptr& resource)
{
  auto& session = resource->session();
  smtk::attribute::FileItem::Ptr filenameItem = this->parameters()->findFile("filename");
  smtk::attribute::StringItem::Ptr labelItem = this->parameters()->findString("label map");
  std::string filename = filenameItem->value();

  vtkSmartPointer<vtkImageData> image;
  // TODO: This should be done *after* the image has been successfully loaded.
  //       However, to make the ITK/VTK ifdef below easier, we put it here.
  //       Move it later.
  auto model = resource->addModel(3, 3, filename);
  auto volume = resource->addVolume();
  model.addCell(volume);
  auto createdItems = m_result->findComponent("created");
  createdItems->appendValue(model.component());
  createdItems->appendValue(volume.component());
#if 1
  // ITK code
  constexpr unsigned int Dimension = 3;

  using PixelType = unsigned short;
  using ImageType = itk::Image<PixelType, Dimension>;

  using ReaderType = itk::ImageFileReader<ImageType>;
  ReaderType::Pointer reader = ReaderType::New();
  if (!reader)
  {
    smtkInfoMacro(this->log(), "Unable to create reader for \"" << filename << "\".");
    return m_result;
  }
  reader->SetFileName(filename);
  reader->Update();

  ImageType::Pointer img = reader->GetOutput();
  if (!img)
  {
    smtkInfoMacro(this->log(), "Unable to read image from: " << filename << ".");
    return m_result;
  }

  // session->addStorage(volume.entity(), img);

  //Extract data from ITK image
  auto region = reader->GetOutput()->GetLargestPossibleRegion();
  auto size = region.GetSize();
  auto spacing = reader->GetOutput()->GetSpacing();
  auto origin = reader->GetOutput()->GetOrigin();
  auto directions = reader->GetOutput()->GetDirection();
  auto directions_vnl = directions.GetVnlMatrix();
  directions_vnl.inplace_transpose();

  //Set up image importer
  vtkSmartPointer<vtkImageImport> importer = vtkSmartPointer<vtkImageImport>::New();
  importer->SetWholeExtent(0, size[0] - 1, 0, size[1] - 1, 0, size[2] - 1);
  importer->SetDataExtentToWholeExtent();
  importer->SetDataDirection(directions_vnl.data_block());
  importer->SetDataSpacing(spacing[0], spacing[1], spacing[2]);
  importer->SetDataOrigin(origin[0], origin[1], origin[2]);
  importer->SetDataScalarType(VTK_UNSIGNED_SHORT);
  importer->SetNumberOfScalarComponents(1);
  importer->SetImportVoidPointer(reader->GetOutput()->GetBufferPointer());
  importer->Update();

  image = vtkSmartPointer<vtkImageData>::New();
  image->DeepCopy(importer->GetOutput());
  auto scalars = image->GetPointData()->GetScalars();
  if (scalars)
  {
    auto lkup = scalars->GetLookupTable();
    if (!lkup)
    {
      vtkNew<vtkLookupTable> greyscale;
      greyscale->SetSaturationRange(0, 0);
      greyscale->SetValueRange(0, 1);
      greyscale->Build();
      scalars->SetLookupTable(greyscale);
      lkup = greyscale;
    }
    lkup->SetTableRange(scalars->GetRange());
  }

#else
  // For now, load a surrogate VTK image
  vtkNew<vtkXMLImageDataReader> reader;
  reader->SetFileName(filename.c_str());
  reader->Update();
  image = reader->GetOutput();
#endif // 0
  session->addStorage(volume.entity(), image);
  if (image->GetPointData()->GetScalars())
  {
    volume.setName(image->GetPointData()->GetScalars()->GetName());
  }
  operation::MarkGeometry(resource).markModified(volume.component());

  m_result->findInt("outcome")->setValue(0, static_cast<int>(Import::Outcome::SUCCEEDED));
  return m_result;
}

Import::Result Import::importMedMesh(const Resource::Ptr& resource)
{
  auto& session = resource->session();
  smtk::attribute::FileItem::Ptr filenameItem = this->parameters()->findFile("filename");
  smtk::attribute::StringItem::Ptr labelItem = this->parameters()->findString("label map");

  for (size_t j = 0; j < filenameItem->numberOfValues(); j++)
  {
    vtkSmartPointer<vtkMedReader> reader = vtkSmartPointer<vtkMedReader>::New();
    reader->SetFileName(filenameItem->value(j).c_str());
    reader->Update();

    auto stem = smtk::common::Paths::stem(filenameItem->value(j));
    auto model = resource->addModel(3, 3, stem);
    auto createdItems = m_result->findComponent("created");
    createdItems->appendValue(model.component());
    CellEntity srf;
    CellEntity vol;
    for (vtkIdType i = 0; i < reader->GetNumberOfBlocks(); i++)
    {
      const MedDataType type = reader->GetType(i);
      std::string typeName;
      CellEntity cell;
      vtkDataObject* data = nullptr;
      switch (type)
      {
        case MedDataType::POLYDATA:
          cell = resource->addFace();
          srf = cell;
          typeName = " surface";
          data = reader->GetPolyDataOutput(i);
          break;
        case MedDataType::UNSTRUCTUREDGRID:
          cell = resource->addVolume();
          vol = cell;
          typeName = " volume";
          data = nullptr;
          break;
        case MedDataType::NONE:
          continue; // No SMTK representation... skip.
      }
      cell.setName(stem + typeName);

      createdItems->appendValue(cell.component());

      if (data)
      {
        session->addStorage(cell.entity(), data);
        operation::MarkGeometry(resource).markModified(cell.component());
      }
    }

    // TODO: This is a hack based on the way MED files are being used
    // by aeva. In general, it is possible there are multiple volumes
    // and/or surfaces. However, without a more formal specification
    // for the MED file format, it is unclear how to infer relationships
    // among volumes and surfaces.
    if (srf.isValid() && vol.isValid())
    {
      vol.addRawRelation(srf);
      srf.addRawRelation(vol);
      model.addCell(vol);
    }
    else if (srf.isValid())
    {
      model.addCell(srf);
    }
  }

  m_result->findInt("outcome")->setValue(0, static_cast<int>(Import::Outcome::SUCCEEDED));
  return m_result;
}

const char* Import::xmlDescription() const
{
  return Import_xml;
}

smtk::resource::ResourcePtr importResource(const std::string& filename)
{
  Import::Ptr importResource = Import::create();
  importResource->parameters()->findFile("filename")->setValue(filename);
  Import::Result result = importResource->operate();
  if (result->findInt("outcome")->value() != static_cast<int>(Import::Outcome::SUCCEEDED))
  {
    return smtk::resource::ResourcePtr();
  }
  return result->findResource("resource")->value();
}
} // namespace aeva
} // namespace session
} // namespace smtk
