//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
// .NAME vtkLIDARElevationFilter - generate scalars along a specified direction
// .SECTION Description
// This filter is the same as the one in VTK, except for the addition of
// a vtkTransform, which if present, is appied to each point before assigning
// a scalar value.  The output point is NOT modified.
//
// vtkLIDARElevationFilter is a filter to generate scalar values from a
// dataset.  The scalar values lie within a user specified range, and
// are generated by computing a projection of each dataset point onto
// a line. The line can be oriented arbitrarily. A typical example is
// to generate scalars based on elevation or height above a plane.

#ifndef __vtkLIDARElevationFilter_h
#define __vtkLIDARElevationFilter_h

#include "cmbSystemConfig.h"
#include "vtkCMBFilteringModule.h" // For export macro
#include "vtkDataSetAlgorithm.h"
class vtkTransform;

class VTKCMBFILTERING_EXPORT vtkLIDARElevationFilter : public vtkDataSetAlgorithm
{
public:
  static vtkLIDARElevationFilter* New();
  vtkTypeMacro(vtkLIDARElevationFilter, vtkDataSetAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  // Description:
  // Define one end of the line (small scalar values).  Default is
  // (0,0,0).
  vtkSetVector3Macro(LowPoint, double);
  vtkGetVectorMacro(LowPoint, double, 3);

  // Description:
  // Define other end of the line (large scalar values).  Default is
  // (0,0,1).
  vtkSetVector3Macro(HighPoint, double);
  vtkGetVectorMacro(HighPoint, double, 3);

  // Description:
  // Specify range to map scalars into.  Default is [0, 1].
  vtkSetVector2Macro(ScalarRange, double);
  vtkGetVectorMacro(ScalarRange, double, 2);

  // Description:
  // Specify whether the filter should add elevation info or not (true by default)
  vtkSetMacro(CreateElevation, bool);
  vtkGetMacro(CreateElevation, bool);

  // Description:
  // Transform to apply to points before assigning a scalar value.
  // the point is not transformed for the output.
  void SetTransform(vtkTransform* transform);
  vtkGetObjectMacro(Transform, vtkTransform);
  void SetTransform(double elements[16]);
  void ClearTransform() { this->SetTransform(static_cast<vtkTransform*>(0)); }

protected:
  vtkLIDARElevationFilter();
  ~vtkLIDARElevationFilter() override;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  vtkTransform* Transform;
  double LowPoint[3];
  double HighPoint[3];
  double ScalarRange[2];
  bool CreateElevation;

private:
  vtkLIDARElevationFilter(const vtkLIDARElevationFilter&); // Not implemented.
  void operator=(const vtkLIDARElevationFilter&);          // Not implemented.
};

#endif
