/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCommunicator.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkCommunicator - Used to send/receive messages in a multiprocess environment.
// .SECTION Description
// This is an abstact class which contains functionality for sending
// and receiving inter-process messages. It contains methods for marshaling
// an object into a string (currently used by the MPI communicator but
// not the shared memory communicator).

// .SECTION Caveats
// Communication between systems with different vtkIdTypes is not
// supported. All machines have to have the same vtkIdType.

// .SECTION see also
// vtkMPICommunicator

#ifndef __vtkCommunicator_h
#define __vtkCommunicator_h

#include "vtkObject.h"

class vtkBoundingBox;
class vtkCharArray;
class vtkDataArray;
class vtkDataObject;
class vtkDataSet;
class vtkImageData;
class vtkMultiBlockDataSet;
class vtkTemporalDataSet;

class VTK_PARALLEL_EXPORT vtkCommunicator : public vtkObject
{

public:

  vtkTypeRevisionMacro(vtkCommunicator, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent);

  // Description:
  // Set the number of processes you will be using.  This defaults
  // to the maximum number available.  If you set this to a value
  // higher than the default, you will get an error.
  virtual void SetNumberOfProcesses(int num);
  vtkGetMacro(NumberOfProcesses, int);

  // Description:
  // Tells you which process [0, NumProcess) you are in.
  vtkGetMacro(LocalProcessId, int);

//BTX

  enum Tags
  {
    BROADCAST_TAG       = 10,
    GATHER_TAG          = 11,
    GATHERV_TAG         = 12,
    SCATTER_TAG         = 13,
    SCATTERV_TAG        = 14,
    REDUCE_TAG          = 15,
    BARRIER_TAG         = 16
  };

  enum StandardOperations
  {
    MAX_OP,
    MIN_OP,
    SUM_OP,
    PRODUCT_OP,
    LOGICAL_AND_OP,
    BITWISE_AND_OP,
    LOGICAL_OR_OP,
    BITWISE_OR_OP,
    LOGICAL_XOR_OP,
    BITWISE_XOR_OP
  };

  // Description:
  // A custom operation to use in a reduce command.  Subclass this object to
  // provide your own operations.
  class Operation
  {
  public:
    // Description:
    // Subclasses must overload this method, which performs the actual
    // operations.  The methods should first do a reintepret cast of the arrays
    // to the type suggestsed by \c datatype (which will be one of the VTK type
    // identifiers like VTK_INT, etc.).  Both arrays are considered top be
    // length entries.  The method should perform the operation A*B (where * is
    // a placeholder for whatever operation is actually performed) and store the
    // result in B.  The operation is assumed to be associative.  Commutativity
    // is specified by the Commutative method.
    virtual void Function(const void *A, void *B, vtkIdType length,
                          int datatype) = 0;

    // Description:
    // Subclasses override this method to specify whether their operation
    // is commutative.  It should return 1 if commutative or 0 if not.
    virtual int Commutative() = 0;

    virtual ~Operation() {}
  };

//ETX

  // Description:
  // This method sends a data object to a destination.  
  // Tag eliminates ambiguity
  // and is used to match sends to receives.
  int Send(vtkDataObject* data, int remoteHandle, int tag);

  // Description:
  // This method sends a data array to a destination.  
  // Tag eliminates ambiguity
  // and is used to match sends to receives.
  int Send(vtkDataArray* data, int remoteHandle, int tag);

  // Description:
  // Subclasses have to supply this method to send various arrays of data.
  // The \c type arg is one of the VTK type constants recognized by the
  // vtkTemplateMacro (VTK_FLOAT, VTK_INT, etc.).  \c length is measured
  // in number of values (as opposed to number of bytes).
  virtual int SendVoidArray(const void *data, vtkIdType length, int type,
                            int remoteHandle, int tag) = 0;
  
  // Description:
  // Convenience methods for sending data arrays.
  int Send(const int* data, vtkIdType length, int remoteHandle, int tag) {
    return this->SendVoidArray(data, length, VTK_INT, remoteHandle, tag);
  }
  int Send(const unsigned long* data, vtkIdType length,
           int remoteHandle, int tag) {
    return this->SendVoidArray(data, length,VTK_UNSIGNED_LONG,remoteHandle,tag);
  }
  int Send(const unsigned char* data, vtkIdType length,
           int remoteHandle, int tag) {
    return this->SendVoidArray(data, length,VTK_UNSIGNED_CHAR,remoteHandle,tag);
  }
  int Send(const char* data, vtkIdType length, int remoteHandle, int tag) {
    return this->SendVoidArray(data, length, VTK_CHAR, remoteHandle, tag);
  }
  int Send(const float* data, vtkIdType length, int remoteHandle, int tag) {
    return this->SendVoidArray(data, length, VTK_FLOAT, remoteHandle, tag);
  }
  int Send(const double* data, vtkIdType length, int remoteHandle, int tag) {
    return this->SendVoidArray(data, length, VTK_DOUBLE, remoteHandle, tag);
  }
#ifdef VTK_USE_64BIT_IDS
  int Send(const vtkIdType* data, vtkIdType length, int remoteHandle, int tag) {
    return this->SendVoidArray(data, length, VTK_ID_TYPE, remoteHandle, tag);
  }
#endif


  // Description:
  // This method receives a data object from a corresponding send. It blocks
  // until the receive is finished. 
  int Receive(vtkDataObject* data, int remoteHandle, int tag);

  // Description:
  // The caller does not have to know the data type before this call is made.
  // It returns the newly created object.
  vtkDataObject *ReceiveDataObject(int remoteHandle, int tag);

  // Description:
  // This method receives a data array from a corresponding send. It blocks
  // until the receive is finished. 
  int Receive(vtkDataArray* data, int remoteHandle, int tag);

  // Description:
  // Subclasses have to supply this method to receive various arrays of data.
  // The \c type arg is one of the VTK type constants recognized by the
  // vtkTemplateMacro (VTK_FLOAT, VTK_INT, etc.).  \c length is measured
  // in number of values (as opposed to number of bytes).
  virtual int ReceiveVoidArray(void *data, vtkIdType length, int type,
                               int remoteHandle, int tag) = 0;

  // Description:
  // Convenience methods for receiving data arrays.
  int Receive(int* data, vtkIdType length, int remoteHandle, int tag) {
    return this->ReceiveVoidArray(data, length, VTK_INT, remoteHandle, tag);
  }
  int Receive(unsigned long* data, vtkIdType length, int remoteHandle, int tag){
    return this->ReceiveVoidArray(data, length, VTK_UNSIGNED_LONG, remoteHandle,
                                  tag);
  }
  int Receive(unsigned char* data, vtkIdType length, int remoteHandle, int tag){
    return this->ReceiveVoidArray(data, length, VTK_UNSIGNED_CHAR, remoteHandle,
                                  tag);
  }
  int Receive(char* data, vtkIdType length, int remoteHandle, int tag) {
    return this->ReceiveVoidArray(data, length, VTK_CHAR, remoteHandle, tag);
  }
  int Receive(float* data, vtkIdType length, int remoteHandle, int tag) {
    return this->ReceiveVoidArray(data, length, VTK_FLOAT, remoteHandle, tag);
  }
  int Receive(double* data, vtkIdType length, int remoteHandle, int tag) {
    return this->ReceiveVoidArray(data, length, VTK_DOUBLE, remoteHandle, tag);
  }
#ifdef VTK_USE_64BIT_IDS
  int Receive(vtkIdType* data, vtkIdType length, int remoteHandle, int tag) {
    return this->ReceiveVoidArray(data, length, VTK_ID_TYPE, remoteHandle, tag);
  }
#endif

  //---------------------- Collective Operations ----------------------

  // Description:
  // Will block the processes until all other processes reach the Barrier
  // function.
  virtual void Barrier();

  // Description:
  // Broadcast sends the array in the process with id \c srcProcessId to all of
  // the other processes.  All processes must call these method with the same
  // arguments in order for it to complete.
  int Broadcast(int *data, vtkIdType length, int srcProcessId) {
    return this->BroadcastVoidArray(data, length, VTK_INT, srcProcessId);
  }
  int Broadcast(unsigned long *data, vtkIdType length, int srcProcessId) {
    return this->BroadcastVoidArray(data,length,VTK_UNSIGNED_LONG,srcProcessId);
  }
  int Broadcast(unsigned char *data, vtkIdType length, int srcProcessId) {
    return this->BroadcastVoidArray(data,length,VTK_UNSIGNED_CHAR,srcProcessId);
  }
  int Broadcast(char *data, vtkIdType length, int srcProcessId) {
    return this->BroadcastVoidArray(data, length, VTK_CHAR, srcProcessId);
  }
  int Broadcast(float *data, vtkIdType length, int srcProcessId) {
    return this->BroadcastVoidArray(data, length, VTK_FLOAT, srcProcessId);
  }
  int Broadcast(double *data, vtkIdType length, int srcProcessId) {
    return this->BroadcastVoidArray(data, length, VTK_DOUBLE, srcProcessId);
  }
#ifdef VTK_USE_64BIT_IDS
  int Broadcast(vtkIdType *data, vtkIdType length, int srcProcessId) {
    return this->BroadcastVoidArray(data, length, VTK_ID_TYPE, srcProcessId);
  }
#endif
  int Broadcast(vtkDataObject *data, int srcProcessId);
  int Broadcast(vtkDataArray *data, int srcProcessId);

  // Description:
  // Gather collects arrays in the process with id \c destProcessId.  Each
  // process (including the destination) sends the contents of its send buffer
  // to the destination process.  The destination process receives the
  // messages and stores them in rank order.  The \c length argument
  // (which must be the same on all processes) is the length of the
  // sendBuffers.  The \c recvBuffer (on te destination process) must be of
  // length length*numProcesses.  Gather is the inverse operation of Scatter.
  int Gather(const int *sendBuffer, int *recvBuffer,
             vtkIdType length, int destProcessId) {
    return this->GatherVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_INT, destProcessId);
  }
  int Gather(const unsigned long *sendBuffer, unsigned long *recvBuffer,
             vtkIdType length, int destProcessId) {
    return this->GatherVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_UNSIGNED_LONG, destProcessId);
  }
  int Gather(const unsigned char *sendBuffer, unsigned char *recvBuffer,
             vtkIdType length, int destProcessId) {
    return this->GatherVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_UNSIGNED_CHAR, destProcessId);
  }
  int Gather(const char *sendBuffer, char *recvBuffer,
             vtkIdType length, int destProcessId) {
    return this->GatherVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_CHAR, destProcessId);
  }
  int Gather(const float *sendBuffer, float *recvBuffer,
             vtkIdType length, int destProcessId) {
    return this->GatherVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_FLOAT, destProcessId);
  }
  int Gather(const double *sendBuffer, double *recvBuffer,
             vtkIdType length, int destProcessId) {
    return this->GatherVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_DOUBLE, destProcessId);
  }
#ifdef VTK_USE_64BIT_IDS
  int Gather(const vtkIdType *sendBuffer, vtkIdType *recvBuffer,
             vtkIdType length, int destProcessId) {
    return this->GatherVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_ID_TYPE, destProcessId);
  }
#endif
  int Gather(vtkDataArray *sendBuffer, vtkDataArray *recvBuffer,
             int destProcessId);

  // Description:
  // GatherV is the vector variant of Gather.  It extends the functionality of
  // Gather by allowing a varying count of data from each process.
  // GatherV collects arrays in the process with id \c destProcessId.  Each
  // process (including the destination) sends the contents of its send buffer
  // to the destination process.  The destination process receives the
  // messages and stores them in rank order.  The \c sendLength argument
  // defines how much the local process sends to \c destProcessId and
  // \c recvLengths is an array containing the amount \c destProcessId
  // receives from each process, in rank order.
  int GatherV(const int* sendBuffer, int* recvBuffer, 
              vtkIdType sendLength, vtkIdType* recvLengths, vtkIdType* offsets,
              int destProcessId) {
    return this->GatherVVoidArray(sendBuffer, recvBuffer,
                                  sendLength, recvLengths,
                                  offsets, VTK_INT, destProcessId);
  }
  int GatherV(const unsigned long* sendBuffer, unsigned long* recvBuffer, 
              vtkIdType sendLength, vtkIdType* recvLengths, vtkIdType* offsets,
              int destProcessId) {
    return this->GatherVVoidArray(sendBuffer, recvBuffer,
                                  sendLength, recvLengths,
                                  offsets, VTK_UNSIGNED_LONG, destProcessId);
  }
  int GatherV(const unsigned char* sendBuffer, unsigned char* recvBuffer, 
              vtkIdType sendLength, vtkIdType* recvLengths, vtkIdType* offsets,
              int destProcessId) {
    return this->GatherVVoidArray(sendBuffer, recvBuffer,
                                  sendLength, recvLengths,
                                  offsets, VTK_UNSIGNED_CHAR, destProcessId);
  }
  int GatherV(const char* sendBuffer, char* recvBuffer, 
              vtkIdType sendLength, vtkIdType* recvLengths, vtkIdType* offsets,
              int destProcessId) {
    return this->GatherVVoidArray(sendBuffer, recvBuffer,
                                  sendLength, recvLengths,
                                  offsets, VTK_CHAR, destProcessId);
  }
  int GatherV(const float* sendBuffer, float* recvBuffer, 
              vtkIdType sendLength, vtkIdType* recvLengths, vtkIdType* offsets,
              int destProcessId) {
    return this->GatherVVoidArray(sendBuffer, recvBuffer,
                                  sendLength, recvLengths,
                                  offsets, VTK_FLOAT, destProcessId);
  }
  int GatherV(const double* sendBuffer, double* recvBuffer, 
              vtkIdType sendLength, vtkIdType* recvLengths, vtkIdType* offsets,
              int destProcessId) {
    return this->GatherVVoidArray(sendBuffer, recvBuffer,
                                  sendLength, recvLengths,
                                  offsets, VTK_DOUBLE, destProcessId);
  }
#ifdef VTK_USE_64BIT_IDS
  int GatherV(const vtkIdType* sendBuffer, vtkIdType* recvBuffer, 
              vtkIdType sendLength, vtkIdType* recvLengths, vtkIdType* offsets,
              int destProcessId) {
    return this->GatherVVoidArray(sendBuffer, recvBuffer,
                                  sendLength, recvLengths,
                                  offsets, VTK_ID_TYPE, destProcessId);
  }
#endif

  // Description:
  // Scatter takes an array in the process with id \c srcProcessId and
  // distributes it.  Each process (including the source) receives a portion of
  // the send buffer.  Process 0 receives the first \c length values, process 1
  // receives the second \c length values, and so on.  Scatter is the inverse
  // operation of Gather.
  int Scatter(const int *sendBuffer, int *recvBuffer,
             vtkIdType length, int srcProcessId) {
    return this->ScatterVoidArray(sendBuffer, recvBuffer, length,
                                  VTK_INT, srcProcessId);
  }
  int Scatter(const unsigned long *sendBuffer, unsigned long *recvBuffer,
             vtkIdType length, int srcProcessId) {
    return this->ScatterVoidArray(sendBuffer, recvBuffer, length,
                                  VTK_UNSIGNED_LONG, srcProcessId);
  }
  int Scatter(const unsigned char *sendBuffer, unsigned char *recvBuffer,
             vtkIdType length, int srcProcessId) {
    return this->ScatterVoidArray(sendBuffer, recvBuffer, length,
                                  VTK_UNSIGNED_CHAR, srcProcessId);
  }
  int Scatter(const char *sendBuffer, char *recvBuffer,
             vtkIdType length, int srcProcessId) {
    return this->ScatterVoidArray(sendBuffer, recvBuffer, length,
                                  VTK_CHAR, srcProcessId);
  }
  int Scatter(const float *sendBuffer, float *recvBuffer,
             vtkIdType length, int srcProcessId) {
    return this->ScatterVoidArray(sendBuffer, recvBuffer, length,
                                  VTK_FLOAT, srcProcessId);
  }
  int Scatter(const double *sendBuffer, double *recvBuffer,
             vtkIdType length, int srcProcessId) {
    return this->ScatterVoidArray(sendBuffer, recvBuffer, length,
                                  VTK_DOUBLE, srcProcessId);
  }
#ifdef VTK_USE_64BIT_IDS
  int Scatter(const vtkIdType *sendBuffer, vtkIdType *recvBuffer,
             vtkIdType length, int srcProcessId) {
    return this->ScatterVoidArray(sendBuffer, recvBuffer, length,
                                  VTK_ID_TYPE, srcProcessId);
  }
#endif
  int Scatter(vtkDataArray *sendBuffer, vtkDataArray *recvBuffer,
             int srcProcessId);

  // Description:
  // ScatterV is the vector variant of Scatter.  It extends the functionality of
  // Scatter by allowing a varying count of data to each process.
  // ScatterV takes an array in the process with id \c srcProcessId and
  // distributes it.  Each process (including the source) receives a portion of
  // the send buffer defined by the \c sendLengths and \c offsets arrays.
  int ScatterV(const int *sendBuffer, int *recvBuffer,
               vtkIdType *sendLengths, vtkIdType *offsets,
               vtkIdType recvLength, int srcProcessId) {
    return this->ScatterVVoidArray(sendBuffer, recvBuffer,
                                   sendLengths, offsets, recvLength,
                                   VTK_INT, srcProcessId);
  }
  int ScatterV(const unsigned long *sendBuffer, unsigned long *recvBuffer,
               vtkIdType *sendLengths, vtkIdType *offsets,
               vtkIdType recvLength, int srcProcessId) {
    return this->ScatterVVoidArray(sendBuffer, recvBuffer,
                                   sendLengths, offsets, recvLength,
                                   VTK_UNSIGNED_LONG, srcProcessId);
  }
  int ScatterV(const unsigned char *sendBuffer, unsigned char *recvBuffer,
               vtkIdType *sendLengths, vtkIdType *offsets,
               vtkIdType recvLength, int srcProcessId) {
    return this->ScatterVVoidArray(sendBuffer, recvBuffer,
                                   sendLengths, offsets, recvLength,
                                   VTK_UNSIGNED_CHAR, srcProcessId);
  }
  int ScatterV(const char *sendBuffer, char *recvBuffer,
               vtkIdType *sendLengths, vtkIdType *offsets,
               vtkIdType recvLength, int srcProcessId) {
    return this->ScatterVVoidArray(sendBuffer, recvBuffer,
                                   sendLengths, offsets, recvLength,
                                   VTK_CHAR, srcProcessId);
  }
  int ScatterV(const float *sendBuffer, float *recvBuffer,
               vtkIdType *sendLengths, vtkIdType *offsets,
               vtkIdType recvLength, int srcProcessId) {
    return this->ScatterVVoidArray(sendBuffer, recvBuffer,
                                   sendLengths, offsets, recvLength,
                                   VTK_FLOAT, srcProcessId);
  }
  int ScatterV(const double *sendBuffer, double *recvBuffer,
               vtkIdType *sendLengths, vtkIdType *offsets,
               vtkIdType recvLength, int srcProcessId) {
    return this->ScatterVVoidArray(sendBuffer, recvBuffer,
                                   sendLengths, offsets, recvLength,
                                   VTK_DOUBLE, srcProcessId);
  }
#ifdef VTK_USE_64BIT_IDS
  int ScatterV(const vtkIdType *sendBuffer, vtkIdType *recvBuffer,
               vtkIdType *sendLengths, vtkIdType *offsets,
               vtkIdType recvLength, int srcProcessId) {
    return this->ScatterVVoidArray(sendBuffer, recvBuffer,
                                   sendLengths, offsets, recvLength,
                                   VTK_ID_TYPE, srcProcessId);
  }
#endif

  // Description:
  // Same as gather except that the result ends up on all processes.
  int AllGather(const int *sendBuffer, int *recvBuffer, vtkIdType length) {
    return this->AllGatherVoidArray(sendBuffer, recvBuffer, length, VTK_INT);
  }
  int AllGather(const unsigned long *sendBuffer,
                unsigned long *recvBuffer, vtkIdType length) {
    return this->AllGatherVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_UNSIGNED_LONG);
  }
  int AllGather(const unsigned char *sendBuffer,
                unsigned char *recvBuffer, vtkIdType length) {
    return this->AllGatherVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_UNSIGNED_CHAR);
  }
  int AllGather(const char *sendBuffer, char *recvBuffer, vtkIdType length) {
    return this->AllGatherVoidArray(sendBuffer, recvBuffer, length, VTK_CHAR);
  }
  int AllGather(const float *sendBuffer, float *recvBuffer, vtkIdType length) {
    return this->AllGatherVoidArray(sendBuffer, recvBuffer, length, VTK_FLOAT);
  }
  int AllGather(const double *sendBuffer,
                double *recvBuffer, vtkIdType length) {
    return this->AllGatherVoidArray(sendBuffer, recvBuffer, length, VTK_DOUBLE);
  }
#ifdef VTK_USE_64BIT_IDS
  int AllGather(const vtkIdType *sendBuffer, vtkIdType *recvBuffer,
                vtkIdType length) {
    return this->AllGatherVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_ID_TYPE);
  }
#endif
  int AllGather(vtkDataArray *sendBuffer, vtkDataArray *recvBuffer);

  // Description:
  // Same as GatherV except that the result is placed in all processes.
  int AllGatherV(const int* sendBuffer, int* recvBuffer, 
                 vtkIdType sendLength, vtkIdType* recvLengths,
                 vtkIdType* offsets) {
    return this->AllGatherVVoidArray(sendBuffer, recvBuffer,
                                     sendLength, recvLengths,
                                     offsets, VTK_INT);
  }
  int AllGatherV(const unsigned long* sendBuffer, unsigned long* recvBuffer, 
                 vtkIdType sendLength, vtkIdType* recvLengths,
                 vtkIdType* offsets) {
    return this->AllGatherVVoidArray(sendBuffer, recvBuffer,
                                     sendLength, recvLengths,
                                     offsets, VTK_UNSIGNED_LONG);
  }
  int AllGatherV(const unsigned char* sendBuffer, unsigned char* recvBuffer, 
                 vtkIdType sendLength, vtkIdType* recvLengths,
                 vtkIdType* offsets) {
    return this->AllGatherVVoidArray(sendBuffer, recvBuffer,
                                     sendLength, recvLengths,
                                     offsets, VTK_UNSIGNED_CHAR);
  }
  int AllGatherV(const char* sendBuffer, char* recvBuffer, 
                 vtkIdType sendLength, vtkIdType* recvLengths,
                 vtkIdType* offsets) {
    return this->AllGatherVVoidArray(sendBuffer, recvBuffer,
                                     sendLength, recvLengths,
                                     offsets, VTK_CHAR);
  }
  int AllGatherV(const float* sendBuffer, float* recvBuffer, 
                 vtkIdType sendLength, vtkIdType* recvLengths,
                 vtkIdType* offsets) {
    return this->AllGatherVVoidArray(sendBuffer, recvBuffer,
                                     sendLength, recvLengths,
                                     offsets, VTK_FLOAT);
  }
  int AllGatherV(const double* sendBuffer, double* recvBuffer, 
                 vtkIdType sendLength, vtkIdType* recvLengths,
                 vtkIdType* offsets) {
    return this->AllGatherVVoidArray(sendBuffer, recvBuffer,
                                     sendLength, recvLengths,
                                     offsets, VTK_DOUBLE);
  }
#ifdef VTK_USE_64BIT_IDS
  int AllGatherV(const vtkIdType* sendBuffer, vtkIdType* recvBuffer, 
                 vtkIdType sendLength, vtkIdType* recvLengths,
                 vtkIdType* offsets) {
    return this->AllGatherVVoidArray(sendBuffer, recvBuffer,
                                     sendLength, recvLengths,
                                     offsets, VTK_ID_TYPE);
  }
#endif

  // Description:
  // Reduce an array to the given destination process.  This version of Reduce
  // takes an identifier defined in the
  // vtkCommunicator::StandardOperations enum to define the operation.
  int Reduce(const int *sendBuffer, int *recvBuffer,
             vtkIdType length, int operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_INT, operation, destProcessId);
  }
  int Reduce(const unsigned long *sendBuffer, unsigned long *recvBuffer,
             vtkIdType length, int operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_UNSIGNED_LONG, operation, destProcessId);
  }
  int Reduce(const unsigned char *sendBuffer, unsigned char *recvBuffer,
             vtkIdType length, int operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_UNSIGNED_CHAR, operation, destProcessId);
  }
  int Reduce(const char *sendBuffer, char *recvBuffer,
             vtkIdType length, int operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_CHAR, operation, destProcessId);
  }
  int Reduce(const float *sendBuffer, float *recvBuffer,
             vtkIdType length, int operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_FLOAT, operation, destProcessId);
  }
  int Reduce(const double *sendBuffer, double *recvBuffer,
             vtkIdType length, int operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_DOUBLE, operation, destProcessId);
  }
#ifdef VTK_USE_64BIT_IDS
  int Reduce(const vtkIdType *sendBuffer, vtkIdType *recvBuffer,
             vtkIdType length, int operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_ID_TYPE, operation, destProcessId);
  }
#endif
  int Reduce(vtkDataArray *sendBuffer, vtkDataArray *recvBuffer,
             int operation, int destProcessId);

  // Description:
  // Reduce an array to the given destination process.  This version of Reduce
  // takes a custom operation as a subclass of vtkCommunicator::Operation.
  int Reduce(const int *sendBuffer, int *recvBuffer,
             vtkIdType length, Operation *operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_INT, operation, destProcessId);
  }
  int Reduce(const unsigned long *sendBuffer, unsigned long *recvBuffer,
             vtkIdType length, Operation *operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_UNSIGNED_LONG, operation, destProcessId);
  }
  int Reduce(const unsigned char *sendBuffer, unsigned char *recvBuffer,
             vtkIdType length, Operation *operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_UNSIGNED_CHAR, operation, destProcessId);
  }
  int Reduce(const char *sendBuffer, char *recvBuffer,
             vtkIdType length, Operation *operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_CHAR, operation, destProcessId);
  }
  int Reduce(const float *sendBuffer, float *recvBuffer,
             vtkIdType length, Operation *operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_FLOAT, operation, destProcessId);
  }
  int Reduce(const double *sendBuffer, double *recvBuffer,
             vtkIdType length, Operation *operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_DOUBLE, operation, destProcessId);
  }
#ifdef VTK_USE_64BIT_IDS
  int Reduce(const vtkIdType *sendBuffer, vtkIdType *recvBuffer,
             vtkIdType length, Operation *operation, int destProcessId) {
    return this->ReduceVoidArray(sendBuffer, recvBuffer, length,
                                 VTK_ID_TYPE, operation, destProcessId);
  }
#endif
  int Reduce(vtkDataArray *sendBuffer, vtkDataArray *recvBuffer,
             Operation *operation, int destProcessId);

  // Description:
  // Same as Reduce except that the result is placed in all of the processes.
  int AllReduce(const int *sendBuffer, int *recvBuffer,
                vtkIdType length, int operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_INT, operation);
  }
  int AllReduce(const unsigned long *sendBuffer, unsigned long *recvBuffer,
                vtkIdType length, int operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_UNSIGNED_LONG, operation);
  }
  int AllReduce(const unsigned char *sendBuffer, unsigned char *recvBuffer,
                vtkIdType length, int operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_UNSIGNED_CHAR, operation);
  }
  int AllReduce(const char *sendBuffer, char *recvBuffer,
                vtkIdType length, int operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_CHAR, operation);
  }
  int AllReduce(const float *sendBuffer, float *recvBuffer,
                vtkIdType length, int operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_FLOAT, operation);
  }
  int AllReduce(const double *sendBuffer, double *recvBuffer,
                vtkIdType length, int operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_DOUBLE, operation);
  }
#ifdef VTK_USE_64BIT_IDS
  int AllReduce(const vtkIdType *sendBuffer, vtkIdType *recvBuffer,
                vtkIdType length, int operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_ID_TYPE, operation);
  }
#endif
  int AllReduce(vtkDataArray *sendBuffer, vtkDataArray *recvBuffer,
                int operation);
  int AllReduce(const int *sendBuffer, int *recvBuffer,
                vtkIdType length, Operation *operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_INT, operation);
  }
  int AllReduce(const unsigned long *sendBuffer, unsigned long *recvBuffer,
                vtkIdType length, Operation *operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_UNSIGNED_LONG, operation);
  }
  int AllReduce(const unsigned char *sendBuffer, unsigned char *recvBuffer,
                vtkIdType length, Operation *operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_UNSIGNED_CHAR, operation);
  }
  int AllReduce(const char *sendBuffer, char *recvBuffer,
                vtkIdType length, Operation *operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_CHAR, operation);
  }
  int AllReduce(const float *sendBuffer, float *recvBuffer,
                vtkIdType length, Operation *operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_FLOAT, operation);
  }
  int AllReduce(const double *sendBuffer, double *recvBuffer,
                vtkIdType length, Operation *operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_DOUBLE, operation);
  }
#ifdef VTK_USE_64BIT_IDS
  int AllReduce(const vtkIdType *sendBuffer, vtkIdType *recvBuffer,
                vtkIdType length, Operation *operation) {
    return this->AllReduceVoidArray(sendBuffer, recvBuffer, length,
                                    VTK_ID_TYPE, operation);
  }
#endif
  int AllReduce(vtkDataArray *sendBuffer, vtkDataArray *recvBuffer,
                Operation *operation);

  // Description:
  // Subclasses should reimplement these if they have a more efficient
  // implementation.
  virtual int BroadcastVoidArray(void *data, vtkIdType length, int type,
                                 int srcProcessId);
  virtual int GatherVoidArray(const void *sendBuffer, void *recvBuffer,
                              vtkIdType length, int type, int destProcessId);
  virtual int GatherVVoidArray(const void *sendBuffer, void *recvBuffer,
                               vtkIdType sendLength, vtkIdType *recvLengths,
                               vtkIdType *offsets, int type, int destProcessId);
  virtual int ScatterVoidArray(const void *sendBuffer, void *recvBuffer,
                               vtkIdType length, int type, int srcProcessId);
  virtual int ScatterVVoidArray(const void *sendBuffer, void *recvBuffer,
                                vtkIdType *sendLengths, vtkIdType *offsets,
                                vtkIdType recvLength, int type,
                                int srcProcessId);
  virtual int AllGatherVoidArray(const void *sendBuffer, void *recvBuffer,
                                 vtkIdType length, int type);
  virtual int AllGatherVVoidArray(const void *sendBuffer, void *recvBuffer,
                                  vtkIdType sendLength, vtkIdType *recvLengths,
                                  vtkIdType *offsets, int type);
  virtual int ReduceVoidArray(const void *sendBuffer, void *recvBuffer,
                              vtkIdType length, int type,
                              int operation, int destProcessId);
  virtual int ReduceVoidArray(const void *sendBuffer, void *recvBuffer,
                              vtkIdType length, int type,
                              Operation *operation, int destProcessId);
  virtual int AllReduceVoidArray(const void *sendBuffer, void *recvBuffer,
                                 vtkIdType length, int type,
                                 int operation);
  virtual int AllReduceVoidArray(const void *sendBuffer, void *recvBuffer,
                                 vtkIdType length, int type,
                                 Operation *operation);

  static void SetUseCopy(int useCopy);

  // Description:
  // Determine the global bounds for a set of processes.  BBox is 
  // initially set (outside of the call to the local bounds of the process 
  // and will be modified to be the global bounds - this default implementation
  // views the processors as a heap tree with the root being processId = 0
  // If either rightHasBounds or leftHasBounds is not 0 then the 
  // corresponding int will be set to 1 if the right/left processor has
  // bounds else it will be set to 0
  // The last three arguements are the tags to be used when performing
  // the operation
  virtual int ComputeGlobalBounds(int processorId, int numProcesses,
                                  vtkBoundingBox *bounds,
                                  int *rightHasBounds = 0,
                                  int *leftHasBounds = 0,
                                  int hasBoundsTag = 288402, 
                                  int localBoundsTag = 288403,
                                  int globalBoundsTag = 288404);

  // Description: 
  // Some helper functions when dealing with heap tree - based
  // algorthims - we don't need a function for getting the right
  // processor since it is 1 + theLeftProcessor
  static int GetParentProcessor(int pid);
  static int GetLeftChildProcessor(int pid);

  // Description:
  // Convert a data object into a string that can be transmitted and vice versa.
  // Returns 1 for success and 0 for failure.
  // WARNING: This will only work for types that have a vtkDataWriter class.
  static int MarshalDataObject(vtkDataObject *object, vtkCharArray *buffer);
  static int UnMarshalDataObject(vtkCharArray *buffer, vtkDataObject *object);

protected:
  
  int WriteDataArray(vtkDataArray *object);
  int ReadDataArray(vtkDataArray *object);

  vtkCommunicator();
  ~vtkCommunicator();

  // Internal methods called by Send/Receive(vtkDataObject *... ) above.
  int SendElementalDataObject(vtkDataObject* data, int remoteHandle, int tag);
  int SendMultiBlockDataSet(vtkMultiBlockDataSet* data, int remoteHandle, int tag);
  int SendTemporalDataSet(vtkTemporalDataSet* data, int remoteHandle, int tag);
  int ReceiveDataObject(vtkDataObject* data, 
                        int remoteHandle, int tag, int type=-1);
  int ReceiveElementalDataObject(vtkDataObject* data, 
                                 int remoteHandle, int tag);
  int ReceiveMultiBlockDataSet(
    vtkMultiBlockDataSet* data, int remoteHandle, int tag);
  int ReceiveTemporalDataSet(
    vtkTemporalDataSet* data, int remoteHandle, int tag);

  int MaximumNumberOfProcesses;
  int NumberOfProcesses;

  int LocalProcessId;

  static int UseCopy;

private:
  vtkCommunicator(const vtkCommunicator&);  // Not implemented.
  void operator=(const vtkCommunicator&);  // Not implemented.
};

#endif // __vtkCommunicator_h


