/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkTickPlacer3D.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkTickPlacer3D.h"

#include "vtkBitArray.h"
#include "vtkCamera.h"
#include "vtkCoordinate.h"
#include "vtkDoubleArray.h"
#include "vtkIntArray.h"
#include "vtkObjectFactory.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkStringArray.h"
#include "vtkTextProperty.h"
#include "vtkTextRenderer.h"
#include "vtkVectorOperators.h"

#include "vtksys/RegularExpression.hxx"

#include <cassert>
#include <cmath>
#include <cstdio>
#include <iomanip>
#include <sstream>

// We need to use _snprintf and monkey with the exponent format on older MSVC:
#if defined(_MSC_VER) && _MSC_VER < 1900
#define NEED_MSVC_SNPRINTF_HACK
#endif

// Define for verbose debugging output.
//#define TB3D_DEBUG

#ifdef TB3D_DEBUG
#define DBG_OUT(x) std::cerr << x << "\n"
#else
#define DBG_OUT(x)
#endif

vtkStandardNewMacro(vtkTickPlacer3D)
vtkCxxSetObjectMacro(vtkTickPlacer3D, CustomTickValues, vtkDoubleArray)
vtkCxxSetObjectMacro(vtkTickPlacer3D, TextProperty, vtkTextProperty)
vtkCxxSetObjectMacro(vtkTickPlacer3D, Renderer, vtkRenderer)

namespace {

double truncToOneSigFig(double value)
{
  if (std::fabs(value) < 1e-15)
  {
    return 0.;
  }
  double order = std::floor(std::log10(std::fabs(value)));
  double factor = std::pow(10, order);
  return std::floor(value / factor) * factor;
}

} // end anon namespace

//------------------------------------------------------------------------------
void vtkTickPlacer3D::PrintSelf(std::ostream &os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::Update()
{
  if (!this->TextProperty)
  {
    this->ClearResults();
    vtkErrorMacro("No text property set.");
    return;
  }

  if (!this->Renderer)
  {
    this->ClearResults();
    vtkErrorMacro("Renderer not set!");
    return;
  }

  // Ensure that a vtkTextRenderer override class is available:
  if (!vtkTextRenderer::GetInstance())
  {
    vtkErrorMacro("Cannot locate an implementation of vtkTextRenderer. "
                  "(Linking to the vtkRenderingFreeType library is the "
                  "standard way to resolve this issue).");
    this->ClearResults();
    return;
  }

  int renDPI = this->Renderer->GetRenderWindow()
      ? this->Renderer->GetRenderWindow()->GetDPI()
      : 72;

  if (this->BuildTime > this->GetMTime() &&
      this->BuildTime > this->Renderer->GetMTime() &&
      this->BuildTime > this->Renderer->GetActiveCamera()->GetMTime() &&
      this->DPI == renDPI)
  {
    DBG_OUT("Up to date.");
    return;
  }

  this->DPI = renDPI;

  DBG_OUT("###  Rebuilding.  ###");

  this->ClearResults();
  this->Prepare();
  this->PopulateResults();
  this->PruneOverlaps();

  this->BuildTime.Modified();
}

//------------------------------------------------------------------------------
vtkIdType vtkTickPlacer3D::GetNumberOfTicks()
{
  return this->TickValues->GetNumberOfTuples();
}

//------------------------------------------------------------------------------
vtkTickPlacer3D::vtkTickPlacer3D()
  : MaximumNumberOfTicks(0),
    Notation(Standard),
    Precision(2),
    Format(NULL),
    Renderer(NULL),
    TextProperty(NULL),
    CustomTickValues(NULL),
    DPI(0),
    FirstTick(0.),
    Delta(-1.)
{
  this->Range[0] = 0.;
  this->Range[1] = 0.;
  this->SetFormat("%g");
  this->Converter->SetCoordinateSystemToWorld();
}

//------------------------------------------------------------------------------
vtkTickPlacer3D::~vtkTickPlacer3D()
{
  this->SetTextProperty(NULL);
  this->SetRenderer(NULL);
  this->SetCustomTickValues(NULL);
  this->SetFormat(NULL);
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::Prepare()
{
  // Do coordinate conversions:
  this->Converter->SetViewport(this->Renderer);
  this->Converter->SetValue(this->StartPoint.GetData());
  int *tmpIv = this->Converter->GetComputedDisplayValue(this->Renderer);
  this->StartPixel.Set(static_cast<double>(tmpIv[0]),
                       static_cast<double>(tmpIv[1]));

  this->Converter->SetValue(this->EndPoint.GetData());
  tmpIv = this->Converter->GetComputedDisplayValue(this->Renderer);
  this->EndPixel.Set(static_cast<double>(tmpIv[0]),
                     static_cast<double>(tmpIv[1]));

  this->WCVector = this->EndPoint - this->StartPoint;
  this->DCVector = this->EndPixel - this->StartPixel;

  DBG_OUT("Range: " << this->Range[0] << " " << this->Range[1] << "\n"
          << "StartWC: " << this->StartPoint[0] << " " << this->StartPoint[1]
          << " " << this->StartPoint[2] << "\n"
          << "EndWC: " << this->EndPoint[0] << " " << this->EndPoint[1]
          << " " << this->EndPoint[2] << "\n"
          << "WCVector: " << this->WCVector[0] << " " << this->WCVector[1]
          << " " << this->WCVector[2] << "\n"
          << "WCVecLen: " << this->WCVector.Norm() << "\n"
          << "StartDC: " << this->StartPixel[0]
          << " " << this->StartPixel[1] << "\n"
          << "EndDC: " << this->EndPixel[0]
          << " " << this->EndPixel[1] << "\n"
          << "DCVector: " << this->DCVector[0]
          << " " << this->DCVector[1] << "\n"
          << "DCVecLen: " << this->DCVector.Norm());

  this->GetTickSpecification();
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::GetTickSpecification()
{
  // If custom tick values are specified, no need to look at range, etc:
  if (this->CustomTickValues != NULL)
  {
    return;
  }

  // Try to get an idea of how large the rendered labels will be, which largely
  // depends on format settings and the magnitude of the displayed values.
  // Look at a few key points of the range, and check which formatted string
  // is longest. The longest string's rendered bounds are used to determine
  // tick spacing.
  double dist = this->Range[1] - this->Range[0];
  vtkStdString samples[5] = {
    this->FormatScalar(truncToOneSigFig(this->Range[0])),
    this->FormatScalar(truncToOneSigFig(this->Range[0] + dist * 0.1)),
    this->FormatScalar(truncToOneSigFig(this->Range[0] + dist * 0.2)),
    this->FormatScalar(truncToOneSigFig(this->Range[0] + dist * 0.5)),
    this->FormatScalar(truncToOneSigFig(this->Range[1]))
  };
  std::size_t longestSample = 0;
  {
    std::size_t sampleSize = samples[0].size();
    for (std::size_t i = 1; i < 5; ++i)
    {
      if (samples[i].size() > sampleSize)
      {
        sampleSize = samples[i].size();
        longestSample = i;
      }
    }
  }
  vtkRecti sampleRect = this->GetTextBoundingRect(samples[longestSample]);
  DBG_OUT("Sample: '" << samples[longestSample] << "' Bounds: "
          << sampleRect[0] << " " << sampleRect[1] << " "
          << sampleRect[2] << " " << sampleRect[3]);

  // Scale the display-space vector so either its x component or y component
  // match the width/height of the rendered text. This represents the display
  // space offset if we were to tightly pack these labels along the vector.
  // Compute scaling factors that will make the x/y components match the
  // image width/height. The smaller vector will be the one we want:
  double xVecScaleFactor = std::fabs(sampleRect.GetWidth() /
                                     this->DCVector.GetX());
  double yVecScaleFactor = std::fabs(sampleRect.GetHeight() /
                                     this->DCVector.GetY());
  // 1.2 gives the labels a bit of breathing room.
  double offsetScaleFactor = std::min(xVecScaleFactor, yVecScaleFactor) * 1.2;
  vtkVector2d offsetVec = this->DCVector * offsetScaleFactor;
  DBG_OUT("OffsetVec: " << offsetVec[0] << " " << offsetVec[1]);

  // The maximum number of labels are determined by the ratio of offsetVector
  // to the display-coord vector:
  double numLabels = std::floor(this->DCVector.Norm() / offsetVec.Norm());

  if (this->MaximumNumberOfTicks > 0 &&
      static_cast<vtkIdType>(numLabels) > this->MaximumNumberOfTicks)
  {
    numLabels = static_cast<double>(this->MaximumNumberOfTicks);
  }

  // Figure out a 'nice' scalar delta for the
  if (numLabels < 1.)
  { // Don't place intermediate labels if we don't have space for them:
    this->Delta = -1.;
    DBG_OUT("No room for intermediate labels.");
    return;
  }

  this->Delta = dist / numLabels;

  // 'Nice-ify' the delta using the same technique as vtkAxis.
  // The order of magnitude for the delta value:
  double dOrder = std::floor(std::log10(this->Delta));
  double dFactor = pow(10., dOrder);

  // Round the delta to a single sigfig, always rounding up to 1, 2, 5, or 10:
  this->Delta /= dFactor;
  if (this->Delta <= 1.0)
  {
    this->Delta = 1.0;
  }
  else if (this->Delta <= 2.0)
  {
    this->Delta = 2.0;
  }
  else if (this->Delta <= 5.0)
  {
    this->Delta = 5.0;
  }
  else
  {
    this->Delta = 10.0;
  }
  this->Delta *= dFactor;

  // Now compute the first tick mark inside the range. This is the first value
  // greater than Range[0] where val%delta == 0.
  this->FirstTick = (std::floor(this->Range[0] / this->Delta) + 1.) *
      this->Delta;

  DBG_OUT("Range: " << this->Range[0] << " --> " << this->Range[1] << " "
          << "Delta: " << this->Delta << " FirstTick: " << this->FirstTick);
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::ClearResults()
{
  this->TickPositions->Reset();
  this->TickValues->Reset();
  this->TickLabels->Reset();
  this->TickLabelMask->Reset();
  this->TickLabelBounds->Reset();
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::PopulateResults()
{
  this->TickPositions->SetNumberOfComponents(1);
  this->TickValues->SetNumberOfComponents(1);
  this->TickLabels->SetNumberOfComponents(1);
  this->TickLabelMask->SetNumberOfComponents(1);
  this->TickLabelBounds->SetNumberOfComponents(4);

  // Just add the custom tick values if set:
  if (this->CustomTickValues)
  {
    double valueRange = this->Range[1] - this->Range[0];
    vtkIdType numTicks = this->CustomTickValues->GetNumberOfTuples();
    for (vtkIdType i = 0; i < numTicks; ++i)
    {
      double value = this->CustomTickValues->GetTypedComponent(i, 0);
      if (value < this->Range[0] || value > this->Range[1])
      {
        continue;
      }

      double fraction = (value - this->Range[0]) / valueRange;
      this->AddTick(value, this->StartPoint + (this->WCVector * fraction));
    }
    return;
  }

  // The first endpoint label:
  this->AddTick(this->Range[0], this->StartPoint);

  if (this->Delta > 0.)
  {
    // WC offset between tick positions:
    vtkVector3d offset = this->WCVector *
        (this->Delta / (this->Range[1] - this->Range[0]));
    // WC of first tick position:
    vtkVector3d tickPos = this->StartPoint + this->WCVector *
        ((this->FirstTick - this->Range[0]) /
         (this->Range[1] - this->Range[0]));
    // Scalar value at first tick
    double tickVal = this->FirstTick;

    // Add intermediate tick marks:
    while (tickVal < this->Range[1])
    {
      this->AddTick(tickVal, tickPos);
      tickVal += this->Delta;
      tickPos = tickPos + offset;
    }
  }

  // Add last endpoint label:
  this->AddTick(this->Range[1], this->EndPoint);
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::AddTick(double scalarValue, const vtkVector3d &posWC)
{
  this->Converter->SetValue(const_cast<double*>(posWC.GetData()));
  vtkVector2i posDC(this->Converter->GetComputedDisplayValue(this->Renderer));

  vtkStdString label = this->FormatScalar(scalarValue);
  vtkRecti bounds = this->GetTextBoundingRect(label);
  bounds.Translate(posDC);

  double fraction = (scalarValue - this->Range[0]) /
      (this->Range[1] - this->Range[0]);

  this->TickPositions->InsertNextTypedTuple(&fraction);
  this->TickValues->InsertNextTypedTuple(&scalarValue);
  this->TickLabels->InsertNextValue(label.c_str());
  this->TickLabelMask->InsertNextValue(1);
  this->TickLabelBounds->InsertNextTypedTuple(bounds.GetData());

  DBG_OUT("Tick added at scalar position: " << label);
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::RemoveTick(vtkIdType idx)
{
  DBG_OUT("Tick removed at scalar position: "
          << this->TickLabels->GetValue(idx));

  assert(idx < this->GetNumberOfTicks());
  this->TickLabelMask->SetValue(idx, 0);
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::PruneOverlaps()
{
  // No pruning for custom labels:
  if (this->CustomTickValues)
  {
    return;
  }

  vtkIdType numTicks = this->GetNumberOfTicks();
  // Test first, then last, then all in between.
  this->PruneOverlappingLabels(0, true);
  this->PruneOverlappingLabels(numTicks - 1, true);

  for (vtkIdType keeper = 1; keeper < numTicks - 1; ++keeper)
  {
    this->PruneOverlappingLabels(keeper, false);
  }
}

//------------------------------------------------------------------------------
void vtkTickPlacer3D::PruneOverlappingLabels(vtkIdType keeper, bool isEndpoint)
{
  if (this->TickLabelMask->GetValue(keeper) == 0)
  { // Already pruned
    return;
  }

  // Endpoints get checked against all other labels. Non-endpoints only
  // get checked against those with higher indices.
  vtkIdType cur = isEndpoint ? 0 : keeper + 1;
  vtkIdType end = this->GetNumberOfTicks();

  vtkRecti curRect;
  vtkRecti keeperRect;
  this->TickLabelBounds->GetTypedTuple(keeper, keeperRect.GetData());

  for (; cur < end; ++cur)
  {
    if (cur == keeper)
    {
      continue;
    }

    if (this->TickLabelMask->GetValue(cur) == 0)
    { // Already pruned
      continue;
    }

    // Test for overlap:
    this->TickLabelBounds->GetTypedTuple(cur, curRect.GetData());
    if (keeperRect.IntersectsWith(curRect))
    {
      this->RemoveTick(cur);
    }
  }
}

//------------------------------------------------------------------------------
// Adapted from vtkAxis::GenerateSimpleLabel
vtkStdString vtkTickPlacer3D::FormatScalar(double value) const
{
  // If value is near zero and least an order of magnitude smaller than the
  // range, replace value with exactly 0.:
  if (std::fabs(value) < 1e-15 &&
      this->Range[1] - this->Range[0] > std::fabs(value) * 10.)
  {
    value = 0.;
  }

  vtkStdString result;
  if (this->Notation == Printf)
  {
    const std::size_t bufferSize = 256;
    char buffer[bufferSize];

    // Force the two digit exponent on MSVC. See vtkAxis::GenerateSprintfLabel.
#ifdef NEED_MSVC_SPRINTF_HACK
    unsigned int oldExpFormat = _set_output_format(_TWO_DIGIT_EXPONENT);
    _snprintf(buffer, bufferSize - 1, this->Format, value);
    buffer[bufferSize - 1] = '\0';
    _set_output_format(oldExpFormat);
#else
    std::snprintf(buffer, bufferSize, this->Format, value);
#endif

    result = vtkStdString(buffer);
  }
  else // format via stringstream:
  {
    std::ostringstream str;
    str.imbue(std::locale::classic());
    if (this->Notation != Standard)
    {
      str.precision(this->Precision);
      if (this->Notation == Scientific)
      {
        str.setf(std::ios::scientific, std::ios::floatfield);
      }
      else if (this->Notation == Fixed)
      {
        str.setf(std::ios::fixed, std::ios::floatfield);
      }
    }
    str << value;
    result = str.str();
  }

  // Strip out leading zeros on the exponent:
  vtksys::RegularExpression regExp("[Ee][+-]");
  if (regExp.find(result))
  {
    vtkStdString::iterator it = result.begin() + regExp.start() + 2;
    while (it != result.end() && *it == '0')
    {
      it = result.erase(it);
    }

    // If the exponent is 0, remove the e+ bit, too.
    if (it == result.end())
    {
      result.erase(regExp.start());
    }
  }

  return result;
}

//------------------------------------------------------------------------------
vtkRecti vtkTickPlacer3D::GetTextBoundingRect(const vtkStdString &str)
{
  vtkTextRenderer *tren = vtkTextRenderer::GetInstance();
  vtkTextRenderer::Metrics metrics;
  tren->GetMetrics(this->TextProperty, str, metrics, this->DPI);
  return vtkRecti(metrics.BoundingBox[0], metrics.BoundingBox[2],
                  metrics.BoundingBox[1] - metrics.BoundingBox[0] + 1,
                  metrics.BoundingBox[3] - metrics.BoundingBox[2] + 1);
}
