/*****************************************************************************
*
* Copyright (c) 2000 - 2017, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                            avtRAGEFileFormat.C                           //
// ************************************************************************* //

#include <avtRAGEFileFormat.h>

#include <vtkFloatArray.h>
#include <vtkDoubleArray.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnstructuredGrid.h>

#include <avtDatabaseMetaData.h>

#include <DBOptionsAttributes.h>
#include <Expression.h>

#include <ImproperUseException.h>
#include <InvalidVariableException.h>
#include <InvalidFilesException.h>

#include <df.h>
#include <VisItStreamUtil.h>

#include <locale>

using std::string;
using std::vector;
using std::map;
using std::pair;
using std::locale;

template <class T> static void
swapXY(vtkDataArray *arr, int nx, int ny, T *vals);

static bool
extractStr(const string &str, const string &key, string &val);

static bool
extractNum(const string &str, const string &key, double &val);


// ****************************************************************************
//  Method: avtRAGEFileFormat constructor
//
//  Programmer: pugmire -- generated by xml2avt
//  Creation:   Mon Oct 20 14:11:41 PST 2014
//
// ****************************************************************************

avtRAGEFileFormat::avtRAGEFileFormat(const char *filename)
    : avtSTSDFileFormat(filename)
{
    initialized = false;
    time = 0.0;
    cycle = 0;
}


// ****************************************************************************
//  Method: avtRAGEFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: pugmire -- generated by xml2avt
//  Creation:   Mon Oct 20 14:11:41 PST 2014
//
// ****************************************************************************

void
avtRAGEFileFormat::FreeUpResources()
{
}

//****************************************************************************
// Method:  avtRAGEFileFormat::Initialize()
//
// Purpose:
//   
//
// Programmer:  Dave Pugmire
// Creation:    October 20, 2014
//
// Modifications:
//
//****************************************************************************

void
avtRAGEFileFormat::Initialize()
{
    if (initialized)
        return;

    DFSDrestart();
    int nDims, tag=700;
    int32 dimSizes[3];

    while (DFSDgetdims(GetFilename(), &nDims, dimSizes, 3) != -1)
    {
        meshInfo mi;
        varInfo vi;
        for (int i = 0; i < nDims; i++)
            mi.dimNum.push_back(dimSizes[i]);

        DFSDgetNT(&vi.dataType);
        
        int lastRef = DFSDlastref();
        int dLen = DFANgetdesclen(GetFilename(), tag, lastRef);
        char *str = new char[dLen+1];
        if (DFANgetdesc(GetFilename(), tag, lastRef, str, dLen) == -1)
            EXCEPTION1(InvalidFilesException, GetFilename());
        
        str[dLen] = '\0';
        string descr(str);
        double xmin, xmax, ymin, ymax, zmin=0.0, zmax=0.0;
        string var;
        
        extractNum(descr, "time", time);
        extractNum(descr, "xmin", xmin);
        extractNum(descr, "xmax", xmax);
        extractNum(descr, "ymin", ymin);
        extractNum(descr, "ymax", ymax);
        mi.dimMin.push_back(xmin);
        mi.dimMax.push_back(xmax);
        mi.dimMin.push_back(ymin);
        mi.dimMax.push_back(ymax);
        if (nDims == 3)
        {
            extractNum(descr, "zmin", zmin);
            extractNum(descr, "zmax", zmax);
            mi.dimMin.push_back(zmin);
            mi.dimMax.push_back(zmax);
        }
        extractStr(descr, "pioName", var);

        string meshNm = "mesh_" + var;
        meshes[meshNm] = mi;
        vi.meshNm = meshNm;
        vars[var] = vi;
        
        delete [] str;
    }

    //parse out the cycle
    string f = GetFilename();
    locale loc;
    size_t i;
    for (i = f.size()-1; i > 0; i--)
        if (!isdigit(f[i], loc))
            break;
    string meow = f.substr(i+1, string::npos);
    cycle = atoi(f.substr(i+1, string::npos).c_str());
    
    initialized = true;
}


// ****************************************************************************
//  Method: avtRAGEFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: pugmire -- generated by xml2avt
//  Creation:   Mon Oct 20 14:11:41 PST 2014
//
// ****************************************************************************

void
avtRAGEFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md)
{
    Initialize();
    
    map<string, meshInfo>::iterator mit;
    for (mit = meshes.begin(); mit != meshes.end(); mit++)
    {
        int nDims = mit->second.dimNum.size();
        avtMeshMetaData *mesh = new avtMeshMetaData;
        mesh->name = mit->first;
        mesh->meshType = AVT_RECTILINEAR_MESH;
        mesh->spatialDimension = nDims;
        mesh->topologicalDimension = nDims;

        mesh->xUnits = "cm";
        mesh->yUnits = "cm";
        mesh->zUnits = "cm";

        md->Add(mesh);
    }

    map<string, varInfo>::iterator vit;
    for (vit = vars.begin(); vit != vars.end(); vit++)
        AddScalarVarToMetaData(md, vit->first, vit->second.meshNm, AVT_ZONECENT);
}


// ****************************************************************************
//  Method: avtRAGEFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: pugmire -- generated by xml2avt
//  Creation:   Mon Oct 20 14:11:41 PST 2014
//
// ****************************************************************************

vtkDataSet *
avtRAGEFileFormat::GetMesh(const char *meshname)
{
    Initialize();

    map<string, meshInfo>::iterator mi = meshes.find(meshname);
    if (mi == meshes.end())
        return NULL;

    vtkFloatArray *coords[3] = {vtkFloatArray::New(),
                                vtkFloatArray::New(),
                                vtkFloatArray::New()};

    int dims[3] = {1,1,1};
    int nDims = mi->second.dimNum.size();
    for (int i = 0; i < nDims; i++)
    {
        int sz = mi->second.dimNum[i] + 1;
        float x0 = mi->second.dimMin[i], x1 = mi->second.dimMax[i];
        float dx = (x1-x0)/float(sz-1);
        coords[i]->SetNumberOfTuples(sz);
        dims[i] = sz;
       
        float x = x0;
        for (int j = 0; j < sz; j++, x+=dx)
            coords[i]->SetTuple1(j, x);
    }
    
    if (nDims < 3)
    {
        coords[2]->SetNumberOfTuples(1);
        coords[2]->SetTuple1(0,0);
    }
    if (nDims < 2)
    {
        coords[1]->SetNumberOfTuples(1);
        coords[1]->SetTuple1(0,0);
    }

    vtkRectilinearGrid *rv = vtkRectilinearGrid::New();
    rv->SetDimensions(dims);
    rv->SetXCoordinates(coords[0]);
    rv->SetYCoordinates(coords[1]);
    rv->SetZCoordinates(coords[2]);
    coords[0]->Delete();
    coords[1]->Delete();
    coords[2]->Delete();

    return rv;
}


// ****************************************************************************
//  Method: avtRAGEFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      varname    The name of the variable requested.
//
//  Programmer: pugmire -- generated by xml2avt
//  Creation:   Mon Oct 20 14:11:41 PST 2014
//
// ****************************************************************************

vtkDataArray *
avtRAGEFileFormat::GetVar(const char *varname)
{
    Initialize();
    
    map<string, varInfo>::iterator vit = vars.find(varname);
    if (vit == vars.end())
        EXCEPTION1(InvalidVariableException, varname);
    map<string, meshInfo>::iterator mit = meshes.find(vit->second.meshNm);
    if (mit == meshes.end())
        EXCEPTION1(InvalidVariableException, varname);

    int nDims = mit->second.dimNum.size();
    int32 dims[3] = {1,1,1};
    
    for (int i = 0; i < nDims; i++)
        dims[i] = mit->second.dimNum[i];
    
    int numTuples = dims[0]*dims[1]*dims[2];
    vtkDataArray *arr = NULL;
    if (vit->second.dataType == DFNT_FLOAT32)
        arr = vtkFloatArray::New();
    else if (vit->second.dataType == DFNT_FLOAT64)
        arr = vtkDoubleArray::New();
    else
        EXCEPTION1(InvalidVariableException, varname);
    
    arr->SetNumberOfTuples(numTuples);
    
    if (DFSDgetdata(GetFilename(), nDims, dims, arr->GetVoidPointer(0)) == -1)
        EXCEPTION1(InvalidVariableException, varname);
    
    return arr;
}


// ****************************************************************************
//  Method: avtRAGEFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      varname    The name of the variable requested.
//
//  Programmer: pugmire -- generated by xml2avt
//  Creation:   Mon Oct 20 14:11:41 PST 2014
//
// ****************************************************************************

vtkDataArray *
avtRAGEFileFormat::GetVectorVar(const char *varname)
{
    Initialize();
    return NULL;
}

//****************************************************************************
// Method:  swapXY
//
// Purpose:
//   Swap the XY order of the data.
//
// Programmer:  Dave Pugmire
// Creation:    November  3, 2014
//
// Modifications:
//
//****************************************************************************

template <class T> static void
swapXY(vtkDataArray *arr, int nx, int ny, T *vals)
{
    int idx = 0;
    for (int i = 0; i < nx; i++)
        for (int j = 0; j < ny; j++)
            arr->SetTuple1(idx++, vals[i*ny+j]);
}

//****************************************************************************
// Method:  extractStr
//
// Purpose:
//   Swap the XY order of the data.
//
// Programmer:  Dave Pugmire
// Creation:    November  24, 2014
//
// Modifications:
//
//****************************************************************************

static bool
extractStr(const string &str, const string &key, string &val)
{
    size_t i = str.find(key);
    if (i == string::npos)
        return false;
    
    string sub = str.substr(i+key.size(), string::npos);
    char a[128], b[128];
    sscanf(sub.c_str(), "%s %s", a, b);
    val.resize(strlen(a)-2);
    for (int i = 1; i < strlen(a)-1; i++)
        val[i-1] = a[i];

    //cout<<key<<" --> "<<val<<endl;
    return true;
}

//****************************************************************************
// Method:  extractNum
//
// Purpose:
//   Swap the XY order of the data.
//
// Programmer:  Dave Pugmire
// Creation:    November  24, 2014
//
// Modifications:
//
//****************************************************************************

static bool
extractNum(const string &str, const string &key, double &val)
{
    size_t i = str.find(key);
    if (i == string::npos)
        return false;
    
    string sub = str.substr(i+key.size(), string::npos);
    char a[128];
    sscanf(sub.c_str(), "%lf %s", &val, a);
    
    //cout<<key<<" --> "<<val<<endl;
    return true;
}

