// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
// .NAME Test of vtkBMPReader
// .SECTION Description
//

#include "vtkBMPReader.h"
#include "vtkImageData.h"
#include "vtkImageMapToColors.h"
#include "vtkImageViewer.h"
#include "vtkLookupTable.h"
#include "vtkRegressionTestImage.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkRenderer.h"
#include "vtkSmartPointer.h"

#include <iostream>

int TestBMPReader(int argc, char* argv[])
{

  if (argc <= 1)
  {
    std::cout << "Usage: " << argv[0] << " <bmp file>" << std::endl;
    return EXIT_FAILURE;
  }

  std::string filename = argv[1];

  vtkSmartPointer<vtkBMPReader> BMPReader = vtkSmartPointer<vtkBMPReader>::New();

  // Check the image can be read
  if (!BMPReader->CanReadFile(filename.c_str()))
  {
    std::cerr << "CanReadFile failed for " << filename << "\n";
    return EXIT_FAILURE;
  }

  // Read the input image
  BMPReader->SetFileName(filename.c_str());
  BMPReader->Update();

  // Read and display the image properties
  int depth = BMPReader->GetDepth();
  std::cout << "depth: " << depth << std::endl;

  const char* fileExtensions = BMPReader->GetFileExtensions();
  std::cout << "fileExtensions: " << fileExtensions << std::endl;

  const char* descriptiveName = BMPReader->GetDescriptiveName();
  std::cout << "descriptiveName: " << *descriptiveName << std::endl;

  vtkSmartPointer<vtkLookupTable> lookupTable = BMPReader->GetLookupTable();
  lookupTable->Print(std::cout);

  const unsigned char* colors = BMPReader->GetColors();
  unsigned char const* first = reinterpret_cast<unsigned char*>(&colors);
  unsigned char const* last = reinterpret_cast<unsigned char*>(&colors + 1);
  std::cout << "colors: ";
  while (first != last)
  {
    std::cout << (int)*first << ' ';
    ++first;
  }
  std::cout << std::endl;

  int allow8BitBMP = 1;
  BMPReader->SetAllow8BitBMP(allow8BitBMP);
  std::cout << "allow8BitBMP: " << BMPReader->GetAllow8BitBMP() << std::endl;

  // Visualize
  vtkSmartPointer<vtkImageMapToColors> map = vtkSmartPointer<vtkImageMapToColors>::New();
  map->SetInputConnection(BMPReader->GetOutputPort());
  map->SetLookupTable(BMPReader->GetLookupTable());
  map->SetOutputFormatToRGB();

  vtkSmartPointer<vtkImageViewer> imageViewer = vtkSmartPointer<vtkImageViewer>::New();
  imageViewer->SetInputConnection(map->GetOutputPort());
  imageViewer->SetColorWindow(256);
  imageViewer->SetColorLevel(127.5);

  vtkNew<vtkRenderWindowInteractor> renderWindowInteractor;
  imageViewer->SetupInteractor(renderWindowInteractor);
  imageViewer->Render();

  vtkRenderWindow* renWin = imageViewer->GetRenderWindow();
  int retVal = vtkRegressionTestImage(renWin);
  if (retVal == vtkRegressionTester::DO_INTERACTOR)
  {
    renderWindowInteractor->Start();
  }

  return !retVal;
}
