#=============================================================================
#
#  Copyright (c) Kitware, Inc.
#  All rights reserved.
#  See LICENSE.txt for details.
#
#  This software is distributed WITHOUT ANY WARRANTY; without even
#  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the above copyright notice for more information.
#
#=============================================================================
"""
Export script for Shallow Water 2D applications
"""
import imp
import os
import sys
import smtk

# so devs don't complain about pyc files in source
sys.dont_write_bytecode = True

# Explicitly load adhcommon.py, so that it reloads each time
module_name = 'adhcommon'
abs_path = os.path.abspath(__file__)
abs_dir = os.path.dirname(abs_path)
module_args = imp.find_module(module_name, [abs_dir])
imp.load_module(module_name, *module_args)
adh = sys.modules.get(module_name)

# Define placeholder/passthrough functions for custom writer functions
# Actual implementations, prefixed by "_", are further below
def write_outputseries(scope, item, card_format, context_id):
  return _write_outputseries(scope, item, card_format, context_id)

def write_hydrotol(scope, item, card_format, context_id):
  return _write_hydrotol(scope, item, card_format, context_id)

def write_genCon(scope, item, card_format, context_id):
  return _write_genCon(scope, item, card_format, context_id)

# ---------------------------------------------------------------------
#
# Dictionary of formatters for each output card
#
# Card format types are: val, idval, bc, multival
# Arguments are: (item name, opcode, comment=None, subitems=None, \
#  custom_writer=None)
#
# ---------------------------------------------------------------------
fmt = adh.CardFormat
format_table = {
  'Solvers': [
    #OperationParams
    fmt.val('MemoryIncrementBlockSize', 'OP INC', item_path='OperationParams'),
    fmt.val('PreconditioningBlocks', 'OP BLK', item_path='OperationParams'),
    fmt.val('PreconditionerType', 'OP PRE', item_path='OperationParams'),
    #TemporalSchemeBox
    fmt.val('TemporalSchemeCoefficient', 'OP TEM', item_path='TemporalSchemeBox'),
    #PetrovGalerkinBox
    fmt.val('PetrovGalerkinCoefficient', 'OP TPG', item_path='PetrovGalerkinBox'),
    #VesselMovementBox
    fmt.val('VesselMovement', 'OP BT', item_path='VesselMovementBox'),
    fmt.val('VesselEntrainment', 'OP BTS', item_path='VesselMovementBox'),
    fmt.val('SW2Gradients', 'OP NF2'),
    #IterationParams
    fmt.val('NonLinearTolMaxNorm', 'IP NTL', item_path='IterationParams'),
    fmt.val('MaxNonLinearIters', 'IP NIT', item_path='IterationParams'),
    fmt.val('MaxLinearIters', 'IP MIT', item_path='IterationParams'),
    fmt.val('NonLinearTolMaxChange', 'IP ITL', item_path='IterationParams'),
    fmt.val('RungeKuttaTol', 'IP RTL', item_path='IterationParams')
  ],

  'Time': [
    fmt.val('StartTime', 'TC T0', subitem_names=['Value']),
    fmt.val('EndTime', 'TC TF', subitem_names=['Value']),
    fmt.val('MaxParams', 'TC IDT', 
      subitem_names=['TimestepSizeSeries','InputTimeUnits'],
      item_path='TimestepSizeParameters/TimestepSizeType'),
    fmt.val('AutoParams', 'TC ATF', 
      subitem_names=['InitialTimeStep', 'TimeSeries'],
      item_path='TimestepSizeParameters/TimestepSizeType'),
    fmt.val('SteadyStateParams', 'TC STD',
      subitem_names=['InitialTimeStep', 'MaximumTimestepSize'],
      item_path='TimestepSizeParameters/TimestepSizeType'),
    fmt.val('QuasiUnsteadyParams', 'TC STH',
      subitem_names=['SteadyStateHydrodynamicCondition', 'InputTimeUnits', 'MaxIterations', 'InitialTimeStepSize'],
      item_path='TimestepSizeParameters/TimestepSizeType'),
    # Using a custom writer for hydrodynamics tolerance because it should
    # only be written if QuasiUnsteadyParams are picked and HydrodynamicsTol
    # is enabled
    fmt.val('HydrodynamicsTol', 'IP SST', item_path='TimestepSizeParameters/TimestepSizeType',
      custom_writer=write_hydrotol),
    fmt.val('PrintAdaptedMeshes', 'PC ADP'),
    fmt.val('HotStartFile', 'PC HOT'),
    fmt.val('OutputParams', 'OC', 
      subitem_names=['OutputFunction'],
      item_path='OutputSeriesType'),
    # requires a custom writer because the subitem names are part of an extensible group
    fmt.val('AutoOutputParams', 'OS', 
      subitem_names=['StartTime', 'EndTime', 'TimeInterval', 'Units'],
      item_path='OutputSeriesType',
      custom_writer=write_outputseries)
  ],

  'SWMaterial': [
    fmt.idval('KinematicEddyViscosity', 'MP EVS',
      subitem_names=['Exx', 'Eyy', 'Exy']),
    fmt.idval('CoriolisLatitude', 'MP COR'),
    #RefinementParameters
    fmt.idval('MaxRefineLevels', 'MP ML', item_path='RefinementParameters'),
    fmt.idval('HydroRefineTol', 'MP SRT', item_path='RefinementParameters'),
    #Friction Parameters
    fmt.idval('MNGParams', 'FR MNG', subitem_names=['MNGValue'],item_path='FrictionType'),
    fmt.idval('ERHParams', 'FR ERH', subitem_names=['ERHValue'],item_path='FrictionType'),
    fmt.idval('SAVParams', 'FR SAV', subitem_names=['SAVERHValue','SAVUSHValue'],item_path='FrictionType'),
    fmt.idval('URVParams', 'FR URV', subitem_names=['URVERHValue','AvgStemDiameter','AvgStemDensity'],item_path='FrictionType')
  ],

  # Boundary Conditions
  'WaterSurfElev':       
    [fmt.bc('WSEParameters', 'NB OTW',
      subitem_names=['Value'])],
  'TotalDischarge':      
    [fmt.bc('TDParameters', 'NB DIS',
      subitem_names=['Value'])],
  'VelocityNB':
    [fmt.bc('NeumannVelocity', 'NB OVL',
     subitem_names=['Value'])],
  'Rainfall/Evaporation':            
    [fmt.bc('NeumannVelocity', 'NB OVL',
      subitem_names=['Value'])],
  'VelocityDB':
    [fmt.bc('DirichletVelocity', 'DB OVL',
      subitem_names=['XValue', 'YValue'])],
  'SupercriticalInflow':
    [fmt.bc('DirichletVelocityDepth', 'DB OVH',
      subitem_names=['XValue', 'YValue', 'DValue'])],
  'LidElevation':        
    [fmt.bc('LidParameters', 'DB LDE',
      subitem_names=['Value'])],
  'WaterDepthLid':       
    [fmt.bc('WDLParameters', 'DB LDH',
      subitem_names=['Value'])],
  'FloatingStationary':  
    [fmt.bc('FSParameters', 'DB LID',
      subitem_names=['Value'])],
  'MNGParams': [fmt.idval('MNGParams', 'FR MNG',
                     subitem_names=['MNGValue'],
                     item_path='FrictionBC/FrictionType'),],
  'ERHParams': [fmt.idval('ERHParams', 'FR ERH', 
                     subitem_names=['ERHValue'],
                     item_path='FrictionBC/FrictionType'),],
  'SAVParams': [fmt.idval('SAVParams', 'FR SAV', 
                    subitem_names=['SAVERHValue','SAVUSHValue'],
                    item_path='FrictionBC/FrictionType'),],
  'URVParams': [ fmt.idval('URVParams', 'FR URV', 
                    subitem_names=['URVERHValue','AvgStemDiameter','AvgStemDensity'],
                    item_path='FrictionBC/FrictionType')],

  'Globals': [
    fmt.val('Gravity', 'MP G'),
    fmt.val('KinMolViscosity', 'MP MU'),
    fmt.val('ReferenceDensity', 'MP RHO'),
    fmt.val('ManningsUnitConstant', 'MP MUC'),
    fmt.val('WetDryLimits', 'MP DTL')
  ],

  'GeneralConstituent': [ 
    fmt.idval('ReferenceConcentration',  'CN CON'),
    fmt.idconval('DefaultTurbulentDiffRate','MP DF',
        item_path='ConstituentParams/DefaultTransportParams',
        custom_writer=write_genCon)
  ]
    #fmt.idconval('DefaultTransportRefTol', 'MP TRT',
    #    item_path='ConstituentParams/DefaultTransportParams'),
    #fmt.conbc('DefaultConstituentBC', 'TRN',
    #    subitem_names=['ConstituentBCType','Value'],
    #    item_path='ConstituentBCs')]


  #   subitem_names=['ReferenceConcentration']) ],
  # 'SalinityConstituent':    [ fmt.idval('SalConstituentParams',  'CN SAL',
  #   subitem_names=['ReferenceConcentration']) ],
  # 'VorticityConstituent':   [ fmt.idval('VortConstituentParams', 'CN VOR',
  #   subitem_names=['NormalizationFactor', 'AsTerm', 'DsTerm']) ],
  # 'TemperatureConstituent': [ fmt.idval('TempConstituentParams', 'CN TEM',
  #   subitem_names=['ReferenceConcentration']) ],
}


# ---------------------------------------------------------------------
def ExportCMB(spec):
    '''Entry function, called by CMB to write export files

    Returns boolean indicating success
    Parameters
    ----------
    spec: Top-level object passed in from CMB
    '''
    #print 'Enter ExportCMB()'

    # Initialize scope instance to store spec values and other info
    if not spec:
      print "No spec object"
      return False

    scope = adh.init_scope(spec)
    if scope.logger.hasErrors():
      print scope.logger.convertToString()
      print 'FILES NOT WRITTEN because of errors'
      return False
    scope.format_table = format_table

    print 'Analysis types:', scope.analysis_types
    if not scope.analysis_types:
      msg = 'No analysis types selected'
      print 'ERROR:', msg
      scope.logger.addError(msg)
      return False
    else:
      print 'Categories:', sorted(list(scope.categories))

    # Write mesh file
    if scope.mesh_collection is None:
      print 'WARNING: No mesh collection; cannot write .2dm file'
    else:
      mesh_filename = scope.output_filebase + '.2dm'
      mesh_path = os.path.join(scope.output_directory, mesh_filename)
      print 'Writing mesh data to', mesh_path
      status = smtk.io.exportMesh(
        mesh_path,
        scope.mesh_collection,
        scope.model.manager(),
        scope.matid_property_name)
      print 'mesh write returned status', status

    # Open output file and start exporting content
    completed = False
    bc_filename = scope.output_filebase + '.bc'
    bc_path = os.path.join(scope.output_directory, bc_filename)
    with open(bc_path, 'w') as scope.output:
      scope.output.write('OP SW2\n')
      n = len(scope.constituent_dict)
      scope.output.write('OP TRN %d\n' % n)
      # Call write-content functions in specified top-level order
      att_type_list = [
        'GeneralConstituent'
       #'Solvers', 'Globals', 'SWMaterial','Time'
      ]
      # att_type_list = [
      #   'Solvers', 'Globals', 'SWMaterial', 'BoundaryCondition',
      #   'Friction', 'Globals', 'Constituent'
      # ]
      for att_type in att_type_list:
        ok = adh.write_section(scope, att_type)
        #if not ok:
        #    break

      # Write function attributes
      adh.write_functions(scope)

      # Write MTS cards (material id for each model domain)
      adh.write_MTS_cards(scope)

      # Write NDS & EGS cards for boundary conditions
      adh.write_bc_sets(scope)

      # Last line
      scope.output.write('END\n')
      print 'Wrote', bc_path
      completed = True

    # Write hotstart file if enabled
    completed &= adh.write_hotstart(scope)

    if not completed:
      print 'WARNING: Export terminated unexpectedly -- output might be invalid.'

    return completed

#----------------------------------------------------------------------
def _write_hydrotol(scope, item, card_format, context_id):
  '''Custom writer (implementation) for HydrodynamicsTolerance attribute'''
  # Only write HydrodynamicsTolerance if TimestepType is set to QuasiUnsteadyParams
  if item.owningItem().value(0) == 'QuasiUnsteadyParams':
    if item.isEnabled():
      output_text = card_format.opcode + "  " + str(item.value(0)) + '\n'
      if output_text is not None:
        scope.output.write(output_text)



# ---------------------------------------------------------------------
def _write_outputseries(scope, item, card_format, context_id):
  '''Custom writer (implementation) for OutputSeries attribute

  Writes either OC or OS card, depending on item'd discrete value
    0 == OC (Output Control Series)
    1 == OS (Auto-Build Time Series)
  '''
  discrete_value = item.value(0)
  if discrete_value == 'AutoOutputParams':
    # Get function (series) ID for TimeSeriesData
    time_series_item = adh.find_active_child(scope, item, 'TimeSeriesData')

    # Get number of points
    n = time_series_item.numberOfGroups()

    series_id = adh.get_function_id(scope, time_series_item, 0)

    scope.output.write('OS %d %d %d\n' % (series_id, n, 0))

    # Process subgroup items, one line per
    if n < 1:  # but safety first
      return

    for i in range(n):
      #print 'i', i
      start_item = adh.find_subgroup_item(time_series_item, i, 'StartTime')
      start = start_item.value(0)

      end_item = adh.find_subgroup_item(time_series_item, i, 'EndTime')
      end = end_item.value(0)

      interval_item = adh.find_subgroup_item(time_series_item, i, 'TimeInterval')
      interval = interval_item.value(0)

      units_item = adh.find_subgroup_item(time_series_item, i, 'Units')
      units = units_item.value(0)

      scope.output.write('%8s %8s %8s   %s\n' % (start, end, interval, units))
  else:
    msg = 'Unexpected value for OutputSeries: %d (expecting 0 or 1)' % \
      discrete_value
    print 'WARNING:', msg
    scope.logger.addWarning(msg)

    # ---------------------------------------------------------------------
def _write_genCon(scope, item, card_format, context_id):
  '''Custom writer (implementation) for General Constituent attribute

  Writes either OC or OS card, depending on item'd discrete value
    0 == OC (Output Control Series)
    1 == OS (Auto-Build Time Series)
  '''
  #Need to check to see if there are any Material Specific values specified
  # Grab the Constituent Params Group
  cp = smtk.to_concrete(item.owningItem().owningItem())
  mstp = smtk.to_concrete(cp.find("MatSpecificTransportParams"))
  #Get the material specifics group
  ms = smtk.to_concrete(mstp.item(0))
  #create a list of materials that have specific transport params
  ms_mats = set()

  for i in range(ms.numberOfGroups()):
    #find the subitem in the material specifics group
    #which is the item name without Default in it
    sub_item = ms.find(item.name().lstrip('Default'))
    if sub_item:
      sub_item_pos = smtk.to_concrete(sub_item).position()
    else:
      print "Error finding subitem %s"%(item.name().lstrip('Default'))
      break


    #grab the material item from each of the groups to get the mat_index
    m = smtk.to_concrete(ms.item(i,0))
    mat_index = scope.material_list[m.value(0).id()]
    #add the material to the list that have specific transport params
    ms_mats.add(m.value(0).id())

    #get the sub item value (Turbulent Diffusion Rate or Transport Refinement Rate)
    val = smtk.to_concrete(ms.item(i,sub_item_pos))
    scope.output.write('%s %s %s %s\n' % (card_format.opcode, mat_index, context_id, val.value(0)))
  #determine which materials do not have specific transport params
  #and set them to the default values
  default_mats = set(scope.material_list.keys()) - ms_mats
  for dm in default_mats:
    mat_index = scope.material_list[dm]
    scope.output.write('%s %s %s %s\n' % (card_format.opcode, mat_index, context_id, item.value(0)))
