import os
import unittest
import vtk, qt, ctk, slicer
from slicer.ScriptedLoadableModule import *
import logging
from slicer.util import VTKObservationMixin
import textwrap

#
# Home
#

class Home(ScriptedLoadableModule):
  """Uses ScriptedLoadableModule base class, available at:
  https://github.com/Slicer/Slicer/blob/master/Base/Python/slicer/ScriptedLoadableModule.py
  """

  def __init__(self, parent):
    ScriptedLoadableModule.__init__(self, parent)
    self.parent.title = "Home" # TODO make this more human readable by adding spaces
    self.parent.categories = [""]
    self.parent.dependencies = ["SegmentEditor", "DICOM"]
    self.parent.contributors = ["Sam Horvath (Kitware Inc.)"] # replace with "Firstname Lastname (Organization)"
    self.parent.helpText = """
This is the Home module for the AevaSliceer application
"""
    self.parent.helpText += self.getDefaultModuleDocumentationLink()
    self.parent.acknowledgementText = """
This file was originally developed by Jean-Christophe Fillion-Robin, Kitware Inc.
and Steve Pieper, Isomics, Inc. and was partially funded by NIH grant 3P41RR013218-12S1.
""" # replace with organization, grant and thanks.

#
# HomeWidget
#

class HomeWidget(ScriptedLoadableModuleWidget, VTKObservationMixin):
  """Uses ScriptedLoadableModuleWidget base class, available at:
  https://github.com/Slicer/Slicer/blob/master/Base/Python/slicer/ScriptedLoadableModule.py
  """

  def __init__(self, parent):
    ScriptedLoadableModuleWidget.__init__(self, parent)
    VTKObservationMixin.__init__(self)

  def setup(self):
    ScriptedLoadableModuleWidget.setup(self)

    # Load widget from .ui file (created by Qt Designer)
    uiWidget = slicer.util.loadUI(self.resourcePath('UI/Home.ui'))
    self.layout.addWidget(uiWidget)
    self.ui = slicer.util.childWidgetVariables(uiWidget)
    uiWidget.setPalette(slicer.util.mainWindow().style().standardPalette())

    self.modifyWindowUI()

    
    #Collect tab indicies
    self.ui.ModuleTabWidget.currentChanged.connect(self.onTabChanged)
    self.DataTabIndex = self.ui.ModuleTabWidget.indexOf(self.ui.DataTab)
    self.EditorTabIndex = self.ui.ModuleTabWidget.indexOf(self.ui.EditorTab)
    self.MeshingTabIndex = self.ui.ModuleTabWidget.indexOf(self.ui.MeshingTab)
    self.CurrentIndex = -1

    #Segment Editor    
    self.segWidget = slicer.modules.segmenteditor.createNewWidgetRepresentation()
    self.ui.EditorTab.layout().addWidget(self.segWidget)
    self.ui.EditorTab.layout().addStretch(1)
  

    #Meshing
    self.meshWidget = slicer.modules.segmentmesher.createNewWidgetRepresentation()
    self.ui.MeshingTab.layout().addWidget(self.meshWidget)
    self.ui.MeshingTab.layout().addStretch(1)

    #Data Widget
    self.dataWidget = slicer.modules.data.createNewWidgetRepresentation()
    self.dicomButton = qt.QPushButton('Show/Hide DICOM Browser')    
    self.ui.DataTab.layout().addWidget(self.dicomButton)
    self.ui.DataTab.layout().addWidget(self.dataWidget)
    
    #Initialize tab tracking
    self.ui.ModuleTabWidget.setCurrentIndex(self.DataTabIndex)
    self.onTabChanged(self.DataTabIndex)
    
    #Make sure DICOM widget exists
    slicer.app.connect("startupCompleted()", self.setupDICOMBrowser)
    
    #Apply style
    self.applyStyle()

  def onTabChanged(self, tabIndex):
    if tabIndex == self.CurrentIndex:
      return

    #Exit Previous Tab
    if self.CurrentIndex == self.EditorTabIndex:
      self.segWidget.exit()

    #Enter New Tab
    if tabIndex == self.EditorTabIndex:
      self.segWidget.enter()    

    #Update Current Tab
    self.CurrentIndex = tabIndex
  
  def setupDICOMBrowser(self):
    #Make sure that the DICOM widget exists
    slicer.modules.dicom.widgetRepresentation()
    self.dicomButton.clicked.connect(self.toggleDICOMBrowser)
    
    #For some reason, the browser is instantiated as not hidden. Close
    #so that the 'isHidden' check works as required
    slicer.modules.DICOMWidget.browserWidget.close()

  def toggleDICOMBrowser(self):
    if slicer.modules.DICOMWidget.browserWidget.isHidden():
        slicer.modules.DICOMWidget.onOpenBrowserWidget()
    else:
      slicer.modules.DICOMWidget.browserWidget.close()
  
  def applyStyle(self):
    # Style
    stylesheetfile = self.resourcePath('Home.qss')
    with open(stylesheetfile,"r") as fh:
      slicer.app.styleSheet = fh.read()

    # central = slicer.util.findChild(slicer.util.mainWindow(), name='CentralWidget')
    # central.setStyleSheet("background-color: #464449")  
        
  def modifyWindowUI(self):
    slicer.util.setModuleHelpSectionVisible(False)
    slicer.util.setApplicationLogoVisible(False)

    slicer.util.mainWindow().moduleSelector().modulesMenu().removeModule('Annotations', False)
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeModule('Markups', False)
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeModule('Transforms', False)
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeModule('Data', False)
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeModule('SegmentEditor', False)

    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Informatics')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Registration')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Segmentation')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Quantification')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Diffusion')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Converters')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Utilities')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Developer Tools')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Legacy')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('IGT')
    slicer.util.mainWindow().moduleSelector().modulesMenu().removeCategory('Filtering')
    #slicer.util.mainWindow().moduleSelector().modulesMenu().allModulesCategoryVisible= False


    slicer.util.setToolbarsVisible(True)
    keepToolbars = [
      slicer.util.findChild(slicer.util.mainWindow(), 'MainToolBar'),
      slicer.util.findChild(slicer.util.mainWindow(), 'ViewToolBar'),
      slicer.util.findChild(slicer.util.mainWindow(), 'ModuleSelectorToolBar'),
      slicer.util.findChild(slicer.util.mainWindow(), 'MouseModeToolBar')
      ]
    slicer.util.setToolbarsVisible(False, keepToolbars)  

#
# HomeLogic
#

class HomeLogic(ScriptedLoadableModuleLogic):
  """This class should implement all the actual
  computation done by your module.  The interface
  should be such that other python code can import
  this class and make use of the functionality without
  requiring an instance of the Widget.
  Uses ScriptedLoadableModuleLogic base class, available at:
  https://github.com/Slicer/Slicer/blob/master/Base/Python/slicer/ScriptedLoadableModule.py
  """

  def run(self):
    pass  
  
  
class HomeTest(ScriptedLoadableModuleTest):
  """
  This is the test case for your scripted module.
  Uses ScriptedLoadableModuleTest base class, available at:
  https://github.com/Slicer/Slicer/blob/master/Base/Python/slicer/ScriptedLoadableModule.py
  """

  def setUp(self):
    """ Do whatever is needed to reset the state - typically a scene clear will be enough.
    """
    slicer.mrmlScene.Clear(0)

  def runTest(self):
    """Run as few or as many tests as needed here.
    """
    self.setUp()
    self.test_Home1()

  def test_Home1(self):
    """ Ideally you should have several levels of tests.  At the lowest level
    tests should exercise the functionality of the logic with different inputs
    (both valid and invalid).  At higher levels your tests should emulate the
    way the user would interact with your code and confirm that it still works
    the way you intended.
    One of the most important features of the tests is that it should alert other
    developers when their changes will have an impact on the behavior of your
    module.  For example, if a developer removes a feature that you depend on,
    your test should break so they know that the feature is needed.
    """

    self.delayDisplay("Starting the test")
    #
    # first, get some data
    #
    
    logic = HomeLogic()
    self.delayDisplay('Test passed!')


#
# Class for avoiding python error that is caused by the method SegmentEditor::setup
# http://issues.slicer.org/view.php?id=3871
#
class HomeFileWriter(object):
  def __init__(self, parent):
    pass
