import cv2
import json
import numpy
import os
import random
import sys
random.seed()


class Randomizer:
    def __init__(self, video, activities):
        # processing file
        self.file = {}
        self.file['path'] = video
        self.file['name'] = video.split(os.path.sep)[-1]
        cap = cv2.VideoCapture(video)
        self.file['length'] = int(cap.get(cv2.CAP_PROP_FRAME_COUNT))
        self.file['fps'] = int(cap.get(cv2.CAP_PROP_FPS))
        self.file['width'] = int(cap.get(cv2.CAP_PROP_FRAME_WIDTH))
        self.file['height'] = int(cap.get(cv2.CAP_PROP_FRAME_HEIGHT))
        self.activities = activities
        # random config
        path = os.path.dirname(__file__)
        config_path = os.path.join(path, "random_config.json")
        with open(config_path, 'r') as fd:
            self.config = json.load(fd)

    # Distributions
    def uniform(self, a: float, b: float) -> float:
        """Uniform distribution between `a` and `b`."""
        return a + (b-a) * random.random()

    def normal(self, mean: float, stdev: float) -> float:
        """Normal distribution with parameter `mean` and `stdev` (standard
        deviation)."""
        return numpy.random.normal(mean, stdev)

    # Generation of single elements
    def _pick_from_seq(self, seq) -> object:
        """Return a random element from the sequence `seq`."""
        return None if seq is None else random.choice(seq)

    def _generate_presence_conf(self) -> float:
        """Generate presenceConf number using X distribution."""
        distrib = self.config["presenceConfDistr"]
        try:
            fct = getattr(self, distrib)
        except NameError:
            print("", file=sys.stderr)
        params = self.config["presenceConfDistrParams"]
        return fct(*params)

    def _generate_frames(self, frameNbr: int) -> tuple:
        """Generate starting and ending frame using X distribution."""
        starting_frame = int(self.uniform(0, frameNbr-1))
        ending_frame = starting_frame + int(self.normal(
            *self.config['durationDistrParams']))
        if ending_frame >= frameNbr:
            ending_frame = frameNbr - 1
        return (starting_frame, ending_frame) if ending_frame > \
            starting_frame else self._generate_frames(frameNbr)

    def _generate_activities_number(self, frameNbr: int) -> int:
        """Generate a number of activities for a given frame number `frameNbr`
        using normal distribution."""
        # WARNING
        # TEMPORARY CODE
        # NEED TO USE CORRECT PARAMETERS
        return int(self.normal(*self.config['activitiesNumberDistrParams']))

    #  Only public method
    def generate_random_activities(self) -> dict:
        """Return a list of random activities"""
        res = {}
        res['filesProcessed'] = [self.file['name']]
        res['activities'] = []

        for i in range(self._generate_activities_number(self.file['length'])):
            activity = {}
            activity["activity"] = self._pick_from_seq(self.activities)
            activity["activityID"] = i + 1
            start, end = self._generate_frames(self.file['length'])
            activity["localization"] = {
                self.file['name']: {
                    start: 1,
                    end: 0
                }
            }
            activity["presenceConf"] = self._generate_presence_conf()
            res['activities'].append(activity)
        return res
