import abc
import logging
import sys

from diva_evaluation_cli.bin.private_src.implementation.status.status_factory import StatusFactory

class ActevCommand():
    """ Abstract class that represents an actev command.
    """
    __metaclass__ = abc.ABCMeta

    def __init__(self, command, entry_point, before_entry_point=None, after_entry_point=None):
        """ 
        @param command: string representing the name of the command
        """
        self.command = command
        self.entry_point = entry_point
        self.before_entry_point = before_entry_point
        self.after_entry_point = after_entry_point

    @abc.abstractmethod
    def cli_parser(self, arg_parser):
        """ Configure the description and the arguments (positional and optional) to parse.

        @param arg_parser: python arg parser to describe how to parse the command
        """ 
        return

    def before_command(self, args):
        """ Execute an action before executing the command
        """
        if self.before_entry_point:
            self.before_entry_point(**args.__dict__)

    def after_command(self, args):
        """ Execute an action after executing the command
        """
        if self.after_entry_point:
            self.after_entry_point(**args.__dict__)

    def command(self, args):
        """ Gets arguments and passe them to an entry point. Catch the exception occured.

        @param args: list of arguments passed during the command call
        """
        del args.__dict__['object']
        del args.__dict__['func']

        try:
            logging.info("Starting %s" % self.command)
            StatusFactory.generateStatus(self, 'start', args.__dict__)

            self.before_command(args)
            self.entry_point(**args.__dict__)
            self.after_command(args)

            logging.info("%s done" % self.command)
            StatusFactory.generateStatus(self, 'done', args.__dict__)
        except:
            logging.exception("Issue during %s" % self.command)
            StatusFactory.generateStatus(self, 'issue', args.__dict__)
            sys.exit(1)
    
