/*=========================================================================

  Program:   Visualization Library
  Module:    FTCoords.hh
  Language:  C++
  Date:      $Date$
  Version:   $Revision$

This file is part of the Visualization Library. No part of this file or its 
contents may be copied, reproduced or altered in any way without the express
written consent of the authors.

Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen 1993, 1994 

=========================================================================*/
// .NAME vlFloatTCoords - floating point representation of texture coordinates
// .SECTION Description
// vlFloatTCoords is a concrete implementation of vlTCoords. Texture
// coordinates are represented using float values.

#ifndef __vlFloatTCoords_h
#define __vlFloatTCoords_h

#include "TCoords.hh"
#include "FArray.hh"

class vlFloatTCoords : public vlTCoords
{
public:
  vlFloatTCoords() {};
  vlFloatTCoords(const vlFloatTCoords& ftc) {this->TC = ftc.TC;this->Dimension = ftc.Dimension;};
  vlFloatTCoords(int sz, int d=2, int ext=1000):TC(d*sz,d*ext) {this->Dimension=d;};
  ~vlFloatTCoords() {};
  int Allocate(const int sz, const int dim=2, const int ext=1000) {return this->TC.Allocate(dim*sz,dim*ext);};
  void Initialize() {this->TC.Initialize();};
  char *GetClassName() {return "vlFloatTCoords";};

  // vlTCoords interface
  vlTCoords *MakeObject(int sze, int d=2, int ext=1000);
  char *GetDataType() {return "float";};
  int GetNumberOfTCoords() {return (this->TC.GetMaxId()+1)/this->Dimension;};
  void Squeeze() {this->TC.Squeeze();};
  float *GetTCoord(int i) {return this->TC.GetPtr(this->Dimension*i);};
  void GetTCoord(int i,float tc[3]) {this->vlTCoords::GetTCoord(i,tc);};
  void SetTCoord(int i, float *tc);
  void InsertTCoord(int i, float *tc);
  int InsertNextTCoord(float *tc);

  // miscellaneous
  float *WritePtr(const int id, const int number);
  vlFloatTCoords &operator=(const vlFloatTCoords& ftc);
  void operator+=(const vlFloatTCoords& ftc) {this->TC += ftc.TC;};
  void Reset() {this->TC.Reset();};

protected:
  vlFloatArray TC;
};


// Description:
// Get pointer to data. Useful for direct writes into object. MaxId is bumped
// by number (and memory allocated if necessary). Id is the location you 
// wish to write into; number is the number of texture coordinates to write. 
// Make sure the dimension of the texture coordinate is set prior to issuing 
// this call.
inline float *vlFloatTCoords::WritePtr(const int id, const int number)
{
  return this->TC.WritePtr(id,this->Dimension*number);
}

inline void vlFloatTCoords::SetTCoord(int i, float *tc) 
{
  i*=this->Dimension; 
  for(int j=0;j<this->Dimension;j++) this->TC[i+j]=tc[j];
}

inline void vlFloatTCoords::InsertTCoord(int i, float *tc) 
{
  i*=this->Dimension; 
  for(int j=0; j<this->Dimension; j++) this->TC.InsertValue(i+j, tc[j]);
}

inline int vlFloatTCoords::InsertNextTCoord(float *tc) 
{
  int id = this->TC.InsertNextValue(tc[0]);
  for(int j=1; j<this->Dimension; j++) this->TC.InsertNextValue(tc[j]);
  return id/this->Dimension;
}

#endif
