/*=========================================================================

  Program:   Visualization Toolkit
  Module:    Mapper.cc
  Language:  C++
  Date:      $Date$
  Version:   $Revision$

This file is part of the Visualization Toolkit. No part of this file or its 
contents may be copied, reproduced or altered in any way without the express
written consent of the authors.

Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen 1993, 1994 

=========================================================================*/
#include "Mapper.hh"

// Description:
// Construct with initial range (0,1).
vtkMapper::vtkMapper()
{
  this->Input = NULL;

  this->StartRender = NULL;
  this->StartRenderArgDelete = NULL;
  this->StartRenderArg = NULL;
  this->EndRender = NULL;
  this->EndRenderArgDelete = NULL;
  this->EndRenderArg = NULL;

  this->LookupTable = NULL;

  this->ScalarsVisible = 1;
  this->ScalarRange[0] = 0.0; this->ScalarRange[1] = 1.0;

  this->SelfCreatedLookupTable = 0;
}

vtkMapper::~vtkMapper()
{
  if ( this->SelfCreatedLookupTable && this->LookupTable != NULL) 
    this->LookupTable->Delete();
}

// Description:
// Overload standard modified time function. If cut functions is modified,
// then we are modified as well.
unsigned long vtkMapper::GetMTime()
{
  unsigned long mTime=this->MTime.GetMTime();
  unsigned long lutMTime;

  if ( this->LookupTable != NULL )
    {
    lutMTime = this->LookupTable->GetMTime();
    mTime = ( lutMTime > mTime ? lutMTime : mTime );
    }

  return mTime;
}

void vtkMapper::operator=(const vtkMapper& m)
{
  this->SetLookupTable(m.LookupTable);

  this->SetScalarsVisible(m.ScalarsVisible);
  this->SetScalarRange(m.ScalarRange[0], m.ScalarRange[1]);

  this->SetStartRender(m.StartRender,m.StartRenderArg);
  this->SetEndRender(m.EndRender,m.EndRenderArg);
}

// Description:
// Specify a function to be called before rendering process begins.
// Function will be called with argument provided.
void vtkMapper::SetStartRender(void (*f)(void *), void *arg)
{
  if ( f != this->StartRender || arg != this->StartRenderArg )
    {
    // delete the current arg if there is one and a delete meth
    if ((this->StartRenderArg)&&(this->StartRenderArgDelete))
      {
      (*this->StartRenderArgDelete)(this->StartRenderArg);
      }
    this->StartRender = f;
    this->StartRenderArg = arg;
    this->Modified();
    }
}

// Description:
// Set the arg delete method. This is used to free user memory.
void vtkMapper::SetStartRenderArgDelete(void (*f)(void *))
{
  if ( f != this->StartRenderArgDelete)
    {
    this->StartRenderArgDelete = f;
    this->Modified();
    }
}

// Description:
// Set the arg delete method. This is used to free user memory.
void vtkMapper::SetEndRenderArgDelete(void (*f)(void *))
{
  if ( f != this->EndRenderArgDelete)
    {
    this->EndRenderArgDelete = f;
    this->Modified();
    }
}

// Description:
// Specify a function to be called when rendering process completes.
// Function will be called with argument provided.
void vtkMapper::SetEndRender(void (*f)(void *), void *arg)
{
  if ( f != this->EndRender || arg != EndRenderArg )
    {
    // delete the current arg if there is one and a delete meth
    if ((this->EndRenderArg)&&(this->EndRenderArgDelete))
      {
      (*this->EndRenderArgDelete)(this->EndRenderArg);
      }
    this->EndRender = f;
    this->EndRenderArg = arg;
    this->Modified();
    }
}

// Description:
// Specify a lookup table for the mapper to use.
void vtkMapper::SetLookupTable(vtkLookupTable *lut)
{
  if ( this->LookupTable != lut ) 
    {
    if ( this->SelfCreatedLookupTable ) this->LookupTable->Delete();
    this->SelfCreatedLookupTable = 0;
    this->LookupTable = lut;
    this->Modified();
    }
}

vtkLookupTable *vtkMapper::GetLookupTable()
{
  if ( this->LookupTable == NULL ) this->CreateDefaultLookupTable();
  return this->LookupTable;
}

void vtkMapper::CreateDefaultLookupTable()
{
  if ( this->SelfCreatedLookupTable ) this->LookupTable->Delete();
  this->LookupTable = new vtkLookupTable;
  this->SelfCreatedLookupTable = 1;
}

float *vtkMapper::GetCenter()
{
  static float center[3];
  float *bounds;

  bounds = this->GetBounds();
  for (int i=0; i<3; i++) center[i] = (bounds[2*i+1] + bounds[2*i]) / 2.0;
  return center;
}

void vtkMapper::PrintSelf(ostream& os, vtkIndent indent)
{
  vtkObject::PrintSelf(os,indent);

  if ( this->Input )
    {
    os << indent << "Input: (" << this->Input << ")\n";
    }
  else
    {
    os << indent << "Input: (none)\n";
    }

  os << indent << "Build Time: " <<this->BuildTime.GetMTime() << "\n";
  if ( this->StartRender )
    {
    os << indent << "Start Render method defined.\n";
    }
  else
    {
    os << indent << "No Start Render method.\n";
    }

  if ( this->EndRender )
    {
    os << indent << "End Render method defined.\n";
    }
  else
    {
    os << indent << "No End Render method.\n";
    }

  if ( this->LookupTable )
    {
    os << indent << "Lookup Table:\n";
    this->LookupTable->PrintSelf(os,indent.GetNextIndent());
    }
  else
    {
    os << indent << "Lookup Table: (none)\n";
    }
  os << indent << "Scalars Visible: " 
    << (this->ScalarsVisible ? "On\n" : "Off\n");

  float *range = this->GetScalarRange();
  os << indent << "Scalar Range: (" << range[0] << ", " << range[1] << ")\n";
}
