/*=========================================================================

  Program:   Visualization Toolkit
  Module:    LinExtrd.hh
  Language:  C++
  Date:      $Date$
  Version:   $Revision$

This file is part of the Visualization Toolkit. No part of this file
or its contents may be copied, reproduced or altered in any way
without the express written consent of the authors.

Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen 1993, 1994 

=========================================================================*/
// .NAME vtkLinearExtrusionFilter - sweep polygonal data creating "skirt" from free edges and lines, and lines from vertices
// .SECTION Description
// vtkLinearExtrusionFilter is a modelling filter. It takes polygonal data as 
// input and generates polygonal data on output. The input dataset is swept
// according to some extrusion function and creates new polygonal primitives.
// These primitives form a "skirt" or swept surface. For example, sweeping a
// line results in a quadrilateral, and sweeping a triangle creates a "wedge".
//    There are a number of control parameters for this filter. You can 
// control whether the sweep of a 2D object (i.e., polygon or triangle strip) 
// is capped with the generating geometry via the "Capping" ivar. Also, you
// can extrude in the direction of a user specified vector, towards a point,
// or in the direction of vertex normals (normals must be provided - use 
// vtkPolyNormals if necessary). The amount of extrusion is controlled by
// the "ScaleFactor" instance variable.
//    The skirt is generated by locating certain topological features. Free 
// edges (edges of polygons or triangle strips only used by one polygon or
// triangle strips) generate surfaces. This is true also of lines or 
// polylines. Vertices generate lines.
//    This filter can be used to create 3D fonts, 3D irregular bar charts,
// or to model 2 1/2D objects like punched plates. It can also be used to 
// create solid objects from 2D polygonal meshes.
// .SECTION Caveats
// Some polygonal objects have no free edges (e.g., sphere). When swept 
// this will result in two separate surfaces if capping is on, or no surface
// if capping is off.
// .SECTION See Also
// vtkRotationalExtrusionFilter

#ifndef __vtkLinearExtrusionFilter_h
#define __vtkLinearExtrusionFilter_h

#include "P2PF.hh"

#define VECTOR_EXTRUSION 1
#define NORMAL_EXTRUSION 2
#define POINT_EXTRUSION 3

class vtkLinearExtrusionFilter : public vtkPolyToPolyFilter 
{
public:
  vtkLinearExtrusionFilter();
  ~vtkLinearExtrusionFilter() {};
  char *GetClassName() {return "vtkLinearExtrusionFilter";};
  void PrintSelf(ostream& os, vtkIndent indent);

  // Description:
  // Set/Get the type of extrusion.
  vtkSetClampMacro(ExtrusionType,int,VECTOR_EXTRUSION,POINT_EXTRUSION);
  vtkGetMacro(ExtrusionType,int);

  // Description:
  // Turn on/off the capping of the skirt.
  vtkSetMacro(Capping,int);
  vtkGetMacro(Capping,int);
  vtkBooleanMacro(Capping,int);

  // Description:
  // Set/Get extrusion scale factor,
  vtkSetMacro(ScaleFactor,float);
  vtkGetMacro(ScaleFactor,float);

  // Description:
  // Set/Get extrusion vector. Only needs to be set if VectorExtrusion is
  // turned on.
  vtkSetVector3Macro(Vector,float);
  vtkGetVectorMacro(Vector,float,3);

  // Description:
  // Set/Get extrusion point. Only needs to be set if PointExtrusion is
  // turned on. This is the point towards which extrusion occurs.
  vtkSetVector3Macro(ExtrusionPoint,float);
  vtkGetVectorMacro(ExtrusionPoint,float,3);

protected:
  void Execute();
  int ExtrusionType;
  int Capping;
  float ScaleFactor;
  float Vector[3];
  float ExtrusionPoint[3];

  //BTX
  float *(vtkLinearExtrusionFilter::*ExtrudePoint)(float x[3], int id, vtkNormals *normals);
  float *ViaNormal(float x[3], int id, vtkNormals *normals=NULL);
  float *ViaVector(float x[3], int id, vtkNormals *normals=NULL);
  float *ViaPoint(float x[3], int id, vtkNormals *normals=NULL);
  //ETX
 
};

#endif
