/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmXMLParser.h"

#include <cctype>
#include <cstring>
#include <iostream>
#include <sstream>

#include <cm3p/expat.h>

#include "cmsys/FStream.hxx"

cmXMLParser::cmXMLParser()
{
  Parser = nullptr;
  ParseError = 0;
  ReportCallback = nullptr;
  ReportCallbackData = nullptr;
}

cmXMLParser::~cmXMLParser()
{
  if (Parser) {
    CleanupParser();
  }
}

int cmXMLParser::Parse(const char* string)
{
  return InitializeParser() && ParseChunk(string, strlen(string)) &&
    CleanupParser();
}

int cmXMLParser::ParseFile(const char* file)
{
  if (!file) {
    return 0;
  }

  cmsys::ifstream ifs(file);
  if (!ifs) {
    return 0;
  }

  std::ostringstream str;
  str << ifs.rdbuf();
  return Parse(str.str().c_str());
}

int cmXMLParser::InitializeParser()
{
  if (Parser) {
    std::cerr << "Parser already initialized" << std::endl;
    ParseError = 1;
    return 0;
  }

  // Create the expat XML parser.
  Parser = XML_ParserCreate(nullptr);
  XML_SetElementHandler(static_cast<XML_Parser>(Parser),
                        &cmXMLParserStartElement, &cmXMLParserEndElement);
  XML_SetCharacterDataHandler(static_cast<XML_Parser>(Parser),
                              &cmXMLParserCharacterDataHandler);
  XML_SetUserData(static_cast<XML_Parser>(Parser), this);
  ParseError = 0;
  return 1;
}

int cmXMLParser::ParseChunk(const char* inputString,
                            std::string::size_type length)
{
  if (!Parser) {
    std::cerr << "Parser not initialized" << std::endl;
    ParseError = 1;
    return 0;
  }
  int res;
  res = ParseBuffer(inputString, length);
  if (res == 0) {
    ParseError = 1;
  }
  return res;
}

int cmXMLParser::CleanupParser()
{
  if (!Parser) {
    std::cerr << "Parser not initialized" << std::endl;
    ParseError = 1;
    return 0;
  }
  int result = !ParseError;
  if (result) {
    // Tell the expat XML parser about the end-of-input.
    if (!XML_Parse(static_cast<XML_Parser>(Parser), "", 0, 1)) {
      ReportXmlParseError();
      result = 0;
    }
  }

  // Clean up the parser.
  XML_ParserFree(static_cast<XML_Parser>(Parser));
  Parser = nullptr;

  return result;
}

int cmXMLParser::ParseBuffer(const char* buffer, std::string::size_type count)
{
  // Pass the buffer to the expat XML parser.
  if (!XML_Parse(static_cast<XML_Parser>(Parser), buffer,
                 static_cast<int>(count), 0)) {
    ReportXmlParseError();
    return 0;
  }
  return 1;
}

int cmXMLParser::ParseBuffer(const char* buffer)
{
  return ParseBuffer(buffer, static_cast<int>(strlen(buffer)));
}

int cmXMLParser::ParsingComplete()
{
  // Default behavior is to parse to end of stream.
  return 0;
}

void cmXMLParser::StartElement(const std::string& name, const char** /*atts*/)
{
  std::cout << "Start element: " << name << std::endl;
}

void cmXMLParser::EndElement(const std::string& name)
{
  std::cout << "End element: " << name << std::endl;
}

void cmXMLParser::CharacterDataHandler(const char* /*inData*/,
                                       int /*inLength*/)
{
}

int cmXMLParser::IsSpace(char c)
{
  return isspace(c);
}

const char* cmXMLParser::FindAttribute(const char** atts,
                                       const char* attribute)
{
  if (atts && attribute) {
    for (const char** a = atts; *a && *(a + 1); a += 2) {
      if (strcmp(*a, attribute) == 0) {
        return *(a + 1);
      }
    }
  }
  return nullptr;
}

void cmXMLParserStartElement(void* parser, const char* name, const char** atts)
{
  // Begin element handler that is registered with the XML_Parser.
  // This just casts the user data to a cmXMLParser and calls
  // StartElement.
  static_cast<cmXMLParser*>(parser)->StartElement(name, atts);
}

void cmXMLParserEndElement(void* parser, const char* name)
{
  // End element handler that is registered with the XML_Parser.  This
  // just casts the user data to a cmXMLParser and calls EndElement.
  static_cast<cmXMLParser*>(parser)->EndElement(name);
}

void cmXMLParserCharacterDataHandler(void* parser, const char* data,
                                     int length)
{
  // Character data handler that is registered with the XML_Parser.
  // This just casts the user data to a cmXMLParser and calls
  // CharacterDataHandler.
  static_cast<cmXMLParser*>(parser)->CharacterDataHandler(data, length);
}

void cmXMLParser::ReportXmlParseError()
{
  XML_Parser parser = static_cast<XML_Parser>(Parser);
  ReportError(static_cast<int>(XML_GetCurrentLineNumber(parser)),
              static_cast<int>(XML_GetCurrentColumnNumber(parser)),
              XML_ErrorString(XML_GetErrorCode(parser)));
}

void cmXMLParser::ReportError(int line, int /*unused*/, const char* msg)
{
  if (ReportCallback) {
    ReportCallback(line, msg, ReportCallbackData);
  } else {
    std::cerr << "Error parsing XML in stream at line " << line << ": " << msg
              << std::endl;
  }
}
