/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmTargetPropCommandBase.h"

#include "cmExecutionStatus.h"
#include "cmGlobalGenerator.h"
#include "cmMakefile.h"
#include "cmProperty.h"
#include "cmStateTypes.h"
#include "cmTarget.h"
#include "cmake.h"

cmTargetPropCommandBase::cmTargetPropCommandBase(cmExecutionStatus& status)
  : Makefile(&status.GetMakefile())
  , Status(status)
{
}

void cmTargetPropCommandBase::SetError(std::string const& e)
{
  Status.SetError(e);
}

bool cmTargetPropCommandBase::HandleArguments(
  std::vector<std::string> const& args, const std::string& prop,
  ArgumentFlags flags)
{
  if (args.size() < 2) {
    SetError("called with incorrect number of arguments");
    return false;
  }

  if (Makefile->IsAlias(args[0])) {
    SetError("can not be used on an ALIAS target.");
    return false;
  }
  // Lookup the target for which property-values are specified.
  Target =
    Makefile->GetCMakeInstance()->GetGlobalGenerator()->FindTarget(args[0]);
  if (!Target) {
    Target = Makefile->FindTargetToUse(args[0]);
  }
  if (!Target) {
    HandleMissingTarget(args[0]);
    return false;
  }
  if ((Target->GetType() != cmStateEnums::EXECUTABLE) &&
      (Target->GetType() != cmStateEnums::STATIC_LIBRARY) &&
      (Target->GetType() != cmStateEnums::SHARED_LIBRARY) &&
      (Target->GetType() != cmStateEnums::MODULE_LIBRARY) &&
      (Target->GetType() != cmStateEnums::OBJECT_LIBRARY) &&
      (Target->GetType() != cmStateEnums::INTERFACE_LIBRARY) &&
      (Target->GetType() != cmStateEnums::UNKNOWN_LIBRARY)) {
    SetError("called with non-compilable target type");
    return false;
  }

  bool system = false;
  unsigned int argIndex = 1;

  if ((flags & PROCESS_SYSTEM) && args[argIndex] == "SYSTEM") {
    if (args.size() < 3) {
      SetError("called with incorrect number of arguments");
      return false;
    }
    system = true;
    ++argIndex;
  }

  bool prepend = false;
  if ((flags & PROCESS_BEFORE) && args[argIndex] == "BEFORE") {
    if (args.size() < 3) {
      SetError("called with incorrect number of arguments");
      return false;
    }
    prepend = true;
    ++argIndex;
  }

  if ((flags & PROCESS_REUSE_FROM) && args[argIndex] == "REUSE_FROM") {
    if (args.size() != 3) {
      SetError("called with incorrect number of arguments");
      return false;
    }
    ++argIndex;

    Target->SetProperty("PRECOMPILE_HEADERS_REUSE_FROM", args[argIndex]);
    ++argIndex;
  }

  Property = prop;

  while (argIndex < args.size()) {
    if (!ProcessContentArgs(args, argIndex, prepend, system)) {
      return false;
    }
  }
  return true;
}

bool cmTargetPropCommandBase::ProcessContentArgs(
  std::vector<std::string> const& args, unsigned int& argIndex, bool prepend,
  bool system)
{
  std::string const& scope = args[argIndex];

  if (scope != "PUBLIC" && scope != "PRIVATE" && scope != "INTERFACE") {
    SetError("called with invalid arguments");
    return false;
  }

  ++argIndex;

  std::vector<std::string> content;

  for (unsigned int i = argIndex; i < args.size(); ++i, ++argIndex) {
    if (args[i] == "PUBLIC" || args[i] == "PRIVATE" ||
        args[i] == "INTERFACE") {
      break;
    }
    content.push_back(args[i]);
  }
  if (!content.empty()) {
    if (Target->GetType() == cmStateEnums::INTERFACE_LIBRARY &&
        scope != "INTERFACE" && Property != "SOURCES") {
      SetError("may only set INTERFACE properties on INTERFACE targets");
      return false;
    }
    if (Target->IsImported() && scope != "INTERFACE") {
      SetError("may only set INTERFACE properties on IMPORTED targets");
      return false;
    }
  }
  return PopulateTargetProperies(scope, content, prepend, system);
}

bool cmTargetPropCommandBase::PopulateTargetProperies(
  const std::string& scope, const std::vector<std::string>& content,
  bool prepend, bool system)
{
  if (content.empty()) {
    return true;
  }
  if (scope == "PRIVATE" || scope == "PUBLIC") {
    if (!HandleDirectContent(Target, content, prepend, system)) {
      return false;
    }
  }
  if (scope == "INTERFACE" || scope == "PUBLIC") {
    HandleInterfaceContent(Target, content, prepend, system);
  }
  return true;
}

void cmTargetPropCommandBase::HandleInterfaceContent(
  cmTarget* tgt, const std::vector<std::string>& content, bool prepend, bool)
{
  if (prepend) {
    const std::string propName = std::string("INTERFACE_") + Property;
    cmProp propValue = tgt->GetProperty(propName);
    const std::string totalContent =
      Join(content) + (propValue ? (";" + *propValue) : std::string());
    tgt->SetProperty(propName, totalContent);
  } else {
    tgt->AppendProperty("INTERFACE_" + Property, Join(content));
  }
}
