/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */

#include "cmStateSnapshot.h"

#include <algorithm>
#include <cassert>
#include <string>

#include <cm/iterator>

#include "cmDefinitions.h"
#include "cmListFileCache.h"
#include "cmProperty.h"
#include "cmPropertyMap.h"
#include "cmState.h"
#include "cmStateDirectory.h"
#include "cmStatePrivate.h"
#include "cmVersion.h"

#if !defined(_WIN32)
#  include <sys/utsname.h>
#endif

#if defined(__CYGWIN__)
#  include "cmSystemTools.h"
#endif

cmStateSnapshot::cmStateSnapshot(cmState* state)
  : State(state)
{
}

std::vector<cmStateSnapshot> cmStateSnapshot::GetChildren()
{
  return Position->BuildSystemDirectory->Children;
}

cmStateSnapshot::cmStateSnapshot(cmState* state,
                                 cmStateDetail::PositionType position)
  : State(state)
  , Position(position)
{
}

cmStateEnums::SnapshotType cmStateSnapshot::GetType() const
{
  return Position->SnapshotType;
}

void cmStateSnapshot::SetListFile(const std::string& listfile)
{
  *Position->ExecutionListFile = listfile;
}

std::string const& cmStateSnapshot::GetExecutionListFile() const
{
  return *Position->ExecutionListFile;
}

bool cmStateSnapshot::IsValid() const
{
  return State && Position.IsValid() ? Position != State->SnapshotData.Root()
                                     : false;
}

cmStateSnapshot cmStateSnapshot::GetBuildsystemDirectory() const
{
  return { State, Position->BuildSystemDirectory->DirectoryEnd };
}

cmStateSnapshot cmStateSnapshot::GetBuildsystemDirectoryParent() const
{
  cmStateSnapshot snapshot;
  if (!State || Position == State->SnapshotData.Root()) {
    return snapshot;
  }
  cmStateDetail::PositionType parentPos = Position->DirectoryParent;
  if (parentPos != State->SnapshotData.Root()) {
    snapshot =
      cmStateSnapshot(State, parentPos->BuildSystemDirectory->DirectoryEnd);
  }

  return snapshot;
}

cmStateSnapshot cmStateSnapshot::GetCallStackParent() const
{
  assert(State);
  assert(Position != State->SnapshotData.Root());

  cmStateSnapshot snapshot;
  cmStateDetail::PositionType parentPos = Position;
  while (parentPos->SnapshotType == cmStateEnums::PolicyScopeType ||
         parentPos->SnapshotType == cmStateEnums::VariableScopeType) {
    ++parentPos;
  }
  if (parentPos->SnapshotType == cmStateEnums::BuildsystemDirectoryType ||
      parentPos->SnapshotType == cmStateEnums::BaseType) {
    return snapshot;
  }

  ++parentPos;
  while (parentPos->SnapshotType == cmStateEnums::PolicyScopeType ||
         parentPos->SnapshotType == cmStateEnums::VariableScopeType) {
    ++parentPos;
  }

  if (parentPos == State->SnapshotData.Root()) {
    return snapshot;
  }

  snapshot = cmStateSnapshot(State, parentPos);
  return snapshot;
}

cmStateSnapshot cmStateSnapshot::GetCallStackBottom() const
{
  assert(State);
  assert(Position != State->SnapshotData.Root());

  cmStateDetail::PositionType pos = Position;
  while (pos->SnapshotType != cmStateEnums::BaseType &&
         pos->SnapshotType != cmStateEnums::BuildsystemDirectoryType &&
         pos != State->SnapshotData.Root()) {
    ++pos;
  }
  return { State, pos };
}

void cmStateSnapshot::PushPolicy(cmPolicies::PolicyMap const& entry, bool weak)
{
  cmStateDetail::PositionType pos = Position;
  pos->Policies = State->PolicyStack.Push(
    pos->Policies, cmStateDetail::PolicyStackEntry(entry, weak));
}

bool cmStateSnapshot::PopPolicy()
{
  cmStateDetail::PositionType pos = Position;
  if (pos->Policies == pos->PolicyScope) {
    return false;
  }
  pos->Policies = State->PolicyStack.Pop(pos->Policies);
  return true;
}

bool cmStateSnapshot::CanPopPolicyScope()
{
  return Position->Policies != Position->PolicyScope;
}

void cmStateSnapshot::SetPolicy(cmPolicies::PolicyID id,
                                cmPolicies::PolicyStatus status)
{
  // Update the policy stack from the top to the top-most strong entry.
  bool previous_was_weak = true;
  for (cmLinkedTree<cmStateDetail::PolicyStackEntry>::iterator psi =
         Position->Policies;
       previous_was_weak && psi != Position->PolicyRoot; ++psi) {
    psi->Set(id, status);
    previous_was_weak = psi->Weak;
  }
}

cmPolicies::PolicyStatus cmStateSnapshot::GetPolicy(cmPolicies::PolicyID id,
                                                    bool parent_scope) const
{
  cmPolicies::PolicyStatus status = cmPolicies::GetPolicyStatus(id);

  if (status == cmPolicies::REQUIRED_ALWAYS ||
      status == cmPolicies::REQUIRED_IF_USED) {
    return status;
  }

  cmLinkedTree<cmStateDetail::BuildsystemDirectoryStateType>::iterator dir =
    Position->BuildSystemDirectory;

  while (true) {
    assert(dir.IsValid());
    cmLinkedTree<cmStateDetail::PolicyStackEntry>::iterator leaf =
      dir->DirectoryEnd->Policies;
    cmLinkedTree<cmStateDetail::PolicyStackEntry>::iterator root =
      dir->DirectoryEnd->PolicyRoot;
    for (; leaf != root; ++leaf) {
      if (parent_scope) {
        parent_scope = false;
        continue;
      }
      if (leaf->IsDefined(id)) {
        status = leaf->Get(id);
        return status;
      }
    }
    cmStateDetail::PositionType e = dir->DirectoryEnd;
    cmStateDetail::PositionType p = e->DirectoryParent;
    if (p == State->SnapshotData.Root()) {
      break;
    }
    dir = p->BuildSystemDirectory;
  }
  return status;
}

bool cmStateSnapshot::HasDefinedPolicyCMP0011()
{
  return !Position->Policies->IsEmpty();
}

std::string const* cmStateSnapshot::GetDefinition(
  std::string const& name) const
{
  assert(Position->Vars.IsValid());
  return cmDefinitions::Get(name, Position->Vars, Position->Root);
}

bool cmStateSnapshot::IsInitialized(std::string const& name) const
{
  return cmDefinitions::HasKey(name, Position->Vars, Position->Root);
}

void cmStateSnapshot::SetDefinition(std::string const& name,
                                    cm::string_view value)
{
  Position->Vars->Set(name, value);
}

void cmStateSnapshot::RemoveDefinition(std::string const& name)
{
  Position->Vars->Unset(name);
}

std::vector<std::string> cmStateSnapshot::ClosureKeys() const
{
  return cmDefinitions::ClosureKeys(Position->Vars, Position->Root);
}

bool cmStateSnapshot::RaiseScope(std::string const& var, const char* varDef)
{
  if (Position->ScopeParent == Position->DirectoryParent) {
    cmStateSnapshot parentDir = GetBuildsystemDirectoryParent();
    if (!parentDir.IsValid()) {
      return false;
    }
    // Update the definition in the parent directory top scope.  This
    // directory's scope was initialized by the closure of the parent
    // scope, so we do not need to localize the definition first.
    if (varDef) {
      parentDir.SetDefinition(var, varDef);
    } else {
      parentDir.RemoveDefinition(var);
    }
    return true;
  }
  // First localize the definition in the current scope.
  cmDefinitions::Raise(var, Position->Vars, Position->Root);

  // Now update the definition in the parent scope.
  if (varDef) {
    Position->Parent->Set(var, varDef);
  } else {
    Position->Parent->Unset(var);
  }
  return true;
}

template <typename T, typename U, typename V>
void InitializeContentFromParent(T& parentContent, T& thisContent,
                                 U& parentBacktraces, U& thisBacktraces,
                                 V& contentEndPosition)
{
  auto parentBegin = parentContent.begin();
  auto parentEnd = parentContent.end();

  auto parentRbegin = cm::make_reverse_iterator(parentEnd);
  auto parentRend = parentContent.rend();
  parentRbegin = std::find(parentRbegin, parentRend, cmPropertySentinal);
  auto parentIt = parentRbegin.base();

  thisContent = std::vector<std::string>(parentIt, parentEnd);

  auto btIt = parentBacktraces.begin() + std::distance(parentBegin, parentIt);
  auto btEnd = parentBacktraces.end();

  thisBacktraces = std::vector<cmListFileBacktrace>(btIt, btEnd);

  contentEndPosition = thisContent.size();
}

void cmStateSnapshot::SetDefaultDefinitions()
{
/* Up to CMake 2.4 here only WIN32, UNIX and APPLE were set.
  With CMake must separate between target and host platform. In most cases
  the tests for WIN32, UNIX and APPLE will be for the target system, so an
  additional set of variables for the host system is required ->
  CMAKE_HOST_WIN32, CMAKE_HOST_UNIX, CMAKE_HOST_APPLE.
  WIN32, UNIX and APPLE are now set in the platform files in
  Modules/Platforms/.
  To keep cmake scripts (-P) and custom language and compiler modules
  working, these variables are still also set here in this place, but they
  will be reset in CMakeSystemSpecificInformation.cmake before the platform
  files are executed. */
#if defined(_WIN32)
  this->SetDefinition("WIN32", "1");
  this->SetDefinition("CMAKE_HOST_WIN32", "1");
  this->SetDefinition("CMAKE_HOST_SYSTEM_NAME", "Windows");
#else
  SetDefinition("UNIX", "1");
  SetDefinition("CMAKE_HOST_UNIX", "1");

#  if defined(__ANDROID__)
  this->SetDefinition("CMAKE_HOST_SYSTEM_NAME", "Android");
#  else
  struct utsname uts_name;
  if (uname(&uts_name) >= 0) {
    SetDefinition("CMAKE_HOST_SYSTEM_NAME", uts_name.sysname);
  }
#  endif
#endif
#if defined(__CYGWIN__)
  std::string legacy;
  if (cmSystemTools::GetEnv("CMAKE_LEGACY_CYGWIN_WIN32", legacy) &&
      cmIsOn(legacy)) {
    this->SetDefinition("WIN32", "1");
    this->SetDefinition("CMAKE_HOST_WIN32", "1");
  }
#endif
#if defined(__APPLE__)
  this->SetDefinition("APPLE", "1");
  this->SetDefinition("CMAKE_HOST_APPLE", "1");
#endif
#if defined(__sun__)
  this->SetDefinition("CMAKE_HOST_SOLARIS", "1");
#endif

  SetDefinition("CMAKE_MAJOR_VERSION",
                std::to_string(cmVersion::GetMajorVersion()));
  SetDefinition("CMAKE_MINOR_VERSION",
                std::to_string(cmVersion::GetMinorVersion()));
  SetDefinition("CMAKE_PATCH_VERSION",
                std::to_string(cmVersion::GetPatchVersion()));
  SetDefinition("CMAKE_TWEAK_VERSION",
                std::to_string(cmVersion::GetTweakVersion()));
  SetDefinition("CMAKE_VERSION", cmVersion::GetCMakeVersion());

  SetDefinition("CMAKE_FILES_DIRECTORY", "/CMakeFiles");

  // Setup the default include file regular expression (match everything).
  Position->BuildSystemDirectory->Properties.SetProperty(
    "INCLUDE_REGULAR_EXPRESSION", "^.*$");
}

void cmStateSnapshot::SetDirectoryDefinitions()
{
  SetDefinition("CMAKE_SOURCE_DIR", State->GetSourceDirectory());
  SetDefinition("CMAKE_CURRENT_SOURCE_DIR", State->GetSourceDirectory());
  SetDefinition("CMAKE_BINARY_DIR", State->GetBinaryDirectory());
  SetDefinition("CMAKE_CURRENT_BINARY_DIR", State->GetBinaryDirectory());
}

void cmStateSnapshot::InitializeFromParent()
{
  cmStateDetail::PositionType parent = Position->DirectoryParent;
  assert(Position->Vars.IsValid());
  assert(parent->Vars.IsValid());

  *Position->Vars = cmDefinitions::MakeClosure(parent->Vars, parent->Root);

  InitializeContentFromParent(
    parent->BuildSystemDirectory->IncludeDirectories,
    Position->BuildSystemDirectory->IncludeDirectories,
    parent->BuildSystemDirectory->IncludeDirectoryBacktraces,
    Position->BuildSystemDirectory->IncludeDirectoryBacktraces,
    Position->IncludeDirectoryPosition);

  InitializeContentFromParent(
    parent->BuildSystemDirectory->CompileDefinitions,
    Position->BuildSystemDirectory->CompileDefinitions,
    parent->BuildSystemDirectory->CompileDefinitionsBacktraces,
    Position->BuildSystemDirectory->CompileDefinitionsBacktraces,
    Position->CompileDefinitionsPosition);

  InitializeContentFromParent(
    parent->BuildSystemDirectory->CompileOptions,
    Position->BuildSystemDirectory->CompileOptions,
    parent->BuildSystemDirectory->CompileOptionsBacktraces,
    Position->BuildSystemDirectory->CompileOptionsBacktraces,
    Position->CompileOptionsPosition);

  InitializeContentFromParent(
    parent->BuildSystemDirectory->LinkOptions,
    Position->BuildSystemDirectory->LinkOptions,
    parent->BuildSystemDirectory->LinkOptionsBacktraces,
    Position->BuildSystemDirectory->LinkOptionsBacktraces,
    Position->LinkOptionsPosition);

  InitializeContentFromParent(
    parent->BuildSystemDirectory->LinkDirectories,
    Position->BuildSystemDirectory->LinkDirectories,
    parent->BuildSystemDirectory->LinkDirectoriesBacktraces,
    Position->BuildSystemDirectory->LinkDirectoriesBacktraces,
    Position->LinkDirectoriesPosition);

  cmProp include_regex =
    parent->BuildSystemDirectory->Properties.GetPropertyValue(
      "INCLUDE_REGULAR_EXPRESSION");
  Position->BuildSystemDirectory->Properties.SetProperty(
    "INCLUDE_REGULAR_EXPRESSION", cmToCStr(include_regex));
}

cmState* cmStateSnapshot::GetState() const
{
  return State;
}

cmStateDirectory cmStateSnapshot::GetDirectory() const
{
  return { Position->BuildSystemDirectory, *this };
}

void cmStateSnapshot::SetProjectName(const std::string& name)
{
  Position->BuildSystemDirectory->ProjectName = name;
}

std::string cmStateSnapshot::GetProjectName() const
{
  return Position->BuildSystemDirectory->ProjectName;
}

void cmStateSnapshot::InitializeFromParent_ForSubdirsCommand()
{
  std::string currentSrcDir = *GetDefinition("CMAKE_CURRENT_SOURCE_DIR");
  std::string currentBinDir = *GetDefinition("CMAKE_CURRENT_BINARY_DIR");
  InitializeFromParent();
  SetDefinition("CMAKE_SOURCE_DIR", State->GetSourceDirectory());
  SetDefinition("CMAKE_BINARY_DIR", State->GetBinaryDirectory());

  SetDefinition("CMAKE_CURRENT_SOURCE_DIR", currentSrcDir);
  SetDefinition("CMAKE_CURRENT_BINARY_DIR", currentBinDir);
}

bool cmStateSnapshot::StrictWeakOrder::operator()(
  const cmStateSnapshot& lhs, const cmStateSnapshot& rhs) const
{
  return lhs.Position.StrictWeakOrdered(rhs.Position);
}

bool operator==(const cmStateSnapshot& lhs, const cmStateSnapshot& rhs)
{
  return lhs.Position == rhs.Position;
}

bool operator!=(const cmStateSnapshot& lhs, const cmStateSnapshot& rhs)
{
  return lhs.Position != rhs.Position;
}
