/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmSourceGroup.h"

#include <utility>

#include <cm/memory>

#include "cmStringAlgorithms.h"

class cmSourceGroupInternals
{
public:
  std::vector<cmSourceGroup> GroupChildren;
};

cmSourceGroup::cmSourceGroup(std::string name, const char* regex,
                             const char* parentName)
  : Name(std::move(name))
{
  Internal = cm::make_unique<cmSourceGroupInternals>();
  SetGroupRegex(regex);
  if (parentName) {
    FullName = cmStrCat(parentName, '\\');
  }
  FullName += Name;
}

cmSourceGroup::~cmSourceGroup() = default;

cmSourceGroup::cmSourceGroup(cmSourceGroup const& r)
{
  Name = r.Name;
  FullName = r.FullName;
  GroupRegex = r.GroupRegex;
  GroupFiles = r.GroupFiles;
  SourceFiles = r.SourceFiles;
  Internal = cm::make_unique<cmSourceGroupInternals>(*r.Internal);
}

cmSourceGroup& cmSourceGroup::operator=(cmSourceGroup const& r)
{
  Name = r.Name;
  GroupRegex = r.GroupRegex;
  GroupFiles = r.GroupFiles;
  SourceFiles = r.SourceFiles;
  *(Internal) = *(r.Internal);
  return *this;
}

void cmSourceGroup::SetGroupRegex(const char* regex)
{
  if (regex) {
    GroupRegex.compile(regex);
  } else {
    GroupRegex.compile("^$");
  }
}

void cmSourceGroup::AddGroupFile(const std::string& name)
{
  GroupFiles.insert(name);
}

std::string const& cmSourceGroup::GetName() const
{
  return Name;
}

std::string const& cmSourceGroup::GetFullName() const
{
  return FullName;
}

bool cmSourceGroup::MatchesRegex(const std::string& name)
{
  return GroupRegex.find(name);
}

bool cmSourceGroup::MatchesFiles(const std::string& name) const
{
  return GroupFiles.find(name) != GroupFiles.cend();
}

void cmSourceGroup::AssignSource(const cmSourceFile* sf)
{
  SourceFiles.push_back(sf);
}

const std::vector<const cmSourceFile*>& cmSourceGroup::GetSourceFiles() const
{
  return SourceFiles;
}

void cmSourceGroup::AddChild(cmSourceGroup const& child)
{
  Internal->GroupChildren.push_back(child);
}

cmSourceGroup* cmSourceGroup::LookupChild(const std::string& name)
{
  for (cmSourceGroup& group : Internal->GroupChildren) {
    // look if descenened is the one were looking for
    if (group.GetName() == name) {
      return (&group); // if it so return it
    }
  }

  // if no child with this name was found return NULL
  return nullptr;
}

cmSourceGroup* cmSourceGroup::MatchChildrenFiles(const std::string& name)
{
  if (MatchesFiles(name)) {
    return this;
  }
  for (cmSourceGroup& group : Internal->GroupChildren) {
    cmSourceGroup* result = group.MatchChildrenFiles(name);
    if (result) {
      return result;
    }
  }
  return nullptr;
}

cmSourceGroup* cmSourceGroup::MatchChildrenRegex(const std::string& name)
{
  for (cmSourceGroup& group : Internal->GroupChildren) {
    cmSourceGroup* result = group.MatchChildrenRegex(name);
    if (result) {
      return result;
    }
  }
  if (MatchesRegex(name)) {
    return this;
  }

  return nullptr;
}

std::vector<cmSourceGroup> const& cmSourceGroup::GetGroupChildren() const
{
  return Internal->GroupChildren;
}
