/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmSourceFile.h"

#include <utility>

#include "cmGlobalGenerator.h"
#include "cmListFileCache.h"
#include "cmMakefile.h"
#include "cmMessageType.h"
#include "cmProperty.h"
#include "cmState.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"
#include "cmake.h"

cmSourceFile::cmSourceFile(cmMakefile* mf, const std::string& name,
                           cmSourceFileLocationKind kind)
  : Location(mf, name, kind)
{
}

std::string const& cmSourceFile::GetExtension() const
{
  return Extension;
}

const std::string cmSourceFile::propLANGUAGE = "LANGUAGE";
const std::string cmSourceFile::propLOCATION = "LOCATION";
const std::string cmSourceFile::propGENERATED = "GENERATED";
const std::string cmSourceFile::propCOMPILE_DEFINITIONS =
  "COMPILE_DEFINITIONS";
const std::string cmSourceFile::propCOMPILE_OPTIONS = "COMPILE_OPTIONS";
const std::string cmSourceFile::propINCLUDE_DIRECTORIES =
  "INCLUDE_DIRECTORIES";

void cmSourceFile::SetObjectLibrary(std::string const& objlib)
{
  ObjectLibrary = objlib;
}

std::string cmSourceFile::GetObjectLibrary() const
{
  return ObjectLibrary;
}

std::string const& cmSourceFile::GetOrDetermineLanguage()
{
  // If the language was set explicitly by the user then use it.
  if (cmProp lang = GetProperty(propLANGUAGE)) {
    // Assign to member in order to return a reference.
    Language = *lang;
    return Language;
  }

  // Perform computation needed to get the language if necessary.
  if (FullPath.empty() && Language.empty()) {
    // If a known extension is given or a known full path is given
    // then trust that the current extension is sufficient to
    // determine the language.  This will fail only if the user
    // specifies a full path to the source but leaves off the
    // extension, which is kind of weird.
    if (Location.ExtensionIsAmbiguous() && Location.DirectoryIsAmbiguous()) {
      // Finalize the file location to get the extension and set the
      // language.
      ResolveFullPath();
    } else {
      // Use the known extension to get the language if possible.
      std::string ext =
        cmSystemTools::GetFilenameLastExtension(Location.GetName());
      CheckLanguage(ext);
    }
  }

  // Use the language determined from the file extension.
  return Language;
}

std::string cmSourceFile::GetLanguage() const
{
  // If the language was set explicitly by the user then use it.
  if (cmProp lang = GetProperty(propLANGUAGE)) {
    return *lang;
  }

  // Use the language determined from the file extension.
  return Language;
}

cmSourceFileLocation const& cmSourceFile::GetLocation() const
{
  return Location;
}

std::string const& cmSourceFile::ResolveFullPath(std::string* error)
{
  if (FullPath.empty()) {
    if (FindFullPath(error)) {
      CheckExtension();
    }
  }
  return FullPath;
}

std::string const& cmSourceFile::GetFullPath() const
{
  return FullPath;
}

bool cmSourceFile::FindFullPath(std::string* error)
{
  // If the file is generated compute the location without checking on disk.
  if (GetIsGenerated()) {
    // The file is either already a full path or is relative to the
    // build directory for the target.
    Location.DirectoryUseBinary();
    FullPath = Location.GetFullPath();
    return true;
  }

  // If this method has already failed once do not try again.
  if (FindFullPathFailed) {
    return false;
  }

  // The file is not generated.  It must exist on disk.
  cmMakefile const* makefile = Location.GetMakefile();
  // Location path
  std::string const& lPath = Location.GetFullPath();
  // List of extension lists
  std::vector<std::string> exts =
    makefile->GetCMakeInstance()->GetAllExtensions();

  // Tries to find the file in a given directory
  auto findInDir = [this, &exts, &lPath](std::string const& dir) -> bool {
    // Compute full path
    std::string const fullPath = cmSystemTools::CollapseFullPath(lPath, dir);
    // Try full path
    if (cmSystemTools::FileExists(fullPath)) {
      FullPath = fullPath;
      return true;
    }
    // Try full path with extension
    for (std::string const& ext : exts) {
      if (!ext.empty()) {
        std::string extPath = cmStrCat(fullPath, '.', ext);
        if (cmSystemTools::FileExists(extPath)) {
          FullPath = extPath;
          return true;
        }
      }
    }
    // File not found
    return false;
  };

  // Try to find the file in various directories
  if (Location.DirectoryIsAmbiguous()) {
    if (findInDir(makefile->GetCurrentSourceDirectory()) ||
        findInDir(makefile->GetCurrentBinaryDirectory())) {
      return true;
    }
  } else {
    if (findInDir({})) {
      return true;
    }
  }

  // Compose error
  std::string err =
    cmStrCat("Cannot find source file:\n  ", lPath, "\nTried extensions");
  for (std::string const& ext : exts) {
    err += " .";
    err += ext;
  }
  if (error != nullptr) {
    *error = std::move(err);
  } else {
    makefile->IssueMessage(MessageType::FATAL_ERROR, err);
  }
  FindFullPathFailed = true;

  // File not found
  return false;
}

void cmSourceFile::CheckExtension()
{
  // Compute the extension.
  std::string realExt = cmSystemTools::GetFilenameLastExtension(FullPath);
  if (!realExt.empty()) {
    // Store the extension without the leading '.'.
    Extension = realExt.substr(1);
  }

  // Look for object files.
  if (Extension == "obj" || Extension == "o" || Extension == "lo") {
    SetProperty("EXTERNAL_OBJECT", "1");
  }

  // Try to identify the source file language from the extension.
  if (Language.empty()) {
    CheckLanguage(Extension);
  }
}

void cmSourceFile::CheckLanguage(std::string const& ext)
{
  // Try to identify the source file language from the extension.
  cmMakefile const* mf = Location.GetMakefile();
  cmGlobalGenerator* gg = mf->GetGlobalGenerator();
  std::string l = gg->GetLanguageFromExtension(ext.c_str());
  if (!l.empty()) {
    Language = l;
  }
}

bool cmSourceFile::Matches(cmSourceFileLocation const& loc)
{
  return Location.Matches(loc);
}

void cmSourceFile::SetProperty(const std::string& prop, const char* value)
{
  if (prop == propINCLUDE_DIRECTORIES) {
    IncludeDirectories.clear();
    if (value) {
      cmListFileBacktrace lfbt = Location.GetMakefile()->GetBacktrace();
      IncludeDirectories.emplace_back(value, lfbt);
    }
  } else if (prop == propCOMPILE_OPTIONS) {
    CompileOptions.clear();
    if (value) {
      cmListFileBacktrace lfbt = Location.GetMakefile()->GetBacktrace();
      CompileOptions.emplace_back(value, lfbt);
    }
  } else if (prop == propCOMPILE_DEFINITIONS) {
    CompileDefinitions.clear();
    if (value) {
      cmListFileBacktrace lfbt = Location.GetMakefile()->GetBacktrace();
      CompileDefinitions.emplace_back(value, lfbt);
    }
  } else {
    Properties.SetProperty(prop, value);
  }

  // Update IsGenerated flag
  if (prop == propGENERATED) {
    IsGenerated = cmIsOn(value);
  }
}

void cmSourceFile::AppendProperty(const std::string& prop,
                                  const std::string& value, bool asString)
{
  if (prop == propINCLUDE_DIRECTORIES) {
    if (!value.empty()) {
      cmListFileBacktrace lfbt = Location.GetMakefile()->GetBacktrace();
      IncludeDirectories.emplace_back(value, lfbt);
    }
  } else if (prop == propCOMPILE_OPTIONS) {
    if (!value.empty()) {
      cmListFileBacktrace lfbt = Location.GetMakefile()->GetBacktrace();
      CompileOptions.emplace_back(value, lfbt);
    }
  } else if (prop == propCOMPILE_DEFINITIONS) {
    if (!value.empty()) {
      cmListFileBacktrace lfbt = Location.GetMakefile()->GetBacktrace();
      CompileDefinitions.emplace_back(value, lfbt);
    }
  } else {
    Properties.AppendProperty(prop, value, asString);
  }

  // Update IsGenerated flag
  if (prop == propGENERATED) {
    IsGenerated = GetPropertyAsBool(propGENERATED);
  }
}

const char* cmSourceFile::GetPropertyForUser(const std::string& prop)
{
  // This method is a consequence of design history and backwards
  // compatibility.  GetProperty is (and should be) a const method.
  // Computed properties should not be stored back in the property map
  // but instead reference information already known.  If they need to
  // cache information in a mutable ivar to provide the return string
  // safely then so be it.
  //
  // The LOCATION property is particularly problematic.  The CMake
  // language has very loose restrictions on the names that will match
  // a given source file (for historical reasons).  Implementing
  // lookups correctly with such loose naming requires the
  // cmSourceFileLocation class to commit to a particular full path to
  // the source file as late as possible.  If the users requests the
  // LOCATION property we must commit now.
  if (prop == propLOCATION) {
    // Commit to a location.
    ResolveFullPath();
  }

  // Similarly, LANGUAGE can be determined by the file extension
  // if it is requested by the user.
  if (prop == propLANGUAGE) {
    // The c_str pointer is valid until `this->Language` is modified.
    return GetOrDetermineLanguage().c_str();
  }

  // Perform the normal property lookup.
  cmProp p = GetProperty(prop);
  return p ? p->c_str() : nullptr;
}

cmProp cmSourceFile::GetProperty(const std::string& prop) const
{
  // Check for computed properties.
  if (prop == propLOCATION) {
    if (FullPath.empty()) {
      return nullptr;
    }
    return &FullPath;
  }

  // Check for the properties with backtraces.
  if (prop == propINCLUDE_DIRECTORIES) {
    if (IncludeDirectories.empty()) {
      return nullptr;
    }

    static std::string output;
    output = cmJoin(IncludeDirectories, ";");
    return &output;
  }

  if (prop == propCOMPILE_OPTIONS) {
    if (CompileOptions.empty()) {
      return nullptr;
    }

    static std::string output;
    output = cmJoin(CompileOptions, ";");
    return &output;
  }

  if (prop == propCOMPILE_DEFINITIONS) {
    if (CompileDefinitions.empty()) {
      return nullptr;
    }

    static std::string output;
    output = cmJoin(CompileDefinitions, ";");
    return &output;
  }

  cmProp retVal = Properties.GetPropertyValue(prop);
  if (!retVal) {
    cmMakefile const* mf = Location.GetMakefile();
    const bool chain =
      mf->GetState()->IsPropertyChained(prop, cmProperty::SOURCE_FILE);
    if (chain) {
      return mf->GetProperty(prop, chain);
    }
    return nullptr;
  }

  return retVal;
}

const char* cmSourceFile::GetSafeProperty(const std::string& prop) const
{
  cmProp ret = GetProperty(prop);
  if (!ret) {
    return "";
  }
  return ret->c_str();
}

bool cmSourceFile::GetPropertyAsBool(const std::string& prop) const
{
  return cmIsOn(GetProperty(prop));
}

void cmSourceFile::SetProperties(cmPropertyMap properties)
{
  Properties = std::move(properties);

  IsGenerated = GetPropertyAsBool(propGENERATED);
}

cmCustomCommand* cmSourceFile::GetCustomCommand() const
{
  return CustomCommand.get();
}

void cmSourceFile::SetCustomCommand(std::unique_ptr<cmCustomCommand> cc)
{
  CustomCommand = std::move(cc);
}
