/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#pragma once

#include "cmConfigure.h" // IWYU pragma: keep

#include <algorithm>
#include <functional>
#include <iterator>

namespace RangeIterators {

template <typename Iter, typename UnaryPredicate>
class FilterIterator
{
public:
  using iterator_category = std::bidirectional_iterator_tag;
  using value_type = typename std::iterator_traits<Iter>::value_type;
  using difference_type = typename std::iterator_traits<Iter>::difference_type;
  using pointer = typename std::iterator_traits<Iter>::pointer;
  using reference = typename std::iterator_traits<Iter>::reference;

  FilterIterator(Iter b, Iter e, UnaryPredicate p)
    : Cur(std::move(b))
    , End(std::move(e))
    , Pred(std::move(p))
  {
    SatisfyPredicate();
  }

  FilterIterator& operator++()
  {
    ++Cur;
    SatisfyPredicate();
    return *this;
  }

  FilterIterator& operator--()
  {
    do {
      --this->Cur;
    } while (!this->Pred(*this->Cur));
    return *this;
  }

  bool operator==(FilterIterator const& other) const
  {
    return Cur == other.Cur;
  }

  bool operator!=(FilterIterator const& other) const
  {
    return !this->operator==(other);
  }

  auto operator*() const -> decltype(*std::declval<Iter>()) { return *Cur; }

private:
  void SatisfyPredicate()
  {
    while (Cur != End && !Pred(*Cur)) {
      ++Cur;
    }
  }

  Iter Cur;
  Iter End;
  UnaryPredicate Pred;
};

template <typename Iter, typename UnaryFunction>
class TransformIterator
{
public:
  using iterator_category = std::bidirectional_iterator_tag;
  using value_type =
    typename std::remove_cv<typename std::remove_reference<decltype(
      std::declval<UnaryFunction>()(*std::declval<Iter>()))>::type>::type;
  using difference_type = typename std::iterator_traits<Iter>::difference_type;
  using pointer = value_type const*;
  using reference = value_type const&;

  TransformIterator(Iter i, UnaryFunction f)
    : Base(std::move(i))
    , Func(std::move(f))
  {
  }

  TransformIterator& operator++()
  {
    ++Base;
    return *this;
  }

  TransformIterator& operator--()
  {
    --Base;
    return *this;
  }

  bool operator==(TransformIterator const& other) const
  {
    return Base == other.Base;
  }

  bool operator!=(TransformIterator const& other) const
  {
    return !this->operator==(other);
  }

  auto operator*() const
    -> decltype(std::declval<UnaryFunction>()(*std::declval<Iter>()))
  {
    return Func(*Base);
  }

private:
  Iter Base;
  UnaryFunction Func;
};

} // namespace RangeIterators

template <typename Iter>
class cmRange
{
public:
  using const_iterator = Iter;
  using value_type = typename std::iterator_traits<Iter>::value_type;
  using difference_type = typename std::iterator_traits<Iter>::difference_type;

  cmRange(Iter b, Iter e)
    : Begin(std::move(b))
    , End(std::move(e))
  {
  }

  Iter begin() const { return Begin; }
  Iter end() const { return End; }
  bool empty() const { return Begin == End; }

  difference_type size() const { return std::distance(Begin, End); }

  cmRange& advance(difference_type amount) &
  {
    std::advance(Begin, amount);
    return *this;
  }

  cmRange advance(difference_type amount) &&
  {
    std::advance(Begin, amount);
    return std::move(*this);
  }

  cmRange& retreat(difference_type amount) &
  {
    std::advance(End, -amount);
    return *this;
  }

  cmRange retreat(difference_type amount) &&
  {
    std::advance(End, -amount);
    return std::move(*this);
  }

  template <typename UnaryPredicate>
  bool all_of(UnaryPredicate p) const
  {
    return std::all_of(Begin, End, std::ref(p));
  }

  template <typename UnaryPredicate>
  bool any_of(UnaryPredicate p) const
  {
    return std::any_of(Begin, End, std::ref(p));
  }

  template <typename UnaryPredicate>
  bool none_of(UnaryPredicate p) const
  {
    return std::none_of(Begin, End, std::ref(p));
  }

  template <typename UnaryPredicate>
  auto filter(UnaryPredicate p) const
    -> cmRange<RangeIterators::FilterIterator<Iter, UnaryPredicate>>
  {
    using It = RangeIterators::FilterIterator<Iter, UnaryPredicate>;
    return { It(Begin, End, p), It(End, End, p) };
  }

  template <typename UnaryFunction>
  auto transform(UnaryFunction f) const
    -> cmRange<RangeIterators::TransformIterator<Iter, UnaryFunction>>
  {
    using It = RangeIterators::TransformIterator<Iter, UnaryFunction>;
    return { It(Begin, f), It(End, f) };
  }

private:
  Iter Begin;
  Iter End;
};

template <typename Iter1, typename Iter2>
bool operator==(cmRange<Iter1> const& left, cmRange<Iter2> const& right)
{
  return left.size() == right.size() &&
    std::equal(left.begin(), left.end(), right.begin());
}

template <typename Iter1, typename Iter2>
auto cmMakeRange(Iter1 begin, Iter2 end) -> cmRange<Iter1>
{
  return { begin, end };
}

template <typename Range>
auto cmMakeRange(Range const& range) -> cmRange<decltype(range.begin())>
{
  return { range.begin(), range.end() };
}

template <typename Range>
auto cmReverseRange(Range const& range) -> cmRange<decltype(range.rbegin())>
{
  return { range.rbegin(), range.rend() };
}
