/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmRST.h"

#include <algorithm>
#include <cctype>
#include <cstddef>
#include <iterator>
#include <utility>

#include "cmsys/FStream.hxx"

#include "cmAlgorithms.h"
#include "cmRange.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"
#include "cmVersion.h"

cmRST::cmRST(std::ostream& os, std::string docroot)
  : OS(os)
  , DocRoot(std::move(docroot))
  , IncludeDepth(0)
  , OutputLinePending(false)
  , LastLineEndedInColonColon(false)
  , Markup(MarkupNone)
  , Directive(DirectiveNone)
  , CMakeDirective("^.. (cmake:)?("
                   "command|variable"
                   ")::[ \t]+([^ \t\n]+)$")
  , CMakeModuleDirective("^.. cmake-module::[ \t]+([^ \t\n]+)$")
  , ParsedLiteralDirective("^.. parsed-literal::[ \t]*(.*)$")
  , CodeBlockDirective("^.. code-block::[ \t]*(.*)$")
  , ReplaceDirective("^.. (\\|[^|]+\\|) replace::[ \t]*(.*)$")
  , IncludeDirective("^.. include::[ \t]+([^ \t\n]+)$")
  , TocTreeDirective("^.. toctree::[ \t]*(.*)$")
  , ProductionListDirective("^.. productionlist::[ \t]*(.*)$")
  , NoteDirective("^.. note::[ \t]*(.*)$")
  , ModuleRST(R"(^#\[(=*)\[\.rst:$)")
  , CMakeRole("(:cmake)?:("
              "command|cpack_gen|generator|variable|envvar|module|policy|"
              "prop_cache|prop_dir|prop_gbl|prop_inst|prop_sf|"
              "prop_test|prop_tgt|"
              "manual"
              "):`(<*([^`<]|[^` \t]<)*)([ \t]+<[^`]*>)?`")
  , InlineLink("`(<*([^`<]|[^` \t]<)*)([ \t]+<[^`]*>)?`_")
  , InlineLiteral("``([^`]*)``")
  , Substitution("(^|[^A-Za-z0-9_])"
                 "((\\|[^| \t\r\n]([^|\r\n]*[^| \t\r\n])?\\|)(__|_|))"
                 "([^A-Za-z0-9_]|$)")
  , TocTreeLink("^.*[ \t]+<([^>]+)>$")
{
  Replace["|release|"] = cmVersion::GetCMakeVersion();
}

bool cmRST::ProcessFile(std::string const& fname, bool isModule)
{
  cmsys::ifstream fin(fname.c_str());
  if (fin) {
    DocDir = cmSystemTools::GetFilenamePath(fname);
    if (isModule) {
      ProcessModule(fin);
    } else {
      ProcessRST(fin);
    }
    OutputLinePending = true;
    return true;
  }
  return false;
}

void cmRST::ProcessRST(std::istream& is)
{
  std::string line;
  while (cmSystemTools::GetLineFromStream(is, line)) {
    ProcessLine(line);
  }
  Reset();
}

void cmRST::ProcessModule(std::istream& is)
{
  std::string line;
  std::string rst;
  while (cmSystemTools::GetLineFromStream(is, line)) {
    if (!rst.empty() && rst != "#") {
      // Bracket mode: check for end bracket
      std::string::size_type pos = line.find(rst);
      if (pos == std::string::npos) {
        ProcessLine(line);
      } else {
        if (line[0] != '#') {
          line.resize(pos);
          ProcessLine(line);
        }
        rst.clear();
        Reset();
        OutputLinePending = true;
      }
    } else {
      // Line mode: check for .rst start (bracket or line)
      if (rst == "#") {
        if (line == "#") {
          ProcessLine("");
          continue;
        }
        if (cmHasLiteralPrefix(line, "# ")) {
          line.erase(0, 2);
          ProcessLine(line);
          continue;
        }
        rst.clear();
        Reset();
        OutputLinePending = true;
      }
      if (line == "#.rst:") {
        rst = "#";
      } else if (ModuleRST.find(line)) {
        rst = "]" + ModuleRST.match(1) + "]";
      }
    }
  }
  if (rst == "#") {
    Reset();
  }
}

void cmRST::Reset()
{
  if (!MarkupLines.empty()) {
    cmRST::UnindentLines(MarkupLines);
  }
  switch (Directive) {
    case DirectiveNone:
      break;
    case DirectiveParsedLiteral:
      ProcessDirectiveParsedLiteral();
      break;
    case DirectiveLiteralBlock:
      ProcessDirectiveLiteralBlock();
      break;
    case DirectiveCodeBlock:
      ProcessDirectiveCodeBlock();
      break;
    case DirectiveReplace:
      ProcessDirectiveReplace();
      break;
    case DirectiveTocTree:
      ProcessDirectiveTocTree();
      break;
  }
  Markup = MarkupNone;
  Directive = DirectiveNone;
  MarkupLines.clear();
}

void cmRST::ProcessLine(std::string const& line)
{
  bool lastLineEndedInColonColon = LastLineEndedInColonColon;
  LastLineEndedInColonColon = false;

  // A line starting in .. is an explicit markup start.
  if (line == ".." ||
      (line.size() >= 3 && line[0] == '.' && line[1] == '.' &&
       isspace(line[2]))) {
    Reset();
    Markup =
      (line.find_first_not_of(" \t", 2) == std::string::npos ? MarkupEmpty
                                                             : MarkupNormal);
    // XXX(clang-tidy): https://bugs.llvm.org/show_bug.cgi?id=44165
    // NOLINTNEXTLINE(bugprone-branch-clone)
    if (CMakeDirective.find(line)) {
      // Output cmake domain directives and their content normally.
      NormalLine(line);
    } else if (CMakeModuleDirective.find(line)) {
      // Process cmake-module directive: scan .cmake file comments.
      std::string file = CMakeModuleDirective.match(1);
      if (file.empty() || !ProcessInclude(file, IncludeModule)) {
        NormalLine(line);
      }
    } else if (ParsedLiteralDirective.find(line)) {
      // Record the literal lines to output after whole block.
      Directive = DirectiveParsedLiteral;
      MarkupLines.push_back(ParsedLiteralDirective.match(1));
    } else if (CodeBlockDirective.find(line)) {
      // Record the literal lines to output after whole block.
      // Ignore the language spec and record the opening line as blank.
      Directive = DirectiveCodeBlock;
      MarkupLines.emplace_back();
    } else if (ReplaceDirective.find(line)) {
      // Record the replace directive content.
      Directive = DirectiveReplace;
      ReplaceName = ReplaceDirective.match(1);
      MarkupLines.push_back(ReplaceDirective.match(2));
    } else if (IncludeDirective.find(line)) {
      // Process the include directive or output the directive and its
      // content normally if it fails.
      std::string file = IncludeDirective.match(1);
      if (file.empty() || !ProcessInclude(file, IncludeNormal)) {
        NormalLine(line);
      }
    } else if (TocTreeDirective.find(line)) {
      // Record the toctree entries to process after whole block.
      Directive = DirectiveTocTree;
      MarkupLines.push_back(TocTreeDirective.match(1));
    } else if (ProductionListDirective.find(line)) {
      // Output productionlist directives and their content normally.
      NormalLine(line);
    } else if (NoteDirective.find(line)) {
      // Output note directives and their content normally.
      NormalLine(line);
    }
  }
  // An explicit markup start followed nothing but whitespace and a
  // blank line does not consume any indented text following.
  else if (Markup == MarkupEmpty && line.empty()) {
    NormalLine(line);
  }
  // Indented lines following an explicit markup start are explicit markup.
  else if (Markup && (line.empty() || isspace(line[0]))) {
    Markup = MarkupNormal;
    // Record markup lines if the start line was recorded.
    if (!MarkupLines.empty()) {
      MarkupLines.push_back(line);
    }
  }
  // A blank line following a paragraph ending in "::" starts a literal block.
  else if (lastLineEndedInColonColon && line.empty()) {
    // Record the literal lines to output after whole block.
    Markup = MarkupNormal;
    Directive = DirectiveLiteralBlock;
    MarkupLines.emplace_back();
    OutputLine("", false);
  }
  // Print non-markup lines.
  else {
    NormalLine(line);
    LastLineEndedInColonColon =
      (line.size() >= 2 && line[line.size() - 2] == ':' && line.back() == ':');
  }
}

void cmRST::NormalLine(std::string const& line)
{
  Reset();
  OutputLine(line, true);
}

void cmRST::OutputLine(std::string const& line_in, bool inlineMarkup)
{
  if (OutputLinePending) {
    OS << "\n";
    OutputLinePending = false;
  }
  if (inlineMarkup) {
    std::string line = ReplaceSubstitutions(line_in);
    std::string::size_type pos = 0;
    for (;;) {
      std::string::size_type* first = nullptr;
      std::string::size_type role_start = std::string::npos;
      std::string::size_type link_start = std::string::npos;
      std::string::size_type lit_start = std::string::npos;
      if (CMakeRole.find(line.c_str() + pos)) {
        role_start = CMakeRole.start();
        first = &role_start;
      }
      if (InlineLiteral.find(line.c_str() + pos)) {
        lit_start = InlineLiteral.start();
        if (!first || lit_start < *first) {
          first = &lit_start;
        }
      }
      if (InlineLink.find(line.c_str() + pos)) {
        link_start = InlineLink.start();
        if (!first || link_start < *first) {
          first = &link_start;
        }
      }
      if (first == &role_start) {
        OS << line.substr(pos, role_start);
        std::string text = CMakeRole.match(3);
        // If a command reference has no explicit target and
        // no explicit "(...)" then add "()" to the text.
        if (CMakeRole.match(2) == "command" && CMakeRole.match(5).empty() &&
            text.find_first_of("()") == std::string::npos) {
          text += "()";
        }
        OS << "``" << text << "``";
        pos += CMakeRole.end();
      } else if (first == &lit_start) {
        OS << line.substr(pos, lit_start);
        std::string text = InlineLiteral.match(1);
        pos += InlineLiteral.end();
        OS << "``" << text << "``";
      } else if (first == &link_start) {
        OS << line.substr(pos, link_start);
        std::string text = InlineLink.match(1);
        bool escaped = false;
        for (char c : text) {
          if (escaped) {
            escaped = false;
            OS << c;
          } else if (c == '\\') {
            escaped = true;
          } else {
            OS << c;
          }
        }
        pos += InlineLink.end();
      } else {
        break;
      }
    }
    OS << line.substr(pos) << "\n";
  } else {
    OS << line_in << "\n";
  }
}

std::string cmRST::ReplaceSubstitutions(std::string const& line)
{
  std::string out;
  std::string::size_type pos = 0;
  while (Substitution.find(line.c_str() + pos)) {
    std::string::size_type start = Substitution.start(2);
    std::string::size_type end = Substitution.end(2);
    std::string substitute = Substitution.match(3);
    auto replace = Replace.find(substitute);
    if (replace != Replace.end()) {
      std::pair<std::set<std::string>::iterator, bool> replaced =
        Replaced.insert(substitute);
      if (replaced.second) {
        substitute = ReplaceSubstitutions(replace->second);
        Replaced.erase(replaced.first);
      }
    }
    out += line.substr(pos, start);
    out += substitute;
    pos += end;
  }
  out += line.substr(pos);
  return out;
}

void cmRST::OutputMarkupLines(bool inlineMarkup)
{
  for (auto line : MarkupLines) {
    if (!line.empty()) {
      line = cmStrCat(" ", line);
    }
    OutputLine(line, inlineMarkup);
  }
  OutputLinePending = true;
}

bool cmRST::ProcessInclude(std::string file, IncludeType type)
{
  bool found = false;
  if (IncludeDepth < 10) {
    cmRST r(OS, DocRoot);
    r.IncludeDepth = IncludeDepth + 1;
    r.OutputLinePending = OutputLinePending;
    if (type != IncludeTocTree) {
      r.Replace = Replace;
    }
    if (file[0] == '/') {
      file = DocRoot + file;
    } else {
      file = DocDir + "/" + file;
    }
    found = r.ProcessFile(file, type == IncludeModule);
    if (type != IncludeTocTree) {
      Replace = r.Replace;
    }
    OutputLinePending = r.OutputLinePending;
  }
  return found;
}

void cmRST::ProcessDirectiveParsedLiteral()
{
  OutputMarkupLines(true);
}

void cmRST::ProcessDirectiveLiteralBlock()
{
  OutputMarkupLines(false);
}

void cmRST::ProcessDirectiveCodeBlock()
{
  OutputMarkupLines(false);
}

void cmRST::ProcessDirectiveReplace()
{
  // Record markup lines as replacement text.
  std::string& replacement = Replace[ReplaceName];
  replacement += cmJoin(MarkupLines, " ");
  ReplaceName.clear();
}

void cmRST::ProcessDirectiveTocTree()
{
  // Process documents referenced by toctree directive.
  for (std::string const& line : MarkupLines) {
    if (!line.empty() && line[0] != ':') {
      if (TocTreeLink.find(line)) {
        std::string const& link = TocTreeLink.match(1);
        ProcessInclude(link + ".rst", IncludeTocTree);
      } else {
        ProcessInclude(line + ".rst", IncludeTocTree);
      }
    }
  }
}

void cmRST::UnindentLines(std::vector<std::string>& lines)
{
  // Remove the common indentation from the second and later lines.
  std::string indentText;
  std::string::size_type indentEnd = 0;
  bool first = true;
  for (size_t i = 1; i < lines.size(); ++i) {
    std::string const& line = lines[i];

    // Do not consider empty lines.
    if (line.empty()) {
      continue;
    }

    // Record indentation on first non-empty line.
    if (first) {
      first = false;
      indentEnd = line.find_first_not_of(" \t");
      indentText = line.substr(0, indentEnd);
      continue;
    }

    // Truncate indentation to match that on this line.
    indentEnd = std::min(indentEnd, line.size());
    for (std::string::size_type j = 0; j != indentEnd; ++j) {
      if (line[j] != indentText[j]) {
        indentEnd = j;
        break;
      }
    }
  }

  // Update second and later lines.
  for (size_t i = 1; i < lines.size(); ++i) {
    std::string& line = lines[i];
    if (!line.empty()) {
      line = line.substr(indentEnd);
    }
  }

  auto it = lines.cbegin();
  size_t leadingEmpty = std::distance(it, cmFindNot(lines, std::string()));

  auto rit = lines.crbegin();
  size_t trailingEmpty =
    std::distance(rit, cmFindNot(cmReverseRange(lines), std::string()));

  if ((leadingEmpty + trailingEmpty) >= lines.size()) {
    // All lines are empty.  The markup block is empty.  Leave only one.
    lines.resize(1);
    return;
  }

  auto contentEnd = cmRotate(lines.begin(), lines.begin() + leadingEmpty,
                             lines.end() - trailingEmpty);
  lines.erase(contentEnd, lines.end());
}
